// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// UpdaterList: manages a list of updaters.
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_UPDATER_UPDATERLIST_H
#define POOMA_NEWFIELD_UPDATER_UPDATERLIST_H

#include "NewField/Updater/UpdaterListItem.h"
#include "NewField/Updater/UpdaterBase.h"
#include "Utilities/RefCountedPtr.h"
#include "Utilities/RefCounted.h"

#include <vector>

#if defined(POOMA_DEBUG_UPDATERS)
#include <iostream>
#endif


//-----------------------------------------------------------------------------
// UpdaterListData is a ref-counted object that holds the data for an
// updater list.
//-----------------------------------------------------------------------------

class UpdaterListData : public RefCounted
{
public:

  //---------------------------------------------------------------------------
  // Constructors.

  // The default constructor simply calls the default constructor for the
  // contained vector.
  
  UpdaterListData() { }
    
  // Copy constructor.

  UpdaterListData(const UpdaterListData &model)
  : data_m(model.data_m) { }
  
  
  //---------------------------------------------------------------------------
  // Copy assignment operator.
  
  UpdaterListData &operator=(const UpdaterListData &rhs)
  {
    data_m = rhs.data_m;
    
    return *this;
  }
  

  //---------------------------------------------------------------------------
  // We need to delete the actual UpdaterListItems.

  ~UpdaterListData() 
    {
      for (int i = 0; i < data_m.size(); i++)
        delete data_m[i]; 
    }


  //---------------------------------------------------------------------------
  // Return the number of UpdaterListItems.
  
  inline int size() const
    {
      return data_m.size();
    }

  
  //---------------------------------------------------------------------------
  // Returns the ith element.
  
  inline UpdaterListItem *elem(int i) const
    {
      return data_m[i];
    }
  
  inline UpdaterListItem* &elem(int i)
    {
      return data_m[i];
    }


  //---------------------------------------------------------------------------
  // Pushes an item on the list.

  inline void push_back(UpdaterListItem *item)
  {
    data_m.push_back(item);
  }
   
private:

  std::vector<UpdaterListItem *> data_m;
};


//-----------------------------------------------------------------------------
// UpdaterList is a container that dispatches events to the list of boundary 
// conditions it contains. 
//-----------------------------------------------------------------------------

class UpdaterList {
public:

  //---------------------------------------------------------------------------
  // The default constructor simply makes an empty list. 
  
  UpdaterList() : list_m(new UpdaterListData) { }


  //---------------------------------------------------------------------------
  // The copy constructor makes a shallow copy of the data.
  
  UpdaterList(const UpdaterList &model)
  : list_m(model.list_m)
    { }


  //---------------------------------------------------------------------------
  // Destructor: trivial.
  
  ~UpdaterList() { }


  //---------------------------------------------------------------------------
  // Replaces the current list with a private copy of itself.
  
  template<class Subject>
  void makeOwnCopy(const Subject &s)
    {
      // Make a copy of the list. 

      list_m = new UpdaterListData(*list_m);
       
      // Now, we need to replace the individual UpdaterListItems
      // with cloned versions with this Subject as the subject. 
      
      for (int i = 0; i < list_m->size(); i++)
        {
          UpdaterRetargetBase<Subject> *u = 
            dynamic_cast<UpdaterRetargetBase<Subject> *>(list_m->elem(i));
          PAssert(u != NULL);
          list_m->elem(i) = u->retarget(s);
        }
    }


  //---------------------------------------------------------------------------
  // Replaces the current list with an empty list.
  
  inline void erase()
    {
     list_m = new UpdaterListData;
    }


  //---------------------------------------------------------------------------
  // Adds an updater to the end of our list.

  template<class Subject, class Category>
  void addUpdater(const Subject &s, const Category &c)
    {
      UpdaterListItem *bc = c.create(s);
      list_m->push_back(bc);

#if defined(POOMA_DEBUG_UPDATERS)
      std::cout << "Adding Updater " << (void *) bc << std::endl;
#endif
    }

  
  //---------------------------------------------------------------------------
  // Notify the updaters about pre-read/post-write events.

  void notifyPreRead() const
    {
      for (int i = 0; i < list_m->size(); ++i)
        list_m->elem(i)->notifyPreRead();
    }

  void notifyPostWrite() const
    {
      for (int i = 0; i < list_m->size(); ++i)
        list_m->elem(i)->notifyPostWrite();
    }

  
  //---------------------------------------------------------------------------
  // Set/clear the dirty flags for all of the updaters.

  void setDirty() const
    {
      for (int i = 0; i < list_m->size(); ++i)
        list_m->elem(i)->setDirty();
    }

  void clearDirty() const
    {
      for (int i = 0; i < list_m->size(); ++i)
        list_m->elem(i)->clearDirty();
    }
  
  
  //---------------------------------------------------------------------------
  // Give access to a specific boundary updater.

  inline UpdaterListItem *operator()(int i) const
    {
#if POOMA_BOUNDS_CHECK
      PInsist2(i >= 0 && i < list_m->size(),
        "UpdaterList bounds error: index = %d, size = %d.",
        i, list_m->size());
#endif

      return list_m->elem(i);
    }

  inline UpdaterListItem * &operator()(int i)
  {
#if POOMA_BOUNDS_CHECK
      PInsist2(i >= 0 && i < list_m->size(),
        "UpdaterList bounds error: index = %d, size = %d.",
        i, list_m->size());
#endif

      return list_m->elem(i);
    }


  //---------------------------------------------------------------------------
  // Return the number of boundary conditions.
  
  inline int size() const { return list_m->size(); }
  
  
private:

  RefCountedPtr<UpdaterListData> list_m;
};


#endif // POOMA_NEWFIELD_UPDATER_UPDATERLIST_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UpdaterList.h,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/07/24 23:36:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
