// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Basic Test2: Simple data parallel expressions.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/NewFields.h"
#include "Utilities/Tester.h"

#include <iostream>
#include <math.h>

//-----------------------------------------------------------------------------
// Main program:
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  // Create the physical domain.
  
  const int NX = 5, NY = 5;
  Interval<1> I(NX), J(NY);
  
  Vector<2,double> origin;
  Vector<2,double> spacings;
  for (int d = 0; d < 2; d++) 
    {
      origin(d) = d;
      spacings(d) = d + 1;
    }

  DomainLayout<2> layout1(Interval<2>(I, J), GuardLayers<2>(1));
  Vert vert;
  
  Field<UniformRectilinear<2>, double, Brick> 
    f(vert, layout1, origin, spacings),
    g(vert, layout1, origin, spacings),
    h(vert, layout1, origin, spacings);
    
  f.addUpdaters(AllConstantFaceBC<double>(0.0));
  g.addUpdaters(AllConstantFaceBC<double>(0.0));
  h.addUpdaters(AllConstantFaceBC<double>(0.0));
    
  for (int i = 0; i <= f.physicalDomain()[0].last(); i++)
    for (int j = 0; j <= f.physicalDomain()[1].last(); j++)
      {
        g(i, j) = 2.0 * i + j;
        h(i, j) = 4.0 - i - 3.0 * j;
      }

  f.update();
  
  tester.out() << "f = 1.0..." << std::endl;  
  f = 1.0;
  tester.out() << f << std::endl;
  tester.check("f = 1.0", sum(f), 25 * 1.0);

  tester.out() << "f -= g..." << std::endl;  
  f -= g;
  tester.out() << f << std::endl;
  tester.check("f -= g", sum(f), -125.0);

  tester.out() << sum(f) << std::endl;

  tester.out() << "f += sin(g) + 2.0 * h..." << std::endl;  
  f += sin(g) + 2.0 * h;
  tester.out() << f << std::endl;
  tester.check("f += sin(g) + 2.0 * h", sum(f), -324.60252, 1.0e-4);

  tester.out() << "TD f += sin(g) + 2.0 * h..." << std::endl;  
  f(f.totalDomain()) += sin(g.all()) + 2.0 * h(h.totalDomain());
  tester.out() << f << std::endl;
  tester.check("TD f += sin(g) + 2.0 * h", sum(f), -524.20503, 1.0e-4);

  tester.out() << "TD f += sin(g) + 2.0 * h..." << std::endl;  
  f.all() += sin(g(g.totalDomain())) + 2.0 * h(h.totalDomain());
  tester.out() << f << std::endl;
  tester.check("TD f += sin(g) + 2.0 * h", sum(f), -723.80755, 1.0e-4);

  int ret = tester.results("BasicTest2");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BasicTest2.cpp,v $   $Author: swhaney $
// $Revision: 1.1 $   $Date: 2000/07/25 01:11:03 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
