// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// A 2nd tour of the new Field class featuring multi-material fields.
//-----------------------------------------------------------------------------

#include "Pooma/NewFields.h"

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  // To declare a field, you first need to set up a layout. This requires
  // knowing the physical vertex-domain and the number of external guard
  // cell layers. Vertex domains contain enough points to hold all of the
  // rectilinear centerings that POOMA is likely to support for quite
  // awhile. Also, it means that the same layout can be used for all
  // fields, regardless of centering.
  
  Interval<2> physicalVertexDomain(4, 4);
  DomainLayout<2> layout(physicalVertexDomain, GuardLayers<2>(1));
  
  // Now, we can declare a field. Let's make this a multi-material
  // field with cell centering.

  typedef Field<UniformRectilinear<2>, double, Brick> Field_t;
  Field_t f(ReplicateSubFields<Cell>(3), layout, 
    Vector<2>(0.0), Vector<2>(1.0, 2.0));

  // Set some data in the field.
  
  f[0].all() = 2.0; f[0] = -1.0; 
  f[1].all() = 3.0; f[1] = -2.0; 
  f[2].all() = 4.0; f[2] = -3.0; 
  
  std::cout << f.all() << std::endl;

  // Try some reductions.

  std::cout << sum(f[0]) << std::endl;
  std::cout << sum(f[1] + f[2]) << std::endl;
  
  // Take a range-based view. Note: the only views allowed for
  // fields with sub-fields are those constructed using Intervals
  // and INodes. The reason is that a Range of cells can lead
  // to non-constant strides through the sub-field elements. One
  // can construct Range-based views of Fields with no subfields.
  // The result is a field with a NoGeometry, GeometryTag.
  
  Range<1> r(-1, 3, 2);
  Range<2> R(r, r);
  std::cout << f[2](R) << std::endl;

  Pooma::finalize();
  return 0; 
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldTour2.cpp,v $   $Author: swhaney $
// $Revision: 1.1 $   $Date: 2000/07/24 23:36:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
