// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_PARTICLES_SPATIAL_LAYOUT_H
#define POOMA_PARTICLES_SPATIAL_LAYOUT_H

//-----------------------------------------------------------------------------
// Classes:
//   SpatialLayout<Geometry, Layout>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
// SpatialLayout<Geometry, Layout> is a particle layout class that is used to
// determine where particles will be located in a parallel environment.
// SpatialLayout uses a domain decomposition algorithm to keep particles
// distributed among patches relative to some Field's layout and geometry.
// It is templated on the geometry and layout of the system,
// and must be used in conjunction with Fields of the same geometry.
// SpatialLayout is a PatchSwapLayout, and inherits the main "swap" routine
// from that class.  It provides a "findPatchNumber" routine that calculates
// patch numbers based on the spatial position of the particles.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Particles/PatchSwapLayout.h"
#include "Partition/SpatialPartition.h"
#include "Utilities/PAssert.h"
#include "Domain/Region.h"

#include <iosfwd>

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
// Forward References
//-----------------------------------------------------------------------------

template<class G, class FL> class SpatialLayout;


//-----------------------------------------------------------------------------
// ParticleInBox summary:
//
// ParticleInBox is a simple partially-specialized class that calculates
// whether a particle with a given position is located within a specified
// Region.  It has one static method "check" that does this comparison.
// ParticleInBox is specialized for dimensions 1 ... 3 to do the check
// very efficiently.
//-----------------------------------------------------------------------------

template<int Dim, class T>
struct ParticleInBox
{
  template<class Pos>
  inline static bool check(const Pos &pos, const Region<Dim,T> &box)
    {
      CTAssert(Dim == Pos::dimensions);
      for (int d=0; d < Dim; ++d)
	if (pos(d) < box[d].min() || pos(d) > box[d].max())
	  return false;
      return true;
    }
};

template<class T>
struct ParticleInBox<1, T>
{
  template<class Pos>
  inline static bool check(const Pos &pos, const Region<1,T> &box)
    {
      CTAssert(Pos::dimensions == 1);
      return (!(pos(0) < box[0].min() || pos(0) > box[0].max()));
    }
};

template<class T>
struct ParticleInBox<2, T>
{
  template<class Pos>
  inline static bool check(const Pos &pos, const Region<2,T> &box)
    {
      CTAssert(Pos::dimensions == 1);
      return (!(pos(0) < box[0].min() || pos(0) > box[0].max() ||
		pos(1) < box[1].min() || pos(1) > box[1].max()));
    }
};

template<class T>
struct ParticleInBox<3, T>
{
  template<class Pos>
  inline static bool check(const Pos &pos, const Region<3,T> &box)
    {
      CTAssert(Pos::dimensions == 1);
      return (!(pos(0) < box[0].min() || pos(0) > box[0].max() ||
		pos(1) < box[1].min() || pos(1) > box[1].max() ||
		pos(2) < box[2].min() || pos(2) > box[2].max()));
    }
};


//-----------------------------------------------------------------------------
// SpatialLayout summary:
//
// SpatialLayout is one of the particle layout classes that determine
// where particles should be located.  It derives from PatchSwapLayout,
// since it is a layout that will use the infrastructure of PatchSwapLayout
// to swap particles from one patch to another.  It does this with a
// spatial decomposition algorithm: given a Geometry, and a Layout from
// a Field (which describes how the Geometry is decomposed into patches
// assigned to memory contexts), particles are moved within patches
// based on their spatial position.
//-----------------------------------------------------------------------------

template<class G, class FL>
class SpatialLayout: public PatchSwapLayout< SpatialLayout<G,FL> >
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  // Utility typedef to refer to ourselves and our base class

  typedef SpatialLayout<G,FL>                This_t;
  typedef PatchSwapLayout<This_t>            Base_t;

  // The geometry type

  typedef G                                  Geometry_t;

  // The field layout type

  typedef FL                                 FieldLayout_t;

  // The mesh and centering types from the geometry

  typedef typename Geometry_t::Mesh_t        Mesh_t;
  typedef typename Geometry_t::Centering_t   Centering_t;

  // The type of data used to store a single coordinate value.
  // This is the type for a single component (e.g., double instead of
  // Vector<3,double>).  PointType_t is the type for a Vector to store
  // positions.

  typedef typename Mesh_t::AxisType_t        AxisType_t;
  typedef typename Mesh_t::PointType_t       PointType_t;

  // Storage for hold particle amounts ... this should be the same
  // as the typedef in Particles.h

  typedef typename Base_t::Size_t            Size_t;

  // The type of array used to store amounts to move to other patches

  typedef typename Base_t::AmountArray_t     AmountArray_t;

  // The type of array used to store patch ID's and indices

  typedef typename Base_t::MoveArray_t       MoveArray_t;

  // The number of dimensions in the mesh, which is the number of
  // indices needed to look up a mesh vertex or an Array value.

  enum { dimensions = Geometry_t::dimensions };

  // The number of dimensions in the coordinate system; this it the
  // number of elements a position vector must have.

  enum { coordinateDimensions = Geometry_t::coordinateDimensions };


  //============================================================
  // Constructors
  //============================================================

  // Default constructor.  The user must call initialize after this is used.

  SpatialLayout()
    : Base_t(*this)
    {
      // The Geometry and Layout must be consistent

      CTAssert(int(Geometry_t::dimensions) == int(FieldLayout_t::dimensions));
    }

  // The main constructor, that takes a Geometry object and an associated
  // field layout object.  This class will make copies of these objects.
  // items.

  SpatialLayout(const Geometry_t &geometry, const FieldLayout_t &layout)
    : Base_t(*this), geometry_m(geometry), fieldLayout_m(layout)
    {
      // The Geometry and Layout must be consistent
      
      CTAssert(int(Geometry_t::dimensions) == int(FieldLayout_t::dimensions));
    }

  // Copy constructor.

  SpatialLayout(const This_t &s)
    : Base_t(*this), geometry_m(s.geometry_m), fieldLayout_m(s.fieldLayout_m)
    {
      // The Geometry and Layout must be consistent
      
      CTAssert(int(Geometry_t::dimensions) == int(FieldLayout_t::dimensions));
    }

  // Initialize with a geometry and associated field layout object.

  void initialize(const Geometry_t &geometry, const FieldLayout_t &layout)
    {
      // Save the provided geometry and layout

      geometry_m.initialize(geometry);
      fieldLayout_m = layout;
    }

  // Initialize with another SpatialLayout object

  inline void initialize(const This_t &s)
    {
      initialize(s.geometry(), s.layout());
    }

  // Assignment operator

  inline This_t &operator=(const This_t &s)
    {
      initialize(s.geometry(), s.layout());
      return *this;
    }


  //============================================================
  // Destructor
  //============================================================

  ~SpatialLayout()
    {
    }


  //============================================================
  // Accessors
  //============================================================

  // Return whether we have been initialized with a field layout and geometry.

  inline bool initialized() const
    {
      return (fieldLayout_m.initialized() && geometry_m.initialized());
    }

  // Return the field layout we're using

  inline const FieldLayout_t &layout() const
    {
      return fieldLayout_m;
    }

  // Return the geometry we're using

  inline const Geometry_t &geometry() const
    {
      return geometry_m;
    }

  // Return the number of patches used by the field layout.  All
  // particle layout objects must provide these methods.

  inline int patchesGlobal() const
    {
      return layout().sizeGlobal();
    }

  inline int patchesLocal() const
    {
      return layout().sizeLocal();
    }

  inline int patchesRemote() const
    {
      return layout().sizeRemote();
    }


  //============================================================
  // Attribute layout initialization
  //============================================================

  // The following method is called by the user of this class to
  // initialize a layout object for attributes that will need to
  // be kept organized by this layout.  For spatial layout, this
  // makes sure the attribute layout has the same number of patches
  // as the array layout, located on the same contexts and with the
  // same processor affinity.  The attribute layout is initialized
  // with zero elements in each patch.

  template<class AL>
  void initializeAttributeLayout(AL &attriblayout)
    {
      // SpatialLayout is given an array layout for a multi-dimensional
      // array of data, that can have several patches.  We use that
      // with a SpatialPartition object to initialize the provided
      // attribute layout object properly.  SpatialPartition will add in
      // domains to attributelayout that are initially empty, have the
      // same total number as the array layout, and the same memory
      // affinity.  Later, the user will add in elements to that layout
      // via create() operations.

      typedef typename AL::Domain_t ALDomain_t;
      attriblayout.initialize(ALDomain_t(), SpatialPartition<FL>(layout()),
                              DefaultSPmapper(layout()));
    }


  //============================================================
  // Particle patch location calculation
  //============================================================

  // Calculate the patch ID that each particle should have in the
  // provided arrays.  The first argument is the global patch ID number
  // for the particles that are provided in this call, and the second
  // argument is an Array storing particle positions for that particular
  // patch.  This routine calculates the patch ID for each of the particles,
  // and stores that patch ID in the third argument (an array of the same
  // length that stores patch ID's).  The final argument is an Array with
  // length equal to the number of patches; this routine should increment
  // the value for the Nth patch in that array by the number of paricles that
  // this routine determines goes in that patch.  Finally, this returns
  // the total number of particles that are destined for patches OTHER than
  // the patch mentioned as the first argument.

  template<class Attr>
  Size_t findPatchNumber(int lid, int gid, const Attr &pos,
			 MoveArray_t &movepid, AmountArray_t &moveamount)
  {
    // Find the current size of this patch

    Size_t size = pos.domain().size();

    // Find the bounding box for this patch in the mesh

    Region<dimensions, AxisType_t> box =
      geometry_m.boundingBox(layout().patchDomain(lid));

    // Create a debug output stream, used if POOMA_PATCHSWAPLAYOUT_DBG
    // is set properly.
    // NOTE: the POOMA_PATCHSWAPLAYOUT_DBG macro is defined in
    // PatchSwapLayout.h

    POOMA_PATCHSWAPLAYOUT_DBG(Inform dbgmsg("SpatialLayout::findPatchNumber");)
    POOMA_PATCHSWAPLAYOUT_DBG(dbgmsg.setOutputContext(-1);)
    POOMA_PATCHSWAPLAYOUT_DBG(dbgmsg << "Finding patch numbers for "<< size)
    POOMA_PATCHSWAPLAYOUT_DBG(       << " particles in local patch " << lid)
    POOMA_PATCHSWAPLAYOUT_DBG(       << ", global patch " << gid)
    POOMA_PATCHSWAPLAYOUT_DBG(       << std::endl;)

    Size_t totmove = 0;

    for (int i = 0; i < size; ++i)
    {
      // Convert the particle position to an index into the Field's
      // domain, using the Geometry, and from this, get the global
      // patch ID for where it goes.

      int npid = gid;
      if (!ParticleInBox<dimensions,AxisType_t>::check(pos(i), box))
      {
	// Calculate the patch ID using the mesh and field layout

	npid = fieldLayout_m.globalID(geometry_m.pointIndex(pos(i)));
	PAssert(npid != gid);

	// Save the patch ID for this particle ... we know that npid
	// is not equal to pid since the particle is outside our box

	moveamount(npid) += 1;
	++totmove;
      }

      // We must update the movepid array for all particles, even local ones.

      PAssert(npid >= 0 && npid < patchesGlobal());
      movepid(i) = npid;

      POOMA_PATCHSWAPLAYOUT_DBG(dbgmsg << "Examining particle " << i)
      POOMA_PATCHSWAPLAYOUT_DBG(       << " with pos = " << pos(i))
      POOMA_PATCHSWAPLAYOUT_DBG(       << ": Global PatchID = " << npid)
      POOMA_PATCHSWAPLAYOUT_DBG(       << std::endl;)
    }

    // Return the total number of particles to locate on other patches.

    return totmove;
  }


  //============================================================
  // I/O
  //============================================================

  template<class Out>
  void print(Out &o) const
    {
      o << "SpatialLayout:\n";
      o << "    Field Layout = " << fieldLayout_m << "\n";
    }

private:
  //============================================================
  // Private data storage
  //============================================================

  // The geometry to use for this spatial layout

  Geometry_t geometry_m;

  // The field layout to use for this spatial layout

  FieldLayout_t fieldLayout_m;
};


//-----------------------------------------------------------------------------
//
// A specialization of the Inform traits used to say that SpatialLayout has
// a print method.
//
//-----------------------------------------------------------------------------

template <class G, class L>
std::ostream &operator<<(std::ostream &o, const SpatialLayout<G,L> &sl)
{
  sl.print(o);
  return o;
}


// } // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif // POOMA_PARTICLES_SPATIAL_LAYOUT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: SpatialLayout.h,v $   $Author: julianc $
// $Revision: 1.32 $   $Date: 2000/06/09 00:38:18 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
