// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// Iota
//
// function:
// iota
//-----------------------------------------------------------------------------

#ifndef POOMA_POOMA_INDICES_H
#define POOMA_POOMA_INDICES_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// iota(domain) is a handy function that returns an Array that contains
// an array of vectors whose elements correspond to index values.  For example,
// iota(Interval<2>(10,10))(3,4) is Vector<2,int>(3,4).
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Tiny/Vector.h"
#include "Engine/IndexFunctionEngine.h"
#include "Array/Array.h"
#include "Pooma/View.h"

//-----------------------------------------------------------------------------
//
// Full Description:
//
//-----------------------------------------------------------------------------

// Ack!!! Pooma names aren't supposed to be all caps.

class IotaFunctor
{
public:
  inline IotaFunctor() { }
  inline IotaFunctor(const IotaFunctor &) { }
  inline IotaFunctor &operator=(const IotaFunctor &) { return *this; }

  inline
  Vector<1, int> operator()(int i1) const
  {
    return Vector<1, int>(i1);
  }

  inline
  Vector<2, int> operator()(int i1, int i2) const
  {
    return Vector<2, int>(i1, i2);
  }

  inline
  Vector<3, int> operator()(int i1, int i2, int i3) const
  {
    return Vector<3, int>(i1, i2, i3);
  }

};

template<int Dim>
struct Iota
{
  typedef Array<Dim, Vector<Dim, int>, IndexFunction<IotaFunctor> > 
    Iota_t;
  typedef typename ComponentView<Loc<1>, Iota_t>::Type_t Index_t;
};

template<int Dim>
inline
typename Iota<Dim>::Iota_t
iota(const Interval<Dim> &domain)
{
  typedef typename Iota<Dim>::Iota_t Iota_t;
  return Iota_t(domain);
}

template<int Dim>
inline
typename Iota<Dim>::Index_t
iotaIndex(const Interval<Dim> &domain, int i)
{
  typedef typename Iota<Dim>::Iota_t Iota_t;
  return Iota_t(domain).comp(i);
}

inline
Array<1, Vector<1, int>, IndexFunction<IotaFunctor> >
iota(int i1);

inline
Array<1, Vector<1, int>, IndexFunction<IotaFunctor> >
iota(int i1)
{
  return Array<1, Vector<1, int>,
    IndexFunction<IotaFunctor> >(Interval<1>(i1));
}

inline
Array<2, Vector<2, int>, IndexFunction<IotaFunctor> >
iota(int i1, int i2);

inline
Array<2, Vector<2, int>, IndexFunction<IotaFunctor> >
iota(int i1, int i2)
{
  return Array<2, Vector<2, int>,
    IndexFunction<IotaFunctor> >(Interval<2>(i1, i2));
}

inline
Array<3, Vector<3, int>, IndexFunction<IotaFunctor> >
iota(int i1, int i2, int i3);

inline
Array<3, Vector<3, int>, IndexFunction<IotaFunctor> >
iota(int i1, int i2, int i3)
{
  return Array<3, Vector<3, int>,
    IndexFunction<IotaFunctor> >(Interval<3>(i1, i2, i3));
}

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_POOMA_INDICES_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Indices.h,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/07/20 15:39:28 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
