// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_TINY_BINARY_TINYMATRIX_OP_H
#define POOMA_TINY_BINARY_TINYMATRIX_OP_H

//-----------------------------------------------------------------------------
// Class: BinaryTinyMatrixEngine
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
// An engine class for representing the sum, product etc between two
// TinyMatrixs.  This is used as part of evaluating expressions.
//-----------------------------------------------------------------------------



//-----------------------------------------------------------------------------
//
// Full Description:
//
// This specialization of TinyMatrixEngine implements an optimization
// for expression evaluation.  Binary operations are done by constructing
// TinyMatrix with a BinaryTinyMatrixOp engine tag, then the result of the 
// operation is constructed by "copying out of" that one.  Copying each 
// element out of one of these causes the binary expression to be evaluated 
// for that point.
//
// This strategy reduces the number of copies that get made duing expression
// evaluations.
//
//-----------------------------------------------------------------------------

//
// The engine tag is just a tag.
//

template<class V1, class V2, class Op>
class BinaryTinyMatrixOp
{
};

//
// Specialization of TinyMatrixEngine for BinaryTinyMatrixOp.
// This has the same interface as any other TinyMatrixEngine of course,
// but the implementation has a const ref to a V1 and a V2.  Getting
// an element applies the operator to V1 and V2, returning the result.
//
// Since it stores references to V1 and V2, don't try to keep one
// of these things around!  This should only be used as part of
// expression evaluation.
//

template<int D1, int D2, class T, class V1, class V2, class Op>
class TinyMatrixEngine<D1,D2,T,BinaryTinyMatrixOp<V1,V2,Op> >
{

public:
 
  //----------------------------------------------------------------------
  // Typedefs
  //----------------------------------------------------------------------

  // Export the input types.
  enum { dimensions=2 };
  typedef T Element_t;
  typedef BinaryTinyMatrixOp<V1,V2,Op> EngineTag_t;

  // Return types for accessor functions.
  typedef T ConstElementRef_t;
  typedef T ElementRef_t;

  // Record the type of the current class.
  typedef TinyMatrixEngine<D1,D2,T, BinaryTinyMatrixOp<V1,V2,Op> > This_t;


  //----------------------------------------------------------------------
  // Constructors and Destructor

  // Construct from two TinyMatrixs and on operator tag.
  TinyMatrixEngine(const V1& v1, const V2& v2, Op op)
    : v1_m(v1), v2_m(v2), op_m(op) {}

  // Construct from two TinyMatrixs and let the op tag contruct itself.
  TinyMatrixEngine(const V1& v1, const V2& v2)
    : v1_m(v1), v2_m(v2) {}

  // Copy ctor just copies the references.
  TinyMatrixEngine(const This_t& x)
    : v1_m(x.v1_m), v2_m(x.v2_m), op_m(x.op_m) {}

  // Let the engine destroy itself.
  ~TinyMatrixEngine() {}

#if !POOMA_NO_TEMPLATE_FRIENDS
  template<int DD1,int DD2, class TT, class EE, int I, int J>
    friend struct TinyMatrixEngineElem;

private:
#endif

  // Just store const refs to objects of type V1 and V2.
  const V1& v1_m;
  const V2& v2_m;
  Op op_m;
};

//-----------------------------------------------------------------------------

//
// Specialization of TinyMatrixElem for BinaryTinyMatrixOp.
//
// The type of the return can be different for each element of the 
// tinymatrix.
//

template<int D1, int D2, class T, class V1, class V2, class Op, int I, int J>
struct TinyMatrixEngineElem<D1,D2,T,BinaryTinyMatrixOp<V1,V2,Op>, I, J >
{
  typedef TinyMatrixEngine<D1,D2,T,BinaryTinyMatrixOp<V1,V2,Op> > V;
  typedef typename TinyMatrixElem<V1,I,J>::Element_t T1;
  typedef typename TinyMatrixElem<V2,I,J>::Element_t T2;
  typedef typename BinaryReturn<T1,T2,Op>::Type_t Element_t;
  typedef Element_t ElementRef_t;
  typedef Element_t ConstElementRef_t;
  static Element_t get(const V& x) 
    { 
      return Op()(
        TinyMatrixElem<V1,I,J>::get(x.v1_m),
        TinyMatrixElem<V2,I,J>::get(x.v2_m));
    }
};


#endif // POOMA_TINY_BINARY_TINYMATRIX_OP_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BinaryTinyMatrixOp.h,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/03/07 13:18:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
