// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes
//  PatchSizeSyncer
//-----------------------------------------------------------------------------

#ifndef POOMA_CHEETAH_PATCHSIZESYNCER_H
#define POOMA_CHEETAH_PATCHSIZESYNCER_H

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Tulip/Messaging.h"
#include "Domain/Grid.h"

#include <vector>
#include <utility>

namespace Pooma {

//-----------------------------------------------------------------------------
// Overview: 
//
// PatchSizeSyncer is used to synchronize a set of Grid objects that are
// used to represent a set of contiguous patches (e.g. in DynamicLayout). 
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Full Description:
//
//   class PatchSizeSyncer
//
//   This class encapsulates the communication that must occur when
//   synchronizing the patch domains for a DynamicLayout.
//   PatchSizeSyncer is used by instantiating a version with a Grid<1>
//   object representing one's local patches and then calling
//   calcGlobalGrid with a grid object that will be filled with the
//   redistributed patches for the global Grid. The DynamicLayout can
//   then re-label the domains in its Nodes to match the new patch
//   distribution.
//
//-----------------------------------------------------------------------------

class PatchSizeSyncer
{
public:
  //============================================================
  // Typedefs
  //============================================================

  typedef Grid<1> Grid_t;

  //============================================================
  // Constructor & Destructor
  //============================================================

  // This sets up the local data for the calculation.

  PatchSizeSyncer(int contextKey, Grid_t &localGrid);

  // Cleans up the grid list on context 0.

  ~PatchSizeSyncer();

  //============================================================
  // Mutators
  //============================================================

  // This gathers the local grids on context 0, renormalizes
  // the local domains, constructs a new global grid, broadcasts
  // it to all contexts, and returns it to the caller.

 void calcGlobalGrid(Grid_t &globalGrid);

  // This is passed to Cheetah and is called when incoming messages 
  // are received.

  void receiveGrid(std::pair<int,Grid_t> &incoming);

private:

  //============================================================
  // Data
  //============================================================

  int myContext_m;

  int numContexts_m;

  int localKey_m;
  
  Grid_t localGrid_m;
  
  typedef std::pair<int,Grid_t *> Elem_t;
  
  std::vector<Elem_t> gridList_m;

  // PatchSizeSyncers should not be copied or assigned.

  PatchSizeSyncer(const PatchSizeSyncer &);
  PatchSizeSyncer &operator=(const PatchSizeSyncer &);

  // This is the tag used for message volleys. It is shared
  // by all PatchSizeSyncer objects.

  static int tag_s;

  // This is the Cheetah stuff. If we don't have Cheetah, this class should
  // work in serial (its a no-op) without sending any messages. All
  // Cheetah stuff should compile away.

#if POOMA_CHEETAH

  friend void Pooma::initializeCheetahHelpers(int contexts);
  friend void Pooma::finalizeCheetahHelpers();

  static Cheetah::MatchingHandler *handler_s;

#endif // POOMA_CHEETAH

};

} // namespace Pooma


// Serialize specialization for std::pair<int,Grid<1> >

#if POOMA_CHEETAH

namespace Cheetah {

template<>
class Serialize<CHEETAH, std::pair<int,Grid<1> > >
{
public:

  typedef Grid<1>                    Grid_t;
  typedef std::pair<int,Grid_t>      Pair_t;
  typedef Serialize<CHEETAH,Grid_t>  GridPacker_t;
  
  static inline int
  size(const Pair_t &a)
  {
    return sizeof(int) + GridPacker_t::size(a.second);
  }

  static inline int
  pack(const Pair_t &a, char *buffer)
  {
    *reinterpret_cast<int *>(buffer) = a.first;
    
    int nBytes = sizeof(int);
    nBytes += GridPacker_t::pack(a.second,buffer+nBytes);
        
    return nBytes;
  }

  static inline int
  unpack(Pair_t* &a, char *buffer)
  {
    int key = *reinterpret_cast<int *>(buffer);

    int nBytes = sizeof(int);
    
    Grid_t *pg;
    nBytes += GridPacker_t::unpack(pg, buffer+nBytes);
    
    a = new Pair_t(key,*pg);
    
    return nBytes;
  }

  static inline void
  cleanup(Pair_t* a)
  {
    delete a;
  }
};

} // namespace Cheetah

#endif // POOMA_CHEETAH

#endif // POOMA_CHEETAH_PATCHSIZESYNCER_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PatchSizeSyncer.h,v $   $Author: jac $
// $Revision: 1.3 $   $Date: 2000/05/30 17:44:06 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
