// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_STATISTICS_H
#define POOMA_UTILITIES_STATISTICS_H

//-----------------------------------------------------------------------------
// Classes: 
//   Implementation
//   Benchmark
//-----------------------------------------------------------------------------

#include <string>
#include <vector>
#include <utility>

class Inform;


//-----------------------------------------------------------------------------
// Statistics keeps statistics about a given POOMA job, and can report on
// a summary of these statistics when asked.
//
// This interface is extensible ... you can add new types of statistics
// by calling 'add(description, initval)' with a string description
// of the stat, and the initial value it should have.
//
// The StatisticsData class is a helper that contains the description and
// data.
//-----------------------------------------------------------------------------

namespace Pooma {

class StatisticsData
{
  friend class Statistics;
  
public:

  const std::string &description() const { return data_m.first; }
  long value() const { return data_m.second; }
  
  void increment(long val = 1) { data_m.second += val; }
  
private:

  // Make sure riff-raff can't instantiate these puppies.

  StatisticsData(const char *description, long initialValue = 0)
  : data_m(description, initialValue)
  { }
  
  ~StatisticsData() { }

  std::pair<std::string, long> data_m;
};

class Statistics {
private:

  //---------------------------------------------------------------------------
  // A NO-OP filter function. Up here because CW requires it.
  
  static long defaultFilter(long val);

public:
  
  //---------------------------------------------------------------------------
  // Constructor: initialize statistics.

  Statistics();

  //---------------------------------------------------------------------------
  // Destructor

  ~Statistics();

  //---------------------------------------------------------------------------
  // Print out the statistics to the given Inform object

  void print(Inform &, long (*filter)(long) = defaultFilter);

  //---------------------------------------------------------------------------
  // Add a statistics object to our list of stats ... return an integer
  // which is the index of the stat in our list, which can be used with
  // the 'incrementStat' and 'decrementStat' methods to change that statistic.

  StatisticsData *add(const char *description, long initval = 0) 
  {
    StatisticsData *sd = new StatisticsData(description, initval);
    statList_m.push_back(sd);
    return sd;
  }

private:

  //---------------------------------------------------------------------------
  // A vector of statistics data objects, which will be used to print
  // out the results at the end.

  std::vector<StatisticsData *> statList_m;
};

} // namespace Pooma

#endif // POOMA_UTILITIES_STATISTICS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Statistics.h,v $   $Author: swhaney $
// $Revision: 1.1 $   $Date: 2000/04/12 00:30:06 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
