// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_UNIQUE_H
#define POOMA_UTILITIES_UNIQUE_H

//-----------------------------------------------------------------------------
// Classes:
// Unique
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
// Overview:
//
// Unique: A simple utility class which can generate a sequence of unique
// values.  It has one static method 'get' which returns the next unique
// value, of type Unique::Value_t.  This should be assumed to be of integral
// type such as int or long.  The values are not guaranteed to be in any
// order, or to be positive or negative, they are just guaranteed to be
// different than the earlier calls to 'get'.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Threads/PoomaMutex.h"

//-----------------------------------------------------------------------------
//
// Full Description:
//
// The Overview pretty much sums it up.  The actual type for the Value_t
// parameter is long, so you can generate unique values up until you reach
// MAX_LONG.  Since we just need to access the static get method, there is
// really no reason to even instantiate a Unique object, just call Unique::get
//
//-----------------------------------------------------------------------------

class Unique
{
public:
  // The type generated by Unique
  typedef long Value_t;

  // The constructor and destructor for Unique do nothing.
  Unique()
    {
    }

  ~Unique()
    {
    }

  // The main public static method, which generates the next unique value.
  static inline Value_t get()
    {
      return next_s++;
    }

  // A thread-safe version of get
  static inline Value_t lockedGet()
    {
      mutex_s.lock();
      Value_t retval = get();
      mutex_s.unlock();
      return retval;
    }

private:
  // The next value to return when get() is called.
  static Value_t next_s;

  // A mutex for use in generating unique values from threads
  static Pooma::Mutex_t mutex_s;
};

// } // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_UTILITIES_UNIQUE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Unique.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/07 13:18:29 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
