// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_ARRAY_CREATELEAF_H
#define POOMA_ARRAY_CREATELEAF_H

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
// Classes:
// CreateLeaf<Expr>
// MakeReturn<T>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// These are the external traits classes that are used to build trees.
// CreateLeaf is used to convert arbitrary classes into expression objects.
// MakeReturn are used to combine expressions together with operators.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Forward declarations
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Traits classes
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// CreateLeaf - external functor class used to convert objects into the
// leaves of the expression tree.
//
// For the current version of pooma, our objects are arrays and we actually
// stick arrays at the leaves of the expression.  Also, we store expressions
// in arrays and want to extract the expression if we use the array in a
// bigger expression.
// (Other users of expression templates might wish to store iterates in the
// expression objects.)
//
// CreateLeaf<T> converts objects of type T to leaf objects and requires
// the following interface:
//
// typedef ... Leaf_t;  // The leaf object
// typedef ... Return_t;      // Type returned by make()
// Return_t make(const T&);   // make the leaf object from the T object
//
// Return_t should equivalent to Leaf_t. (Leaf_t needs to be able
// be constructed with a Return_t.)  We avoid making extra copies by building
// expression trees from references, so define Return_t to be a const ref to
// an Leaf_t.  (Returning by value would be bad, since we would create
// a temporary copy that won't survive until the whole expression is put
// together.)
//
// CreateLeaf is used to construct expression trees.  It should also be
// used when performing operations on the expression tree, such as forEach,
// in order to extract the expression.  For example:
// template<int D,class T,class E>
// void func(const Array<D,T,E>& array)
// {
//   forEach(CreateLeaf<Array<D,T,E> >::make(array),...,...);
// }
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Arrays are leaf objects, we just pass them through unless they have
// Expression engines. 

template<int Dim, class T, class EngineTag>
struct CreateLeaf<Array<Dim, T, EngineTag> >
{
  typedef Array<Dim, T, EngineTag> Input_t;
  typedef Reference<Input_t> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &a)
    {
      return Leaf_t(a);
    }
};

template<int Dim, class T, class Expr>
struct CreateLeaf<Array<Dim, T, ExpressionTag<Expr> > >
{
  typedef Array<Dim, T, ExpressionTag<Expr> > Input_t;
  typedef Expr Leaf_t;
  typedef const Leaf_t &Return_t;
  inline static
  Return_t make(const Input_t &a)
    {
      return a.engine().expression();
    }
};

//-----------------------------------------------------------------------------
// Special case for Scalar<Array> returns ErrorType to avoid
// hairy type computations.

template<int Dim, class T, class EngineTag>
struct CreateLeaf<Scalar<Array<Dim, T, EngineTag> > >
{
  typedef Scalar<Array<Dim, T, EngineTag> > Input_t;
  typedef Scalar<ErrorType> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &)
    {
      return ErrorType();
    }
};

//-----------------------------------------------------------------------------
// MakeReturn is tool used by operator functions to construct the
// expression tree representing that function.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Unary node version

template<class Op,class Leaf>
struct MakeReturn<UnaryNode<Op,Leaf> >
{
  typedef UnaryNode<Op,Leaf> Tree_t;

  typedef typename ForEach<Tree_t,
    DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename UnaryReturn<typename ForEach<Leaf,
    EvalLeaf<dim>,OpCombine>::Type_t,
    Op>::Type_t T_t;
  typedef Engine<dim,T_t,ExpressionTag<Tree_t> > Engine_t;
  typedef Array<dim,T_t,ExpressionTag<Tree_t > > Expression_t;

  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(Engine_t(tree));
    }
};

//-----------------------------------------------------------------------------
// Binary node version

template<class Op,class Left,class Right>
struct MakeReturn<BinaryNode<Op,Left,Right> >
{
  typedef BinaryNode<Op,Left,Right> Tree_t;

  typedef typename ForEach<Tree_t,
    DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename BinaryReturn<typename ForEach<Left,
    EvalLeaf<dim>,OpCombine>::Type_t,
    typename ForEach<Right,EvalLeaf<dim>,OpCombine>::Type_t,
    Op>::Type_t T_t;
  typedef Engine<dim,T_t,ExpressionTag<Tree_t> > Engine_t;
  typedef Array<dim,T_t,ExpressionTag<Tree_t > > Expression_t;

  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(Engine_t(tree));
    }
};

//-----------------------------------------------------------------------------
// Trinary node version

template<class Op,class Cl,class Tr,class Fl>
struct MakeReturn<TrinaryNode<Op,Cl,Tr,Fl> >
{
  typedef TrinaryNode<Op,Cl,Tr,Fl> Tree_t;

  typedef typename ForEach<Tree_t,
    DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename TrinaryReturn<typename ForEach<Cl,
    EvalLeaf<dim>,OpCombine>::Type_t,
    typename ForEach<Tr,EvalLeaf<dim>,OpCombine>::Type_t,
    typename ForEach<Fl,EvalLeaf<dim>,OpCombine>::Type_t,
    Op>::Type_t T_t;
  typedef Engine<dim,T_t,ExpressionTag<Tree_t> > Engine_t;
  typedef Array<dim,T_t,ExpressionTag<Tree_t > > Expression_t;

  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(Engine_t(tree));
    }
};

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CreateLeaf.h,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/07/20 15:36:24 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
