// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Array test 27: compressible operations: compress(), uncompress(),
//                elementsCompressed(), compressionFraction().
//-----------------------------------------------------------------------------

// Bring in Pooma array machinery.

#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Domain/Loc.h"
#include "Domain/Interval.h"
#include "Partition/UniformGridPartition.h"
#include "Layout/UniformGridLayout.h"
#include "Engine/BrickEngine.h"
#include "Engine/CompressibleBrick.h"
#include "Engine/MultiPatchEngine.h"
#include "Engine/RemoteEngine.h"
#include "Array/Array.h"
#include "Array/Reductions.h"


int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  Interval<1> I0(0,2), I1(3,5), I01(2,4);
  Interval<3> I3(6,6,6);
  Loc<3> blocks(2,2,2);
  UniformGridPartition<3> partition(blocks);   
  UniformGridLayout<3> replicated(I3, partition, ReplicatedTag()), 
    distributed(I3, partition, DistributedTag()); 
  Array<3, int, MultiPatch<UniformTag,CompressibleBrick> > a(replicated);
  Array<3, int, CompressibleBrick> b(I3);
  Array<3, int, MultiPatch<UniformTag,Remote<CompressibleBrick> > > c(distributed);

  a = 1;
  b = 1;
  c = 1;
  Pooma::blockAndEvaluate();

  tester.check("a #compressed", elementsCompressed(a), 216L);
  tester.check("a fraction", compressedFraction(a), 1.0);
    
  tester.check("b #compressed", elementsCompressed(b), 216L);
  tester.check("b fraction", compressedFraction(b), 1.0);
    
  tester.check("c #compressed", elementsCompressed(c), 216L);
  tester.check("c fraction", compressedFraction(c), 1.0);
    
  tester.check("bv #compressed", elementsCompressed(b(I1, I1, I0)), 27L);
  tester.check("bv fraction", compressedFraction(b(I1, I1, I0)), 1.0);

  a(4, 5, 1) = 2;
  b(4, 5, 1) = 2;
  c(4, 5, 1) = 2;

  tester.check("a #compressed", elementsCompressed(a), 189L);
  tester.check("a fraction", compressedFraction(a), 0.875, 1e-4);
  
  tester.check("b #compressed", elementsCompressed(b), 0L);
  tester.check("b fraction", compressedFraction(b), 0.0);

  tester.check("c #compressed", elementsCompressed(c), 189L);
  tester.check("c fraction", compressedFraction(c), 0.875, 1e-4);
  
  b(4, 5, 1) = 1;
  compress(b);
    
  tester.check("b #compressed", elementsCompressed(b), 216L);
  tester.check("b fraction", compressedFraction(b), 1.0);
  
  a(I1, I1, I0) = 2;
  c(I1, I1, I0) = 2;
  Pooma::blockAndEvaluate();

  tester.check("a #compressed", elementsCompressed(a), 216L);
  tester.check("a fraction", compressedFraction(a), 1.0);

  tester.check("c #compressed", elementsCompressed(c), 216L);
  tester.check("c fraction", compressedFraction(c), 1.0);
  
  uncompress(a);
  uncompress(b);
  uncompress(c);    

  tester.check("a #compressed", elementsCompressed(a), 0L);
  tester.check("a fraction", compressedFraction(a), 0.0);
     
  tester.check("b #compressed", elementsCompressed(b), 0L);
  tester.check("b fraction", compressedFraction(b), 0.0);
     
  tester.check("c #compressed", elementsCompressed(c), 0L);
  tester.check("c fraction", compressedFraction(c), 0.0);
  
  a(4, 5, 1) = 1;
  c(4, 5, 1) = 1;
  compress(a);
  compress(c);

  tester.check("a #compressed", elementsCompressed(a), 189L);
  tester.check("a fraction", compressedFraction(a), 0.875);
    
  tester.check("av #compressed", elementsCompressed(a(I01, I01, I01)), 23L);
  tester.check("av fraction", compressedFraction(a(I01, I01, I01)), 23./27);
    
  tester.check("c #compressed", elementsCompressed(c), 189L);
  tester.check("c fraction", compressedFraction(c), 0.875);
    
  tester.check("cv #compressed", elementsCompressed(c(I01, I01, I01)), 23L);
  tester.check("cv fraction", compressedFraction(c(I01, I01, I01)), 23./27);

  int ret = tester.results("array_test27");
  Pooma::finalize();
  return ret; 
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: array_test27.cpp,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/06/01 00:22:50 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
