// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
//   BCond<Field<Geom, T, EngineTag>, ComponentBC<N, Category> >
// Functions:
//   
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_COMPONENTBC_H
#define POOMA_BCONDS_COMPONENTBC_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// Class supporting componentwise BCs for Fields.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "BConds/FieldBCondBase.h"

//---------------------------------------------------------------------------
// Super-secret factory method. Actually makes the boundary condition. Called
// by create(). We use this two-step approach so we can specialize behavior
// for specific classes. This is an extra-special specialization for Fields.
//-----------------------------------------------------------------------------
  
template<class Geom, class T, class EngineTag, int N, class Category>
inline
BCondItem *createComponentwiseBoundaryCondition(
  const Field<Geom, T, EngineTag> &f, const ComponentBC<N, Category> &c)
{
  return 
    new BCond<Field<Geom, T, EngineTag>, ComponentBC<N, Category> >
      (f, c);
}  

// ----------------------------------------------------------------------------
// An actual boundary condition class for ComponentBC (partial specialization
// of the BCond class) for Fields
// ----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag, int N, class Category>
class BCond<Field<Geom, T, EngineTag>, ComponentBC<N, Category> > :
  public FieldBCondBase<Field<Geom, T, EngineTag> >
{
public:

  typedef BCond<Field<Geom, T, EngineTag>, ComponentBC<N, Category> > This_t;
  typedef FieldBCondBase<Field<Geom, T, EngineTag> > Base_t;
  
  //---------------------------------------------------------------------------
  // Constructors.

  BCond(const Field<Geom, T, EngineTag> &f, 
    const ComponentBC<N, Category> &bc);

  //---------------------------------------------------------------------------
  // Destructor.

  ~BCond() { }

  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void applyBoundaryCondition();
  
  // Clone and retarget this boundary condition.
  
  Base_t *retarget(const Field<Geom, T, EngineTag> &f) const;

private:

  // Factory method for creating the internal boundary condition.

  template<class ComponentSubject, class ComponentCategory>
  void create(const ComponentSubject &f, const ComponentCategory &c)
  {
    // Create the contained boundary condition.
  
    BCond<ComponentSubject, ComponentCategory> *bc =
      new BCond<ComponentSubject, ComponentCategory>(f, c);

    // Update our source and destination domains.
  
    srcDomain() = bc->srcDomain();
    destDomain() = bc->destDomain();
  
    // Set our internal pointer.
  
    bcItem_m = bc;
  }

  // Store a copy of the BCondCategory object used to construct this.

  BCondItem *bcItem_m;
  ComponentBC<N, Category> bc_m;
};

#include "BConds/ComponentBC.cpp"

#endif // POOMA_BCONDS_COMPONENTBC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ComponentBC.h,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/03/07 13:16:11 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
