// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// 
// PeriodicFaceBC
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_PERIODICFACEBC_H
#define POOMA_BCONDS_PERIODICFACEBC_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// PeriodicFaceBC: BCondCategory class setting all guard layers beyond a
//                   specified (logically) rectilinear mesh face to a the value
//                   from the non-guard element symmetrically across the face
//                   (the face is defined at the last vertex).
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "BConds/BCond.h"
#include "BConds/FieldBCondBase.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Full Description:
// 
// PeriodicFaceBC is a specialization of BCondCategory.
// 
// It represents a periodic boundary condition in one direction (perpendicular
// to the specified face) of a logically rectilinear domain.
//-----------------------------------------------------------------------------

class PeriodicFaceBC : public BCondCategory<PeriodicFaceBC>
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  PeriodicFaceBC(int face) : 
    face_m(face) { }
  PeriodicFaceBC(const PeriodicFaceBC &model) : 
    face_m(model.face())
  { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  PeriodicFaceBC &operator=(const PeriodicFaceBC &rhs)
  {
    face_m = rhs.face();

    return *this;
  }

  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  int face() const { return face_m; }

private:

  int face_m;

};


// ----------------------------------------------------------------------------
// An actual boundary condition class for PeriodicFaceBC (partial
// specialization of the BCond class)
// ----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
class BCond<Field<Geom, T, EngineTag>, PeriodicFaceBC> :
  public FieldBCondBase<Field<Geom, T, EngineTag> >
{
public:

  typedef BCond<Field<Geom, T, EngineTag>, PeriodicFaceBC> This_t;
  typedef FieldBCondBase<Field<Geom, T, EngineTag> > Base_t;
  typedef typename Base_t::Domain_t Domain_t;
  
  //---------------------------------------------------------------------------
  // Constructors.

  BCond(const Field<Geom, T, EngineTag> &f, const PeriodicFaceBC &bc);
  

  //---------------------------------------------------------------------------
  // Destructor.

  ~BCond() {};


  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void applyBoundaryCondition();
  
  // Clone and retarget this boundary condition.
  
  Base_t *retarget(const Field<Geom, T, EngineTag> &f) const;

private:

  // Store a copy of the BCondCategory object used to construct this.
  PeriodicFaceBC bc_m;

};

//-----------------------------------------------------------------------------
// Full Description:
// 
// AllPeriodicFaceBC is a functor class that sets all of a Field's BCs to be 
// PeriodicFaceBC.
//-----------------------------------------------------------------------------

class AllPeriodicFaceBC
{
public:

  template<class Geom, class T, class EngineTag>
  void operator()(Field<Geom, T, EngineTag> &f) const
    {
      for (int i = 0; i < 2 * Geom::dimensions; i++)
        f.addBoundaryCondition(PeriodicFaceBC(i));
    }
};

#include "BConds/PeriodicFaceBC.cpp"

#endif // POOMA_BCONDS_PERIODICFACEBC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PeriodicFaceBC.h,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/03/07 13:16:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
