// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Tests ConstantFaceBC
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"

#include <iostream>

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  // Useful ingredients for constructing and initializing Fields below:

  // Vertex domain:
  int nVerts = 4;
  Interval<1> I(nVerts), J(nVerts);
  Interval<2> domain(I,J);

  // Construct 2D mesh, zero-origin, unit-spacing:
  typedef UniformRectilinearMesh<2> Mesh_t;
  Mesh_t mesh(domain);

  // Construct 2D Cell-centered Geometry, with 1 guard layer:
  GuardLayers<2> gl(1);
  typedef DiscreteGeometry<Cell, Mesh_t> Geometry_t;
  Geometry_t geometry(mesh, gl);

  // --------------------------------------------------------------------------
  // Set a ConstantFaceBC on one face of a scalar Field:

  // Create scalar Field (Field<2,double,....>):
  Field<Geometry_t, double> a(geometry);

  // Assign it to the identifiable values:
  a(a.totalDomain()) = 9.0;
  std::cout << "~~~~~a~~~~~~~" << std::endl 
            << a.all() << std::endl << std::endl;

  // Use a BcondList, for convenience:
  BCondList bcCell;

  // Make ConstantFaceBC BCs, and add to the BCondList:
  double faceConstant = 1.0;
  // High X face
  bcCell.addBoundaryCondition(a, ConstantFaceBC<double>(1, faceConstant)); 

  Range<1> s(-1,nVerts - 1);
  Range<2> ss(s, s);
  
  bcCell.notifyPreRead(ss);
  std::cout << a << std::endl;
  
  
  // --------------------------------------------------------------------------
  // Set ConstantFaceBC on all faces of a vector Field:

  Field<Geometry_t, Vector<3> > b(geometry);

  // Assign it to the identifiable values:
  b(b.totalDomain()) = Vector<3>(9, 9, 9);
  std::cout << "~~~~~b~~~~~~~" << std::endl 
            << b.all() << std::endl << std::endl;

  // Use a BcondList, for convenience:
  BCondList vbc;

  // Make ConstantFaceBC BCs, and add to the BCondList
  Vector<3> faceConstantV(0.0);
  // Low X face
  vbc.addBoundaryCondition(b, ConstantFaceBC<Vector<3> >(0, faceConstantV)); 
  faceConstantV = Vector<3>(1.0);
  // High X face
  vbc.addBoundaryCondition(b, ConstantFaceBC<Vector<3> >(1, faceConstantV)); 
  faceConstantV = Vector<3>(2.0);
  // Low Y face
  vbc.addBoundaryCondition(b, ConstantFaceBC<Vector<3> >(2, faceConstantV)); 
  faceConstantV = Vector<3>(3.0);
  // High Y face
  vbc.addBoundaryCondition(b, ConstantFaceBC<Vector<3> >(3, faceConstantV)); 
  
  vbc.notifyPreRead(ss);
  std::cout << b.all() << std::endl;


  // --------------------------------------------------------------------------
  // Test Vert centering, with zeroizing:
  // --------------------------------------------------------------------------

  // Construct 2D Cell-centered Geometry, with 1 guard layer:
  typedef DiscreteGeometry<Vert, Mesh_t> VGeometry_t;
  VGeometry_t vGeometry(mesh, gl);

  // --------------------------------------------------------------------------
  // Set a ConstantFaceBC on all faces of a scalar Field:

  // Create scalar Field (Field<2,double,....>):
  Field<VGeometry_t, double> av(vGeometry);

  // Assign it to the identifiable values:
  av(av.totalDomain()) = 9.0;
  std::cout << "~~~~~av~~~~~~~" << std::endl 
            << av.all() << std::endl << std::endl;

  // Use a BcondList, for convenience:
  BCondList vbcVert;

  // Make ConstantFaceBC BCs, and add to the BCondList:
  typedef ConstantFaceBC<double> CFBC_t;
  faceConstant = 0.0;
  // Low X face
  vbcVert.addBoundaryCondition(av, CFBC_t(0, faceConstant, true)); 
  faceConstant = 1.0;
  // High X face
  vbcVert.addBoundaryCondition(av, CFBC_t(1, faceConstant, true)); 
  faceConstant = 2.0;
  // Low Y face
  vbcVert.addBoundaryCondition(av, CFBC_t(2, faceConstant, true)); 
  faceConstant = 3.0;
  // High Y face
  vbcVert.addBoundaryCondition(av, CFBC_t(3, faceConstant, true)); 

  Range<1> sv(-1,nVerts);
  Range<2> ssv(sv, sv);

  vbcVert.notifyPreRead(ssv);
  std::cout << av.all() << std::endl;

  Pooma::finalize();
  return 0;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestConstantFaceBC.cpp,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/07 13:16:14 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
