// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// POOMA include files

#include "Connect/Paws/PawsAppPointer.h"
#include "Utilities/PAssert.h"
#include "Pooma/Pooma.h"
#include "Pooma/Configuration.h"

// PAWS C++ API include files

#if POOMA_PAWS
#include "Paws/PawsApplication.h"
#endif


//-----------------------------------------------------------------------------
// static data members of PawsAppPointer
//-----------------------------------------------------------------------------

// The Paws application object.  There is only one of these at a time.  If
// you create a PawsAppPointer and this pointer is null, the application
// will initialize the Paws connection and contact the Paws controller.

PawsApplication *PawsAppPointer::paws_s  = 0;
int              PawsAppPointer::users_s = 0;


//----------------------------------------------------------------------
//
// PawsAppPointer constructor
// The constructor takes a string name; the type is "paws".
//
//----------------------------------------------------------------------

PawsAppPointer::PawsAppPointer(const char *conname, int argc, char *argv[])
  : connected_m(false)
{
  // if we do not have a Paws connection yet, create it now

  if (paws_s == 0)
    {
      PAssert(conname != 0);
#if POOMA_PAWS
      paws_s = new PawsApplication(conname, argc, argv,
				   Pooma::context(), Pooma::contexts());
#endif
    }

  // indicate we're one more object using Paws

  if (connected_m = (paws_s != 0))
    users_s++;
}


//----------------------------------------------------------------------
//
// PawsAppPointer destructor
//
//----------------------------------------------------------------------

PawsAppPointer::~PawsAppPointer()
{
  close();
}


//----------------------------------------------------------------------
//
// Return the Paws connection object.
//
//----------------------------------------------------------------------

PawsApplication &PawsAppPointer::paws() const
{
  PAssert(paws_s != 0);
  PAssert(connected());
  return *paws_s;
}


//----------------------------------------------------------------------
//
// Perform a Paws poll
//
//----------------------------------------------------------------------

void PawsAppPointer::poll()
{
#if POOMA_PAWS
  paws().poll();
#endif
}


//----------------------------------------------------------------------
//
// Wait for a ready signal from the Paws controller
//
//----------------------------------------------------------------------

void PawsAppPointer::ready()
{
#if POOMA_PAWS
  paws().ready();
#endif
}


//----------------------------------------------------------------------
//
// Shut down the connection, and put ourselves in an "unconnected" state.
//
//----------------------------------------------------------------------

void PawsAppPointer::close()
{
  // close paws connection if we're the last one

  if (connected() && --users_s == 0)
    {
#if POOMA_PAWS
      delete paws_s;
#endif
      paws_s = 0;
    }

  connected_m = false;
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PawsAppPointer.cmpl.cpp,v $   $Author: swhaney $
// $Revision: 1.4 $   $Date: 2000/03/07 13:16:21 $
// ----------------------------------------------------------------------
// ACL:rcsinfo



