// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// function:
// shrinkRight
// growRight
//-----------------------------------------------------------------------------

#ifndef POOMA_DOMAIN_SHRINK_H
#define POOMA_DOMAIN_SHRINK_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// shrinkRight(Interval<Dim>,Loc<Dim>) returns an Interval<Dim> which is
// Loc<Dim> shorter in each direction.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Interval.h"
#include "Domain/Loc.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
//-----------------------------------------------------------------------------

template<int Dim>
Interval<Dim> &
shrinkRightInPlace(Interval<Dim> &dom, const Loc<Dim> &s)
{
  for (int d = 0; d < Dim; ++d)
    {
      int a = dom[d].first();
      int b = dom[d].last() - s[d].first();
      dom[d] = Interval<1>(a, b);
    }
  return dom;
}

template<int Dim>
Interval<Dim> &
shrinkRightInPlace(Interval<Dim> &dom, int s)
{
  for (int d = 0; d < Dim; ++d)
    {
      int a = dom[d].first();
      int b = dom[d].last() - s;
      dom[d] = Interval<1>(a, b);
    }
  return dom;
}

template<int Dim>
Interval<Dim> &
growRightInPlace(Interval<Dim> &dom, const Loc<Dim> &s)
{
  for (int d = 0; d < Dim; ++d)
    {
      int a = dom[d].first();
      int b = dom[d].last() + s[d].first();
      dom[d] = Interval<1>(a, b);
    }
  return dom;
}

template<int Dim>
Interval<Dim> &
growRightInPlace(Interval<Dim> &dom, int s)
{
  for (int d = 0; d < Dim; ++d)
    {
      int a = dom[d].first();
      int b = dom[d].last() + s;
      dom[d] = Interval<1>(a, b);
    }
  return dom;
}

template<int Dim>
inline Interval<Dim> 
shrinkRight(const Interval<Dim> &dom, const Loc<Dim> &s)
{
  Interval<Dim> ret(dom);
  return shrinkRightInPlace(ret, s);
}

template<int Dim>
inline Interval<Dim> 
shrinkRight(const Interval<Dim> &dom, int s)
{
  Interval<Dim> ret(dom);
  return shrinkRightInPlace(ret, s);
}

template<int Dim>
inline Interval<Dim> 
growRight(const Interval<Dim> &dom, const Loc<Dim> &s)
{
  Interval<Dim> ret(dom);
  return growRightInPlace(ret, s);
}

template<int Dim>
inline Interval<Dim> 
growRight(const Interval<Dim> &dom, int s)
{
  Interval<Dim> ret(dom);
  return growRightInPlace(ret, s);
}


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_SHRINK_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Shrink.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/07/25 01:08:39 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
