// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// IntersectEngine
//-----------------------------------------------------------------------------

#ifndef POOMA_ENGINE_INTERSECTENGINE_H
#define POOMA_ENGINE_INTERSECTENGINE_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// IntersectEngine provides a common interface for applying the intersector
// object to various engines.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Engine/EngineFunctor.h"
#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
//
// EngineApply<IntersectorTag<Intersector> >
//
// IntersectEngine is simple wrapper that is used to apply intersector
// objects to engines.  It contains a reference to the intersector object
// and for engines with multiple patches it should hand the engine back
// to the intersector.  Typical use would look something like:
//
// IntersectEngine<Intersector> ie(intersector);
// engineFunctor(eng, ie);
//
// This level of indirection allows us to short-circut intersection for
// trivial engines and scalars, and the use of engineFunctor automatically
// deals with expression engines.
//
// The return value for intersection is a boolean that is currently unused.
// (The result of the intersection is stored in the intersector object.)
//
//-----------------------------------------------------------------------------

template<class Inter>
struct IntersectorTag { };

template<class Inter>
struct EngineApply<IntersectorTag<Inter> >
{
  typedef NullCombine Combine_t;

  EngineApply() { }
  ~EngineApply() { }
  inline EngineApply(Inter &i) : intersector_m(i) { }

  Inter &intersector_m;
};


//-----------------------------------------------------------------------------
// The default behaviour for IntersectEngine is to simply return true.
// We assert that the engine is not multi-patch.
//-----------------------------------------------------------------------------

template<class Eng, class Intersect>
struct DefaultEngineApply<Eng, IntersectorTag<Intersect> >
{
  typedef int Type_t;

  inline static
  Type_t apply(const Eng &,
	       const EngineApply<IntersectorTag<Intersect> > &)
  {
    // Engines that are multipatch must specialize this functor
    // to perform the correct intersection.
    CTAssert(!(Eng::multiPatch));
    return true;
  }
};

//-----------------------------------------------------------------------------
// Scalars require no intersection operations.
//-----------------------------------------------------------------------------

template<class T, class Intersect>
struct LeafFunctor<Scalar<T>, EngineApply<IntersectorTag<Intersect> > >
{
  typedef int Type_t;
  static inline
  Type_t apply(const Scalar<T> &,
	       const EngineApply<IntersectorTag<Intersect> > &)
  {
    return 0;
  }
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_ENGINE_INTERSECTENGINE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: IntersectEngine.h,v $   $Author: sa_smith $
// $Revision: 1.12 $   $Date: 2000/07/14 00:35:14 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
