// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Field test 5: Indexing Field expressions.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

#include <iostream>

//-----------------------------------------------------------------------------
// Main program:
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  // Create the physical domain.
  
  Interval<1> I(5), J(-1,5);
  
  Vector<1,double> origin(1.0);
  Vector<1,double> spacings(2.0);
  UniformRectilinearMesh<1> mesh(I, origin, spacings);
  
  // Create the geometry.
  
  DiscreteGeometry<Vert, UniformRectilinearMesh<1> > geom(mesh, GuardLayers<1>(1));
  
  // Make the field.
  
  Field<DiscreteGeometry<Vert, UniformRectilinearMesh<1> > > f(geom), g(geom);
  f(f.totalDomain()) = 10.0;
  g.all() = 30.0;
  
  // Make sure computation is finished.

  Pooma::blockAndEvaluate();

  tester.out() << f.physicalDomain() << std::endl;
  tester.out() << f.totalDomain() << std::endl;
    
  for (int i = 0; i < 5; i++)
    {
      f(i) = i + 1;
      g(i) = 2.0 * f(i);
    }
    
  tester.out() << f << std::endl;
  tester.out() << g << std::endl;
    
  tester.out() << (-f)(1) << std::endl;
  tester.check("(-f)(1)", (-f)(1), -2.0);
  tester.check("sum(-f)", sum(-f), -15.0);
  tester.check("sum(-f.all())", sum(-f.all()), -35.0);
  
  tester.out() << (-f).physicalDomain() << std::endl;
  tester.out() << (-f).totalDomain() << std::endl;
  tester.check("PD -f", (-f).physicalDomain(), I);
  tester.check("TD -f", (-f).totalDomain(), J);
  
  tester.out() << -(f + g) << std::endl;
  tester.out() << (-(f + g))(1) << std::endl;
  tester.check("(-(f + g))(1)", (-(f + g))(1), -6.0);
  tester.check("sum(-(f + g))", sum(-(f + g)), -45.0);
  tester.check("sum((-(f + g)).all())", sum((-(f + g)).all()), -45.0);

  tester.out() << (-(f + g)).physicalDomain() << std::endl;
  tester.out() << (-(f + g)).totalDomain() << std::endl;
  tester.check("PD -(f + g)", (-(f + g)).physicalDomain(), I);
  tester.check("TD -(f + g)", (-(f + g)).totalDomain(), I);

  int ret = tester.results("field_test5");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: field_test5.cpp,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/03/07 13:17:16 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
