// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   Cell
//   Vert
//-----------------------------------------------------------------------------

#ifndef POOMA_GEOMETRY_CENTERINGTAGS_H
#define POOMA_GEOMETRY_CENTERINGTAGS_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// Centering tag classes.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"
#include "Domain/Loc.h"

// ----------------------------------------------------------------------------
// Enumeration of basic 1D (one-component) centering types: for logically
// rectilinear meshes, face and edge centerings are a combination of cell and
// vertex along directions):
// ----------------------------------------------------------------------------

enum PoomaRectilinearCentering { cell=0, vertex=1, vert=1, undefined=23 };

// ----------------------------------------------------------------------------
// Highly generic centering tags. These have sensible definitions for a wide
// range of mesh types, not just logically rectilinear. These imply that all
// components of any multicomponent type like a vector Field are centered
// together.
// ----------------------------------------------------------------------------

// The mesh cell centers, or zone centers. You can redefine where the cell
// centers are in a geometry class using this:
struct Cell 
{ 
  Cell() { }
  Cell(const Cell &) { }
  Cell &operator=(const Cell &) { return *this; }

  PoomaRectilinearCentering 
  centeringTag(int, int = 0, int = 0) const
    { 
      return cell; 
    }
};

// The mesh vertices, or nodes:
struct Vert
{ 
  Vert() { }
  Vert(const Vert &) { }
  Vert &operator=(const Vert &) { return *this; }

  PoomaRectilinearCentering 
  centeringTag(int, int = 0, int = 0) const
    { 
      return vert; 
    }
};

// As the name implies, no centering position defined. Used primarily in
// constructing intermediate Field types in expressions, where the centering
// may be ill-defined (adding together conforming views of Vert and a
// Cell-centered Fields, for example):
struct NoCentering
{ 
  NoCentering() { }
  NoCentering(const NoCentering &) { }
  NoCentering &operator=(const NoCentering &) { return *this; }

  PoomaRectilinearCentering 
  centeringTag(int, int = 0, int = 0) const
    { 
      return undefined; 
    }
};

// ----------------------------------------------------------------------------
// These are specific to logically rectilinear meshes:
// ----------------------------------------------------------------------------

// Need this nontemplate base to share the static data member among all
// specializations of the class to be derived from it:

class RectilinearCenteringBase {
public:
  RectilinearCenteringBase() { }
  RectilinearCenteringBase(const RectilinearCenteringBase &) { }
  static const int allComponents = -999999;
};


// ----------------------------------------------------------------------------
// RectilinearCentering covers all other centering types with respect to
// logically rectilinear meshes. It allows for building up face and edge
// centerings by using cell and vertex centering in particular directions.  It
// also allows for specifying the cell/vertex centering of each component of a
// multicomponent type independently along each of the Dim directions
// defined. TensorRank specifies the number of integers needed to index a
// component (1 for Vector types, 2 for Tensor types, for example);
// NComponents is the number of components, assuming NComponents x NComponents
// for tensors. Could be modified later to admit non-square tensor types if
// needed.
// ----------------------------------------------------------------------------

template <int Dim, class RectilinearCenteringTag, 
  bool Componentwise = RectilinearCenteringTag::componentwise,
  int TensorRank = RectilinearCenteringTag::tensorRank, 
  int NComponents = RectilinearCenteringTag::nComponents>
class RectilinearCentering : public RectilinearCenteringBase
{
  // Interface note: any (partial) specialization of this class should provide
  // a non-static centeringTag() function whose purpose is to return the 1D
  // (one-component) centering for the requested component along the requested
  // direction. Can't put this in the general template here because static
  // stuff isn't grokked by specializations of this template class. Here is
  // what the function should look like:
  //           PoomaRectilinearCentering 
  //           centeringTag(int direction, 
  //                     int c1 = allComponents, int c1 = allComponents);
  //
  // Classes to be used for the RectilinearCenteringTag parameter should
  // contain static public data members to provide the default template values
  // specified above. See one of the examples below, such as FaceRCTag.
};

// These tags serve to specialize the RectilinearCenteringTag parameter, to
// represent the common types of non-Vert/non-Cell centerings (so far, the ones
// provided by POOMA r1):

// Centering on faces perpendicular to direction Direction:

template<int Direction>
class FaceRCTag {
public:
  static const bool componentwise = false;
  static const int tensorRank = 0;
  static const int nComponents = 1;
};

// Centering on edges parallel to direction Direction:

template<int Direction>
class EdgeRCTag {
public:
  static const bool componentwise = false;
  static const int tensorRank = 0;
  static const int nComponents = 1;
};

// Componentwise centering; each component centered on face perpendicular to
// that component's unit-vector direction:

template<int Dim>
class VectorFaceRCTag {
public:
  static const bool componentwise = true;
  static const int tensorRank = 1;
  static const int nComponents = Dim;
};

// Componentwise centering; each component centered on face parallel to that
// component's unit-vector direction:

template<int Dim>
class VectorEdgeRCTag {
public:
  static const bool componentwise = true;
  static const int tensorRank = 1;
  static const int nComponents = Dim;
};

// ----------------------------------------------------------------------------
// Partial specializations of RectilinearCentering containing definitions of
// various common rectilinear centerings of multicomponent types, as
// designated by the tag RectilinearCenteringTag classes defined above. These
// all provide a centeringTag() method as prescribed in the generic
// RectilinearCentering class definition comments above. Some also define a
// domainModifiers() method which specifies the index domain of the centering
// points relative to the vertex domain.
// ----------------------------------------------------------------------------

// All components have vertex centering along direction Direction, and cell
// centering along the other directions.

template<int Dim, int TensorRank, int NComponents, int Direction>
class 
RectilinearCentering<Dim, FaceRCTag<Direction>, false, TensorRank, NComponents>
  : public RectilinearCenteringBase
{
public:
  typedef RectilinearCentering<Dim, FaceRCTag<Direction>, 
    false, TensorRank, NComponents> This_t;

  RectilinearCentering() { }
  RectilinearCentering(const This_t &) { }
  This_t operator=(const This_t &) { return *this; }

  PoomaRectilinearCentering 
  centeringTag(int direction, 
	       int c1 = allComponents, int c2 = allComponents)
    const
  { 
    if (direction == Direction) {
      return vertex;
    } else {
      return cell; 
    }
  }

  const Loc<Dim> domainModifiers() const
  {
    Loc<Dim> dm;
    for (int d = 0; d < Dim; d++) {
      if (d == Direction) {
	dm[d] = 0;  // Use the vertex domain in direction Direction
      } else {
	dm[d] = -1; // Use the cell domain in the other directions
      }
    }
    return dm;
  }

};

// All components have cell centering along direction Direction, and vertex
// centering along the other directions.

template<int Dim, int TensorRank, int NComponents, int Direction>
class 
RectilinearCentering<Dim, EdgeRCTag<Direction>, false, TensorRank, NComponents>
    : public RectilinearCenteringBase
{
public:

  typedef RectilinearCentering<Dim, EdgeRCTag<Direction>, 
    false, TensorRank, NComponents> This_t;

  RectilinearCentering() { }
  RectilinearCentering(const This_t &) { }
  This_t operator=(const This_t &) { return *this; }

  PoomaRectilinearCentering 
  centeringTag(int direction, 
	       int c1 = allComponents, int c2 = allComponents)
    const
  { 
    if (direction == Direction) {
      return cell; 
    } else {
      return vertex;
    }
  }

  Loc<Dim> domainModifiers() const
  {
    Loc<Dim> dm;
    for (int d = 0; d < Dim; d++) {
      if (d == Direction) {
	dm[d] = -1; // Use the cell domain in direction Direction
      } else {
	dm[d] = 0;  // Use the vertex domain in the other directions
      }
    }
    return dm;
  }

};

// Component i has vertex centering along direction i, and cell centering
// along other directions. Only valid for TensorRank=1.

template<int Dim>
class RectilinearCentering<Dim, VectorFaceRCTag<Dim>, true, 1, Dim >
  : public RectilinearCenteringBase
{
public:

  typedef RectilinearCentering<Dim, VectorFaceRCTag<Dim>, true, 1, Dim> This_t;

  RectilinearCentering() { }
  RectilinearCentering(const This_t &) { }
  This_t operator=(const This_t &) { return *this; }

  PoomaRectilinearCentering 
  centeringTag(int direction, int c1 = allComponents, int c2 = allComponents)
    const
  { 
    if (direction == c1) {
      return vertex;
    } else {
      return cell; 
    }
  }

};

// Component i has cell centering along direction i, and vertex centering
// along other directions.  Only valid for TensorRank=1.

template<int Dim>
class RectilinearCentering<Dim, VectorEdgeRCTag<Dim>, true, 1, Dim>
  : public RectilinearCenteringBase
{
public:

  typedef 
  RectilinearCentering<Dim, VectorEdgeRCTag<Dim>, false, 1, Dim> This_t;

  RectilinearCentering() { }
  RectilinearCentering(const This_t &) { }
  This_t operator=(const This_t &) { return *this; }

  PoomaRectilinearCentering 
  centeringTag(int direction, int c1 = allComponents, int c2 = allComponents)
    const
  {
    if (direction == c1) {
      return cell; 
    } else {
      return vertex;
    }
  }

};

// ----------------------------------------------------------------------------
// Centering tags used for component views of fields with multiple centerings.
// Versions for vectors and tensors are supplied.
// ----------------------------------------------------------------------------

template<int NumComp, class Centering>
class RTComponent;

template<class Centering>
class RTComponent<1, Centering>
{
public:

  RTComponent(const Centering &centering, const Loc<1> &c)
  : centering_m(centering), c_m(c)
    { }
   
  PoomaRectilinearCentering 
  centeringTag(int d) const
    { 
      return centering_m.centeringTag(d, c_m[0].first()); 
    }
    
  const Loc<1> &components() const
    {
      return c_m;
    }
    
private:

  Centering centering_m;
  Loc<1> c_m;
};

template<class Centering>
class RTComponent<2, Centering>
{
public:

  RTComponent(const Centering &centering, const Loc<2> &c)
  : centering_m(centering), c_m(c)
  { }
   
  PoomaRectilinearCentering 
  centeringTag(int d) const
    { 
      return centering_m.centeringTag(d, c_m[0].first(), c_m[1].first()); 
    }

  const Loc<2> &components() const
    {
      return c_m;
    }
    
private:

  Centering centering_m;
  Loc<2> c_m;
};

#include "Geometry/CenteringTags.cpp"

#endif // POOMA_GEOMETRY_CENTERINGTAGS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CenteringTags.h,v $   $Author: swhaney $
// $Revision: 1.21 $   $Date: 2000/03/07 13:17:21 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
