// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   RectilinearGeometryBase<Centering, UniformRectilinearMesh<...> >
//-----------------------------------------------------------------------------

#ifndef POOMA_GEOMETRY_RECTILINEARGEOMETRYBASE_H
#define POOMA_GEOMETRY_RECTILINEARGEOMETRYBASE_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// RectilinearGeometryBase<Centering, Mesh>:
//
// RectilinearGeometryBase is a base class that provides the implementation
// for Geometries based on rectilinear meshes.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Loc.h"
#include "Domain/Region.h"
#include "Domain/Contains.h"
#include "Domain/Shrink.h"
#include "Geometry/CenteringTags.h"
#include "Geometry/DiscreteGeometry.h"
#include "Layout/GuardLayers.h"
#include "Tiny/Tensor.h"
#include "Tiny/Vector.h"

//-----------------------------------------------------------------------------
// Full Description:
// 
// RectilinearGeometryBase is partially specialized for different centerings.
// 
// The services it provides are documented in DiscreteGeometry.h.
// 
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
//
// General template.
//
///////////////////////////////////////////////////////////////////////////////

template<class Centering, class Mesh>
class RectilinearGeometryBase;

///////////////////////////////////////////////////////////////////////////////
//
// Vert-centered version.
//
///////////////////////////////////////////////////////////////////////////////

template<class Mesh>
class RectilinearGeometryBase<Vert, Mesh>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The centering type.
  
  typedef Vert Centering_t;
  
  // The mesh type.
  
  typedef Mesh Mesh_t;
  
  // This class.
  
  typedef RectilinearGeometryBase<Centering_t, Mesh_t> This_t;
  
  // The coordinate point type.
  
  typedef typename Mesh_t::PointType_t PointType_t;
  
  // The coordinate system.
  
  typedef typename Mesh_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename Mesh_t::Domain_t Domain_t;
  
  // Dimensions.
  
  enum { dimensions = Mesh_t::dimensions };
  enum { coordinateDimensions = Mesh_t::coordinateDimensions };
  
  // The guard layers type.
  
  typedef GuardLayers<dimensions> GuardLayers_t;

  // Are we component-centered?
  
  static const bool componentCentered = false;

  // Array types
  
  typedef typename Mesh_t::PositionsArray_t PositionArray_t;

  // Bounding box types

  typedef Region<dimensions, typename Mesh_t::AxisType_t> BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Default constructor. Must call initialize function to be a valid object.
  
  RectilinearGeometryBase() { }
  
  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  RectilinearGeometryBase(const Mesh_t &m, 
    const GuardLayers_t &gl = GuardLayers_t())
    : mesh_m(m),
      gl_m(gl)
  { }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  RectilinearGeometryBase(const This_t &model)
   : mesh_m(model.mesh()), 
     gl_m(model.guardLayers())
  { }

  //---------------------------------------------------------------------------
  // View constructor. The domain is assumed to be vertex-based so we don't
  // need to modify it at all.
  
  template <class ViewInitializer>
  RectilinearGeometryBase(const ViewInitializer &init, const Domain_t &dom)
   : mesh_m(init.mesh(), dom), 
     gl_m(0)
  { }
  
  //---------------------------------------------------------------------------
  // Initialize functions.
  
  void initialize(const This_t &model)
  {
    mesh_m = model.mesh();
    gl_m   = model.guardLayers();
  }
  
  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    mesh_m = m;
    gl_m   = gl;
  }

  //---------------------------------------------------------------------------
  // Copy assignment operator (shouldn't be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~RectilinearGeometryBase() { }

  //---------------------------------------------------------------------------
  // Accessors.

  Centering_t centering() const
  {
    return Centering_t();
  }
    
  const GuardLayers_t &guardLayers() const
  {
    return gl_m;
  }
    
  bool initialized() const
  {
    return mesh_m.initialized();
  }

  const Domain_t physicalDomain() const 
  {
    PAssert(initialized()); 
    return mesh_m.physicalDomain(); 
  }

  const Domain_t totalDomain() const 
  {
    PAssert(initialized());
    Domain_t td(physicalDomain());
    return guardLayers().addGuardLayersToDomain(td);
  }
  
  const Mesh_t &mesh() const 
  { 
    PAssert(initialized()); 
    return mesh_m; 
  }
  
  const PositionArray_t &x() const
  {
    PAssert(initialized()); 
    return mesh_m.vertexPositions();
  }

  BoundingRegion_t boundingBox(const Domain_t &domain) const
  {
    int i;

    CTAssert((int)Domain_t::dimensions == (int)dimensions);
    PAssert(initialized());
    PAssert(contains(totalDomain(), domain));

    // Find the indices of the boundary vertices

    Loc<dimensions> a1 = domain.firsts(), a2 = domain.firsts();
    Loc<dimensions> b1 = domain.lasts(),  b2 = domain.lasts();
    for (i = 0; i < dimensions; ++i)
      {
	if (a1[i].first() > physicalDomain()[i].first())
	  a1[i] -= 1;
	if (b2[i].first() < physicalDomain()[i].last())
	  b2[i] += 1;
      }

    // Calculate the positions of the cell midpoints

    PointType_t p1 = (x().read(a1) + x().read(a2)) * 0.5;
    PointType_t p2 = (x().read(b1) + x().read(b2)) * 0.5;

    // Use the cell midpoints to form a Region

    typedef typename BoundingRegion_t::OneDomain_t OneDBoundingRegion_t;

    BoundingRegion_t box;
    for (i=0; i < dimensions; ++i)
      box[i] = OneDBoundingRegion_t(p1(i), p2(i));

    return box;
  }

  inline Loc<dimensions> pointIndex(const PointType_t &point) const
  {
    // From the point in coordinate space, map back to the index
    // that represents the point in the Field in the "vicinity" of this
    // point.  This is made consistent with how the bounding box is
    // calculated, and takes into account the centering.

    return mesh_m.nearestVertex(point);
  }

  inline PointType_t indexPoint(const Loc<dimensions> &index) const
  {
    // From the point in index space, map back to the point
    // represented by the index.  This takes into account the centering.

    return mesh_m.vertexPositions().read(index);
  }

private:

  Mesh_t mesh_m;
  GuardLayers_t gl_m;
};


///////////////////////////////////////////////////////////////////////////////
//
// Cell-centered version.
//
///////////////////////////////////////////////////////////////////////////////

template<class Mesh>
class RectilinearGeometryBase<Cell, Mesh>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The centering type.
  
  typedef Cell Centering_t;
  
  // The mesh type.
  
  typedef Mesh Mesh_t;
  
  // This class.
  
  typedef RectilinearGeometryBase<Centering_t, Mesh_t> This_t;
  
  // The coordinate system.
  
  typedef typename Mesh_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The coordinate point type.
  
  typedef typename Mesh_t::PointType_t PointType_t;
  
  // The domain type.
  
  typedef typename Mesh_t::Domain_t Domain_t;
  
  // Dimensions.
  
  enum { dimensions = Mesh_t::dimensions };
  enum { coordinateDimensions = Mesh_t::coordinateDimensions };
  
  // The guard layers type.
  
  typedef GuardLayers<dimensions> GuardLayers_t;

  // Are we component-centered?
  
  static const bool componentCentered = false;

  // Position computation functor.
  
  class PositionFunctor
  {
    public:
    
      PositionFunctor()
      : geom_m(NULL)
      { }
      
      PositionFunctor(const This_t *geom)
      : geom_m(geom)
      { }
      
      PointType_t operator()(int i0) const
      {
        return geom_m->mesh().vertexPositions().read(i0) + 
          0.5 * geom_m->mesh().vertexDeltas().read(i0);
      }
      
      PointType_t operator()(int i0, int i1) const
      {
        return geom_m->mesh().vertexPositions().read(i0, i1) + 
          0.5 * geom_m->mesh().vertexDeltas().read(i0, i1);
      }

      PointType_t operator()(int i0, int i1, int i2) const
      {
        return geom_m->mesh().vertexPositions().read(i0, i1, i2) + 
          0.5 * geom_m->mesh().vertexDeltas().read(i0, i1, i2);
      }

    private:
    
      const This_t *geom_m;
  };

  // Array types
  
  typedef Array<dimensions, PointType_t, IndexFunction<PositionFunctor> > 
    PositionArray_t;
  
  // Bounding box types

  typedef Region<dimensions, typename Mesh_t::AxisType_t> BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Default constructor. Must call initialize function to be a valid object.
  
  RectilinearGeometryBase() { }
  
  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  RectilinearGeometryBase(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  : mesh_m(m), gl_m(gl), 
    positions_m(gl.addGuardLayersToDomain(m.physicalCellDomain()))
  {
    positions_m.engine().setFunctor(PositionFunctor(this)); 
  }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  RectilinearGeometryBase(const This_t &model)
  : mesh_m(model.mesh()), 
    gl_m(model.guardLayers()), 
    positions_m(model.x())
  { }

  //---------------------------------------------------------------------------
  // View constructor. The domain is assumed to be cell-based so we need to
  // grow it to give the right number of vertices.
  
  template <class ViewInitializer>
  RectilinearGeometryBase(const ViewInitializer &init, const Domain_t &dom)
   : mesh_m(init.mesh(), growRight(dom, 1)), 
     gl_m(0),
     positions_m(dom - dom.firsts())
  {
    positions_m.engine().setFunctor(PositionFunctor(this)); 
  }
  
  //---------------------------------------------------------------------------
  // Initialize functions.
  
  void initialize(const This_t &model)
  {
    initialize(model.mesh(), model.guardLayers());
  }
  
  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    PAssert(!initialized());
    mesh_m.initialize(m);
    gl_m.initialize(gl);
    positions_m.engine().setDomain(
      gl.addGuardLayersToDomain(m.physicalCellDomain())); 
    positions_m.engine().setFunctor(PositionFunctor(this)); 
  }

  //---------------------------------------------------------------------------
  // Copy assignment operator (shouldn't be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~RectilinearGeometryBase() { }
  
  //---------------------------------------------------------------------------
  // Accessors.

  Centering_t centering() const
  {
    return Centering_t();
  }
  
  const GuardLayers_t &guardLayers() const
  {
    return gl_m;
  }
    
  bool initialized() const
  {
    return mesh_m.initialized();
  }  

  const Mesh_t &mesh() const 
  { 
    PAssert(initialized()); 
    return mesh_m; 
  }
  
  const Domain_t physicalDomain() const 
  {
    PAssert(initialized()); 
    return mesh_m.physicalCellDomain(); 
  }
  
  const Domain_t totalDomain() const 
  {
    PAssert(initialized()); 
    return positions_m.domain(); 
  }
  
  const PositionArray_t &x() const
  {
    PAssert(initialized()); 
    return positions_m;
  }
  
  BoundingRegion_t boundingBox(const Domain_t &domain) const
  {
    int i;

    CTAssert((int)Domain_t::dimensions == (int)dimensions);
    PAssert(initialized());
    PAssert(contains(totalDomain(), domain));

    // Find the indices of the boundary vertices

    Loc<dimensions> a = domain.firsts();
    Loc<dimensions> b = domain.lasts();
    b += 1;

    // Calculate the positions of the vertices

    PointType_t p1 = mesh_m.vertexPositions().read(a);
    PointType_t p2 = mesh_m.vertexPositions().read(b);

    // Use the cell midpoints to form a Region

    typedef typename BoundingRegion_t::OneDomain_t OneDBoundingRegion_t;

    BoundingRegion_t box;
    for (i = 0; i < dimensions; ++i)
      box[i] = OneDBoundingRegion_t(p1(i), p2(i));

    return box;
  }

  inline Loc<dimensions> pointIndex(const PointType_t &point) const
  {
    // From the point in coordinate space, map back to the index
    // that represents the point in the Field in the "vicinity" of this
    // point.  This is made consistent with how the bounding box is
    // calculated, and takes into account the centering.

    return mesh_m.vertexBelow(point);
  }

  inline PointType_t indexPoint(const Loc<dimensions> &index) const
  {
    // From the point in index space, map back to the point
    // represented by the index.  This takes into account the centering.

    return ( (mesh_m.vertexPositions().read(index) +
              mesh_m.vertexPositions().read(index+1)) / 2.0);
  }

private:

  Mesh_t mesh_m;
  GuardLayers_t gl_m;  
  PositionArray_t positions_m;
};


///////////////////////////////////////////////////////////////////////////////
//
// RectilinearCentering<Dim,RectilinearCenteringTag,Componentwise,TensorRank,
// NComponents>-centered version.
// 
// For non-componentwise centerings, simpler.
// For componentwise centerings, partial specializations for TensorRank={1,2}
//
///////////////////////////////////////////////////////////////////////////////

// For non-componentwise centerings, only one partial specialization. Element
// type of position array is Position_t (a position vector type).

template<int Dim, class RectilinearCenteringTag, int TensorRank, 
  int NComponents, class Mesh>
class RectilinearGeometryBase<
  RectilinearCentering<Dim, RectilinearCenteringTag, false, TensorRank, 
    NComponents>, Mesh>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The centering type.

  typedef RectilinearCentering<Dim, RectilinearCenteringTag, false, 
    TensorRank, NComponents> Centering_t;

  // The mesh type.

  typedef Mesh Mesh_t;

  // This class.

  typedef RectilinearGeometryBase<Centering_t, Mesh_t> This_t;

  // The coordinate point type.

  typedef typename Mesh_t::PointType_t PointType_t;

  // The position array element type (one position vector per component):

  typedef PointType_t PositionArrayElement_t;

  // The coordinate system.

  typedef typename Mesh_t::CoordinateSystem_t CoordinateSystem_t;

  // Dimensions.

  enum { dimensions = Mesh_t::dimensions };
  enum { coordinateDimensions = Mesh_t::coordinateDimensions };
  
  // The guard layers type.
  
  typedef GuardLayers<dimensions> GuardLayers_t;

  // The domain type.

  typedef typename Mesh_t::Domain_t Domain_t;

  // Are we component-centered?
  
  static const bool componentCentered = true;

  // Position computation functor.

  class PositionFunctor
  {
  public:
   
    PositionFunctor() : geom_m(NULL) { }
   
    PositionFunctor(const This_t *geom) : geom_m(geom) { }
   
    PositionArrayElement_t operator()(int i0) const
    {
      PositionArrayElement_t a(NComponents);
      for (int d = 0; d < Dim; d++) 
        {
	  if (geom_m->centering().centeringTag(d) == cell) 
	    {
	      a(d) = geom_m->mesh().vertexPositions().comp(d).read(i0) + 
		0.5 * geom_m->mesh().vertexDeltas().comp(d).read(i0);
	    } 
	  else 
	    {
	      // geom_m->centering().centeringTag(d) == vert.
	      
	      a(d) = geom_m->mesh().vertexPositions().comp(d).read(i0);
	    }
        }
      return a;
    }
   
    PositionArrayElement_t operator()(int i0, int i1) const
    {
      PositionArrayElement_t a(NComponents);
      for (int d = 0; d < Dim; d++) 
        {
	  if (geom_m->centering().centeringTag(d) == cell) 
	    {
	      a(d) = 
	        geom_m->mesh().vertexPositions().comp(d).read(i0, i1) + 
		0.5 * geom_m->mesh().vertexDeltas().comp(d).read(i0, i1);
	    } 
	  else 
	    {
	      // geom_m->centering().centeringTag(d) == vert.
	      
	      a(d) = 
	        geom_m->mesh().vertexPositions().comp(d).read(i0, i1);
	    }
        }
      return a;
    }

    PositionArrayElement_t operator()(int i0, int i1, int i2) const
    {
      PositionArrayElement_t a(NComponents);
      for (int d = 0; d < Dim; d++) 
        {
	  if (geom_m->centering().centeringTag(d) == cell) 
	    {
	      a(d) = 
	        geom_m->mesh().vertexPositions().comp(d).read(i0, i1, i2) + 
		0.5 * geom_m->mesh().vertexDeltas().comp(d).read(i0, i1, i2);
	    } 
	  else 
	    {
	      // geom_m->centering().centeringTag(d) == vert.
	      
	      a(d) = 
	        geom_m->mesh().vertexPositions().comp(d).read(i0, i1, i2);
	    }
        }
      return a;
    }
 
  private:
 
    const This_t *geom_m;
  };

  // Array types

  typedef Array<Dim, PositionArrayElement_t, 
    IndexFunction<PositionFunctor> > PositionArray_t;

  // Bounding box types

  typedef Region<dimensions, typename Mesh_t::AxisType_t> BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.

  RectilinearGeometryBase(const Mesh_t &m, 
    const GuardLayers_t &gl = GuardLayers_t()) 
    : gl_m(gl), mesh_m(m), 
      physDomain_m(
        growRight(m.physicalDomain(), centering().domainModifiers())),
      positions_m(gl.addGuardLayersToDomain(physDomain_m))
  {
    positions_m.engine().setFunctor(PositionFunctor(this));
  }

  //---------------------------------------------------------------------------
  // Copy constructor.

  RectilinearGeometryBase(const This_t &model) 
  : centering_m(model.centering()), gl_m(model.guardLayers()), 
    mesh_m(model.mesh()), physDomain_m(model.physicalDomain()),
    positions_m(model.x())
  { }

  //---------------------------------------------------------------------------
  // View constructor. We need to adjust the domain according to the modifiers
  // associated with the centering.
  
  template <class ViewInitializer>
  RectilinearGeometryBase(const ViewInitializer &init, const Domain_t &dom)
   : mesh_m(init.mesh(), growRight(dom, -centering().domainModifiers())),
     gl_m(0),
     physDomain_m(dom - dom.firsts()),
     positions_m(physDomain_m)
  {
    positions_m.engine().setFunctor(PositionFunctor(this));
  }
  
  //---------------------------------------------------------------------------
  // Initialize functions.
  
  void initialize(const This_t &model)
  {
    initialize(model.mesh(), model.guardLayers());
  }
  
  void initialize(const Mesh_t &m, const GuardLayers_t &gl = GuardLayers_t())
  {
    PAssert(!initialized());
    mesh_m.initialize(m);
    gl_m.initialize(gl);
    positions_m.engine().setDomain(
      gl.addGuardLayersToDomain(m.physicalDomain())); 
    positions_m.engine().setFunctor(PositionFunctor(this)); 
  }

  //---------------------------------------------------------------------------
  // Copy assignment operator (shouldn't be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~RectilinearGeometryBase() { }

  //---------------------------------------------------------------------------
  // Accessors.

  const Centering_t &centering() const
  {
    return centering_m;
  }
  
  const GuardLayers_t &guardLayers() const
  {
    return gl_m;
  }

  bool initialized() const
  {
    return mesh_m.initialized();
  }

  const Mesh_t &mesh() const 
  { 
    PAssert(initialized()); 
    return mesh_m; 
  }

  const Domain_t &physicalDomain() const 
  {
    PAssert(initialized()); 
    return physDomain_m; 
  }

  const Domain_t totalDomain() const 
  {
    PAssert(initialized()); 
    return positions_m.domain(); 
  }

  const PositionArray_t &x() const
  {
    PAssert(initialized()); 
    return positions_m;
  }

private:

  Centering_t centering_m;
  GuardLayers_t gl_m;
  Mesh_t mesh_m;
  Domain_t physDomain_m;
  PositionArray_t positions_m;
};

// For componentwise centerings, partially specialize on tensor rank. Since the
// only one that exists so far is VectorFace, only need to put in tensor rank
// 1, really; but, since I already did TensorRank=2, leave it here for now.

// TensorRank = 1
// Elementype of position array is Vector<Dim,Position_t> (one position vector
// for each component of the rank-1 Field-element type).

template<int Dim, class RectilinearCenteringTag, int NComponents, class Mesh>
class RectilinearGeometryBase<
  RectilinearCentering<Dim, RectilinearCenteringTag, true, 1, NComponents>, 
  Mesh>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The centering type.

  typedef RectilinearCentering<Dim, RectilinearCenteringTag, true,
    1, NComponents> Centering_t;

  // The mesh type.
 
  typedef Mesh Mesh_t;

  // This class.

  typedef RectilinearGeometryBase<Centering_t, Mesh_t> This_t;

  // The coordinate point type.

  typedef typename Mesh_t::PointType_t PointType_t;

  // The position array element type (one position vector per component):

  typedef Vector<NComponents, PointType_t> PositionArrayElement_t;

  // The coordinate system.

  typedef typename Mesh_t::CoordinateSystem_t CoordinateSystem_t;

  // The domain type.

  typedef typename Mesh_t::Domain_t Domain_t;

  // Dimensions.

  enum { dimensions = Mesh_t::dimensions };
  enum { coordinateDimensions = Mesh_t::coordinateDimensions };
  
  // The guard layers type.
  
  typedef GuardLayers<dimensions> GuardLayers_t;

  // Are we component-centered?
  
  static const bool componentCentered = true;

  // Position computation functor.

  class PositionFunctor
  {
  public:
      
    PositionFunctor() : geom_m(NULL) { }
      
    PositionFunctor(const This_t *geom) : geom_m(geom) { }
      
    PositionArrayElement_t operator()(int i0) const
    {
      PositionArrayElement_t a(NComponents);
      for (int c = 0; c < NComponents; c++) 
        {
	  for (int d = 0; d < Dim; d++) 
	    {
	      if (geom_m->centering().centeringTag(d,c) == cell) 
	        {
	          a(c)(d) = 
	            geom_m->mesh().vertexPositions().comp(d).read(i0) + 
		    0.5 * geom_m->mesh().vertexDeltas().comp(d).read(i0);
	        } 
	      else 
	        {
	          // geom_m->centering().centeringTag(d,c) == vert.
	          
	          a(c)(d) = 
	            geom_m->mesh().vertexPositions().comp(d).read(i0);
	        }
	    }
        }
      return a;
    }
      
    PositionArrayElement_t operator()(int i0, int i1) const
    {
      PositionArrayElement_t a(NComponents);
      for (int c = 0; c < NComponents; c++) 
        {
	  for (int d = 0; d < Dim; d++) 
	    {
	      if (geom_m->centering().centeringTag(d,c) == cell) 
	        {
	          a(c)(d) = 
	            geom_m->mesh().vertexPositions().comp(d).read(i0, i1) + 
		    0.5 * geom_m->mesh().vertexDeltas().comp(d).read(i0, i1);
	        } 
	      else 
	        {
	          // geom_m->centering().centeringTag(d,c) == vert.
	          
	          a(c)(d) = 
	            geom_m->mesh().vertexPositions().comp(d).read(i0, i1);
	        }
	    }
        }
      return a;
    }

    PositionArrayElement_t operator()(int i0, int i1, int i2) const
    {
      PositionArrayElement_t a(NComponents);
      for (int c = 0; c < NComponents; c++) 
        {
	  for (int d = 0; d < Dim; d++) 
	    {
	      if (geom_m->centering().centeringTag(d,c) == cell) 
	        {
	          a(c)(d) = 
		    geom_m->mesh().vertexPositions().comp(d).read(i0,i1,i2) + 
		    0.5 * geom_m->mesh().vertexDeltas().comp(d).read(i0,i1,i2);
	        } 
	      else 
	        {
	          // geom_m->centering().centeringTag(d,c) == vert.
	          
	          a(c)(d) = 
		    geom_m->mesh().vertexPositions().comp(d).read(i0,i1,i2);
	        }
	    }
        }
      return a;
    }
    
  private:
    
    const This_t *geom_m;
  };
  
  // Array types

  typedef Array<Dim, PositionArrayElement_t, 
    IndexFunction<PositionFunctor> > PositionArray_t;

  // Bounding box types

  typedef Region<dimensions, typename Mesh_t::AxisType_t> BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.
  
  RectilinearGeometryBase(const Mesh_t &m, 
    const GuardLayers_t &gl = GuardLayers_t()) 
    : gl_m(gl), mesh_m(m), 
      positions_m(gl.addGuardLayersToDomain(m.physicalDomain()))
  {
    positions_m.engine().setFunctor(PositionFunctor(this));
  }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  RectilinearGeometryBase(const This_t &model) 
  : centering_m(model.centering()), gl_m(model.guardLayers()), 
    mesh_m(model.mesh()), positions_m(model.x())
  { }

  //---------------------------------------------------------------------------
  // View constructor. The domain is assumed to be vertex-based so we don't
  // need to modify it at all.
  
  template <class ViewInitializer>
  RectilinearGeometryBase(const ViewInitializer &init, const Domain_t &dom)
   : mesh_m(init.mesh(), dom),
     gl_m(0),
     positions_m(dom - dom.firsts())
  {
    positions_m.engine().setFunctor(PositionFunctor(this));
  }
  
  //---------------------------------------------------------------------------
  // Initialize functions.
  
  void initialize(const This_t &model)
  {
    initialize(model.mesh(), model.guardLayers());
  }
  
  void initialize(const Mesh_t &m, 
    const GuardLayers_t &gl = GuardLayers_t())
  {
    PAssert(!initialized());
    mesh_m.initialize(m);
    gl_m.initialize(gl);
    positions_m.engine().setDomain(
      gl.addGuardLayersToDomain(m.physicalDomain())); 
    positions_m.engine().setFunctor(PositionFunctor(this)); 
  }

  //---------------------------------------------------------------------------
  // Copy assignment operator (shouldn't be called).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~RectilinearGeometryBase() { }

  //---------------------------------------------------------------------------
  // Accessors.  

  const Centering_t &centering() const
  {
    return centering_m;
  }

  const GuardLayers_t &guardLayers() const
  {
    return gl_m;
  }

  bool initialized() const
  {
    return mesh_m.initialized();
  }
  
  const Mesh_t &mesh() const 
  { 
    PAssert(initialized()); 
    return mesh_m; 
  }
  
  const Domain_t physicalDomain() const 
  {
    PAssert(initialized()); 
    return mesh_m.physicalDomain(); 
  }
  
  const Domain_t totalDomain() const 
  {
    PAssert(initialized()); 
    return positions_m.domain(); 
  }
  
  const PositionArray_t &x() const
  {
    PAssert(initialized()); 
    return positions_m;
  }
  
private:

  Centering_t centering_m;
  GuardLayers_t gl_m;  
  Mesh_t mesh_m;
  PositionArray_t positions_m;
};

// TensorRank = 2
// Elementype of position array is Tensor<Dim,Position_t> (one position
// vector for each component of the rank-1 Field-element type).

template<int Dim, class RectilinearCenteringTag, int NComponents, class Mesh>
class RectilinearGeometryBase<
  RectilinearCentering<Dim, RectilinearCenteringTag, true, 2, NComponents>, 
  Mesh>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // The centering type.
 
  typedef RectilinearCentering<Dim, RectilinearCenteringTag, true, 
    2, NComponents> Centering_t;

  // The mesh type.
 
  typedef Mesh Mesh_t;

  // This class.
 
  typedef RectilinearGeometryBase<Centering_t, Mesh_t> This_t;

  // The coordinate point type.

  typedef typename Mesh_t::PointType_t PointType_t;

  // The position array element type (one position vector per component):

  typedef Tensor<NComponents, PointType_t> PositionArrayElement_t;

  // The coordinate system.

  typedef typename Mesh_t::CoordinateSystem_t CoordinateSystem_t;

  // The domain type.

  typedef typename Mesh_t::Domain_t Domain_t;

  // Dimensions.

  enum { dimensions = Mesh_t::dimensions };
  enum { coordinateDimensions = Mesh_t::coordinateDimensions };
  
  // The guard layers type.
  
  typedef GuardLayers<dimensions> GuardLayers_t;

  // Are we component-centered?
  
  static const bool componentCentered = true;

  // Position computation functor.
  class PositionFunctor
  {
  public:

    PositionFunctor() : geom_m(NULL) { }

    PositionFunctor(const This_t *geom) : geom_m(geom) { }

    PositionArrayElement_t operator()(int i0) const
    {
      PositionArrayElement_t a(NComponents);
      for (int c = 0; c < NComponents; c++) 
        {
	  for (int c2 = 0; c2 < NComponents; c2++) 
	    {
	      for (int d = 0; d < Dim; d++) 
	        {
	          if (geom_m->centering().centeringTag(d,c,c2) == cell) 
	            {
	              a(c,c2)(d) = 
	                geom_m->mesh().vertexPositions().comp(d).read(i0) + 
			0.5 * geom_m->mesh().vertexDeltas().comp(d).read(i0);
	            } 
	          else 
	            {
	              // geom_m->centering().centeringTag(d,c,c2) == vert.
	              
	              a(c,c2)(d) = 
	                geom_m->mesh().vertexPositions().comp(d).read(i0);
	            }
	        }
	    }
        }
      return a;
    }

    PositionArrayElement_t operator()(int i0, int i1) const
    {
      PositionArrayElement_t a(NComponents);
      for (int c = 0; c < NComponents; c++) 
        {
	  for (int c2 = 0; c2 < NComponents; c2++) 
	    {
	      for (int d = 0; d < Dim; d++) 
	        {
	          if (geom_m->centering().centeringTag(d,c,c2) == cell) 
	            {
	              a(c,c2)(d) = 
	                geom_m->mesh().vertexPositions().comp(d).read(i0,i1) + 
			0.5 * 
                        geom_m->mesh().vertexDeltas().comp(d).read(i0,i1);
	            } 
	          else 
	            {
	              // geom_m->centering().centeringTag(d,c,c2) == vert.
	              
	              a(c,c2)(d) = 
	                geom_m->mesh().vertexPositions().comp(d).read(i0,i1);
	            }
	        }
	    }
        }
      return a;
    }

    PositionArrayElement_t operator()(int i0, int i1, int i2) const
    {
      PositionArrayElement_t a(NComponents);
      for (int c = 0; c < NComponents; c++) 
        {
	  for (int c2 = 0; c2 < NComponents; c2++) 
	    {
	      for (int d = 0; d < Dim; d++) 
	        {
	          if (geom_m->centering().centeringTag(d,c,c2) == cell) 
	            {
	              a(c,c2)(d) = 
	                geom_m->
                        mesh().vertexPositions().comp(d).read(i0,i1,i2) + 
			0.5 * 
                        geom_m->mesh().vertexDeltas().comp(d).read(i0,i1,i2);
	            } 
	          else 
	            {
	              // geom_m->centering().centeringTag(d,c,c2) == vert.
	              
	              a(c,c2)(d) = 
	                geom_m->
                        mesh().vertexPositions().comp(d).read(i0,i1,i2);
	            }
	        }
	    }
        }
      return a;
    }

  private:

    const This_t *geom_m;
  };

  // Array types.
  
  typedef Array<Dim, PositionArrayElement_t, 
    IndexFunction<PositionFunctor> > PositionArray_t;

  // Bounding box types

  typedef Region<dimensions, typename Mesh_t::AxisType_t> BoundingRegion_t;

  //---------------------------------------------------------------------------
  // Constructor using a mesh and (optionally) a GuardLayers object.

  RectilinearGeometryBase(const Mesh_t &m, 
    const GuardLayers_t &gl = GuardLayers_t()) 
    : gl_m(gl), mesh_m(m), 
      positions_m(gl.addGuardLayersToDomain(m.physicalDomain()))
  {
    positions_m.engine().setFunctor(PositionFunctor(this));
  }

  //---------------------------------------------------------------------------
  // Copy constructor.

  RectilinearGeometryBase(const This_t &model)
    : centering_m(model.centering()), gl_m(model.guardLayers()), 
      mesh_m(model.mesh()), positions_m(model.x())
  { }

  //---------------------------------------------------------------------------
  // View constructor. The domain is assumed to be vertex-based so we don't
  // need to modify it at all.
  
  template <class ViewInitializer>
  RectilinearGeometryBase(const ViewInitializer &init, const Domain_t &dom)
   : mesh_m(init.mesh(), dom),
     gl_m(0),
     positions_m(dom - dom.firsts())
  {
    positions_m.engine().setFunctor(PositionFunctor(this));
  }
  
  //---------------------------------------------------------------------------
  // Initialize functions.
  
  void initialize(const This_t &model)
  {
    initialize(model.mesh(), model.guardLayers());
  }
  
  void initialize(const Mesh_t &m, 
    const GuardLayers_t &gl = GuardLayers_t())
  {
    PAssert(!initialized());
    mesh_m.initialize(m);
    gl_m.initialize(gl);
    positions_m.engine().setDomain(
      gl.addGuardLayersToDomain(m.physicalDomain())); 
    positions_m.engine().setFunctor(PositionFunctor(this)); 
  }

  //---------------------------------------------------------------------------
  // Copy assignment operator (shallow for mesh due to mesh semantics).
  
  This_t &operator=(const This_t &)
  {
    PAssert(false);
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine for us.
  
  ~RectilinearGeometryBase() { }

  //---------------------------------------------------------------------------
  // Accessors.

  const Centering_t &centering() const
  {
    return centering_m;
  }

  const GuardLayers_t &guardLayers() const
  {
    return gl_m;
  }

  bool initialized() const
  {
    return mesh_m.initialized();
  }

  const Mesh_t &mesh() const 
  { 
    PAssert(initialized()); 
    return mesh_m; 
  }

  const Domain_t physicalDomain() const 
  {
    PAssert(initialized()); 
    return mesh_m->physicalDomain(); 
  }
  
  const Domain_t totalDomain() const 
  {
    PAssert(initialized()); 
    return positions_m.domain(); 
  }

  const PositionArray_t &x() const
  {
    PAssert(initialized()); 
    return positions_m;
  }

private:

  Centering_t centering_m;
  GuardLayers_t gl_m;  
  Mesh_t mesh_m;
  PositionArray_t positions_m;
};

#endif // POOMA_GEOMETRY_RECTILINEARGEOMETRYBASE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RectilinearGeometryBase.h,v $   $Author: julianc $
// $Revision: 1.18 $   $Date: 2000/07/24 15:59:05 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
