// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef MULTISTORAGE_OBJECT_SETS_H
#define MULTISTORAGE_OBJECT_SETS_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "IO/ObjectSets.h"
#include <vector>

//-----------------------------------------------------------------------------
// Typedefs
typedef long StorageSetID_t;

//-----------------------------------------------------------------------------
// Logical object reference used for object sets with multiple
// object storage components.
struct GlobalObjectRef{
  StorageSetID_t storageSetID;
  ObjectID_t localID;
};

//-----------------------------------------------------------------------------
// Comparator for object reference structs.
//
bool operator<(const GlobalObjectRef& r1, const GlobalObjectRef& r2);
bool operator<(const GlobalObjectRef& r1, const GlobalObjectRef& r2){
  if((r1.storageSetID < r2.storageSetID) ||
     (r1.localID < r2.localID) ){
    return true;
  }
  else{
    return false;
  }
}

//-----------------------------------------------------------------------------
// Equality for object reference structs.
bool operator==(const GlobalObjectRef& r1, const GlobalObjectRef& r2);
bool operator==(const GlobalObjectRef& r1, const GlobalObjectRef& r2){
  if((r1.storageSetID == r2.storageSetID) &&
     (r1.localID == r2.localID) ){
       return true;
     }
  else{
    return false;
  }
}

//-----------------------------------------------------------------------------
// Descriptor containing the object reference information plus the
// global ID; used for storing and retrieving global object references
// in the multi-file object set dictionary.
struct GlobalObjectDescriptor{
  ObjectID_t globalID;
  StorageSetID_t storageSetID;
  ObjectID_t localID;
};

//-----------------------------------------------------------------------------
// Serializers and type identification for object descriptor.

template <class Stream>
int serialize(Stream& s, const GlobalObjectDescriptor& d);
template <class Stream>
int serialize(Stream& s, const GlobalObjectDescriptor& d){
  int nBytes= serialize(s,d.globalID);
  nBytes+= serialize(s,d.storageSetID);
  nBytes+= serialize(s,d.localID);
  return nBytes;
}

template <class Stream>
int deserialize(GlobalObjectDescriptor& d, Stream& s);
template <class Stream>
int deserialize(GlobalObjectDescriptor& d, Stream& s){
  int nBytes= deserialize(d.globalID,s);
  nBytes+= deserialize(d.storageSetID,s);
  nBytes+= deserialize(d.localID,s);
  return nBytes;
}

int serialSizeof(const GlobalObjectDescriptor& );
int serialSizeof(const GlobalObjectDescriptor& ){
  return sizeof(StorageSetID_t)+
    2*sizeof(ObjectID_t);
}

std::string PoomaCTTI(const GlobalObjectDescriptor& );
std::string PoomaCTTI(const GlobalObjectDescriptor& ){
  return std::string("GlobalObjectDescriptor");
}



//-----------------------------------------------------------------------------
class MultiStorageObjectSet{
 public:
  // Construct an object set with the given name containing
  // no storage sets a priori.
  MultiStorageObjectSet(const std::string& setName,
			ObjectSetOpenMode mode){
    // Default state is closed until opened successfully.
    openState_m= Closed;
  
    // Store the set name.
    setName_m= setName;
	
    // Mode cases
    switch(mode){
    case Create:
    
      // Call the create() function to perform the initialization.
      create();
      break;
    case Restore:
    
      // Call the restore() function to restore the 
      //   existing information from the dictionary file.
      restore();
      break;
    default:
    
      // Bad mode. Leave uninstantiated and issue a warning.
      pwarn<<"Warning: Unknown mode specified. "
	   <<"ObjectStorageSet "<<setName<<" not instantiated"
	   <<std::endl;
    
      // Then reset the name.
      setName_m="";
      break;
    }              
  }
	
  // Destructor.
  ~MultiStorageObjectSet(){
    // Close if open.
    if(openState_m== Open){
      close();
    }
    // Destroy the pointers.
    ObjectStorageSet* sset;
    for(int i=0;i<numStorageSets();i++){
      sset= storageSets_m[i];
      delete sset;
    }
    delete thisSet_m;
  }
	
  // Open a storage set as a component of the object set.
  // Use the given open mode to access.
  StorageSetID_t openStorageSet(const std::string& storageSetName,
				ObjectSetOpenMode mode){
    // Create a new instance of a storage set component.
    ObjectStorageSet* sset;
    sset= new ObjectStorageSet(storageSetName,mode);
    storageSets_m.push_back(sset);
    // Store the storage set name as a labeled object.
    thisSet_m->store(storageSetName,"ObjectSet::StorageSetName");
    return storageSets_m.size();
  }
	
  // Returns the name of the object set.
  inline std::string name() const{
    // Return the set name.
    return setName_m;
  }
  	
  // Return the number of storage sets in the object set.
  inline int numStorageSets() const{
    return storageSets_m.size();
  }
	
  // Return the number of objects in all component sets.
  inline int numObjects() const{
    return objectMap_m.size();
  }
	
  // Return the name of a component object set.
  inline std::string storageSetName(StorageSetID_t ssID)
    const{
    // Check for validity of storage set ID in proper
    // implementation.
    return storageSets_m[(ssID-1)]->name();
  }
	
  // Returns the type attribute of an object.
  std::string objectType(const ObjectID_t& gid){
    // Look up the global object reference by global ID.
    GlobalObjectRef ref= objectMap_m[gid];
    // Get a pointer to the component object set.
    ObjectStorageSet* sset= storageSets_m[(ref.storageSetID-1)];
    // Query the component object set with the local ID.
    return sset->objectType(ref.localID);
  }
  
  // Returns the label attribute of an object.
  std::string objectLabel(const ObjectID_t& gid){
    // Look up the global object reference by global ID.
    GlobalObjectRef ref= objectMap_m[gid];
    // Get a pointer to the component object set.
    ObjectStorageSet* sset= storageSets_m[(ref.storageSetID-1)];
    // Query the component object set with the local ID.
    return sset->objectLabel(ref.localID);  	
  }
  
  // Main store function.
  template <class T>
    ObjectID_t store(const T& t, StorageSetID_t ssID){
    
    // Obtain a new global object ID.
    ObjectID_t gid= newID();
    
    // Retrieve a pointer to the target storage set
    // IF the storage set ID is valid. <--- Add this assertion later.
    ObjectStorageSet* sset= storageSets_m[(ssID-1)];
    
    // Store and catch the local ID.
    ObjectID_t lid= sset->store(t);
    
    // Compose an object reference.
    GlobalObjectRef ref;
    ref.storageSetID= ssID;
    ref.localID= lid;
    
    // Place the entries in the maps.
    objectMap_m[gid]= ref;
    objectMapInv_m[ref]= gid;
   		
    // Compose an object descriptor.
    GlobalObjectDescriptor desc;
    desc.globalID= gid;
    desc.storageSetID= ssID;
    desc.localID= lid;
   		
    // Store the descriptor as an object in the primary object set.
    thisSet_m->store(desc,"ObjectSet::GlobalObjectDescriptor");
    
    // return the ID.
    return gid;
  }
  
  // Augmented store() function accepts a label attribute.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    ObjectID_t store(const T& t, StorageSetID_t ssID,
		     const std::string& objectLabel){

    // Obtain a new global object ID.
    ObjectID_t gid= newID();
    
    // Retrieve a pointer to the target storage set
    // IF the storage set ID is valid. <--- Add this assertion later.
    ObjectStorageSet* sset= storageSets_m[(ssID-1)];
    
    // Store and catch the local ID using the optional label attribute
    // storage option.
    ObjectID_t lid= sset->store(t,objectLabel);
    
    // Compose an object reference.
    GlobalObjectRef ref;
    ref.storageSetID= ssID;
    ref.localID= lid;
    
    // Place the entries in the maps.
    objectMap_m[gid]= ref;
    objectMapInv_m[ref]= gid;
    
    // Compose an object descriptor.
    GlobalObjectDescriptor desc;
    desc.globalID= gid;
    desc.storageSetID= ssID;
    desc.localID= lid;
   		
    // Store the descriptor as an object in the primary object set.
    thisSet_m->store(desc,"ObjectSet::GlobalObjectDescriptor");
    
    // return the ID.
    return gid;
  }
  // Return the ID of the FIRST object matching the given label.
  ObjectID_t findLabel(const std::string& objectLabel){
    // Sequence through the component sets. Report the global ID
    // of the first match encountered.
    int num= storageSets_m.size();
    ObjectID_t lid, gid;
    GlobalObjectRef ref;
    for(int i=0;i<num;i++){
      lid= storageSets_m[i]->findLabel(objectLabel);
      if(lid!=0){
	ref.storageSetID= (StorageSetID_t) (i+1);
	ref.localID= lid;
	gid= objectMapInv_m[ref];
	return gid;
      }
    }
    return 0;
  }
  
  // Assigns a label to an object that has already been stored.
  void objectLabel(ObjectID_t gid, const std::string& objectLabel){
    // If the global ID exists, tag the object in the component set.
    if(objectExists(gid)){
      GlobalObjectRef ref= objectMap_m[gid];
      ObjectStorageSet* sset= storageSets_m[(ref.storageSetID-1)];
      sset->objectLabel(ref.localID,objectLabel);
    }
    // Otherwise report the problem.
    else{
      pwarn<<"Warning: Object with global ID "<<gid
	   <<" does not exist."<<std::endl;
    }
  }

  
  // Returns the number of objects matching the given label attribute.
  int count(const std::string& objectLabel){
    // Sequence throught the component sets and count the number of
    // matches.
    int num= storageSets_m.size();
    int nmatches=0;
    for(int i=0;i<num;i++){
      nmatches+= storageSets_m[i]->count(objectLabel);
    }
    return nmatches;
  }
 		
  
  // Returns a set of global IDs of objects matching the given label.
  ObjectSubset selectLabel(const std::string& objectLabel){
    // Sequence throught the component sets and count the number of
    // matches.
    ObjectSubset s, scomp;
    // Make sure the subset is empty.
    s.clear();
    int num= storageSets_m.size();
    ObjectSubset::iterator siter;
    GlobalObjectRef ref;
    ObjectID_t lid, gid;
    StorageSetID_t ssid;
    for(int i=0;i<num;i++){
      // Make sure the component subset is empty.
      scomp.clear();
      // Select the component objects matching the label.
      scomp= storageSets_m[i]->selectLabel(objectLabel);
      // Iterate over the subset and convert to global
      // IDs.
      ssid= (StorageSetID_t) (i+1);
      for(siter= scomp.begin(); siter!= scomp.end(); siter++){
	lid= *siter;
	ref.storageSetID= ssid;
	ref.localID= lid;
	gid= objectMapInv_m[ref];
	s.insert(gid);
      }
    }
    return s;
  }
  
  // Main retrieval function instantiates an object given its ID. Returns
  // true or false with success or failure respectively.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    bool retrieve(T& t, ObjectID_t gid){
    
    // Check that the global id is valid.
    if(objectExists(gid)){
      
      // Look up the global object reference by global id.
      GlobalObjectRef ref= objectMap_m[gid];
      
      // Get a pointer to the component object set.
      ObjectStorageSet* sset= storageSets_m[(ref.storageSetID-1)];
  		
      // Invoke the retrieve function on the component set
      // Using the local ID.
      bool result= sset->retrieve(t,ref.localID);
  		
      // Return the result
      return result;

    }
    else{
      pwarn<<"Warning: Object not retrieved. No object with ID "
	   <<gid<<" exists."<<std::endl;
      return false;
    }
  }

  // Augmented retrieval function allows retrieval based on an object
  // label instead of ID.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    ObjectID_t retrieve(T& t, const std::string& objectLabel){
    
    // Find the FIRST global ID that matches the query.
    ObjectID_t gid = findLabel(objectLabel);
	
    // If valid retrieve the object, otherwise return a zero ID.
    if(gid!=0){
      retrieve(t,gid);
      return gid;
    }
    else{
      pwarn<<"Warning: Object not retrieved. No object labeled "
	   <<objectLabel<<" exists."<<std::endl;
      return 0;
    }
  }

  // Close the multi-file set.
  void close(){
    // Close all the component storage sets.
    ObjectStorageSet* sset;
    for(int i=0;i<numStorageSets();i++){
      sset= storageSets_m[i];
      sset->close();
    }
    thisSet_m->close();
    openState_m= Closed;
  }
	
  // Flush the multi-file set.
  void flush(){
    // Flush all the component storage sets.
    ObjectStorageSet* sset;
    for(int i=0;i<numStorageSets();i++){
      sset= storageSets_m[i];
      sset->flush();
    }
    thisSet_m->flush();
  }

  // Check to see in an object with the given global ID is in the set.
  bool objectExists(ObjectID_t gid) const{
    if(objectMap_m.count(gid)==0){
      return false;
    }
    else{
      return true;
    }
  }

 private:
  // Private data members:
	
  ObjectStorageSet* thisSet_m;

  // The container of object storage set references.
  std::vector<ObjectStorageSet*> storageSets_m;
	
  // The name of the object set.
  std::string setName_m;
  
  // The object map from global to local ID.
  std::map<const ObjectID_t, GlobalObjectRef> objectMap_m;

  // The inverse object map from local to global ID.
  std::map<const GlobalObjectRef, ObjectID_t> objectMapInv_m;
  
  // Indicates whether the object set is open or closed.
  ObjectSetOpenState openState_m;
  
  // Private member functions:
  
  void create(){
    // Create an object storage set for the multi-storage set
    // itself.
    thisSet_m = new ObjectStorageSet(setName_m, Create);
    openState_m= Open;
  }
  
  void restore(){
    // Restore the primary storage set.
    thisSet_m= new ObjectStorageSet(setName_m,Restore);
  	
    // Query the primary storage set for the names of
    // of the component storage sets.
    ObjectSubset subset=
      thisSet_m->selectLabel("ObjectSet::StorageSetName");
  	
    // Make sure the storage set vector is clear.
    storageSets_m.clear();
  	
    // Fetch each of the storage set names in turn and restore
    // the associates storage sets.
    ObjectSubset::iterator iter;
    ObjectID_t oid;
    std::string ssname;
    ObjectStorageSet* sset;
    for(iter=subset.begin(); iter!=subset.end(); iter++){
      oid=*iter;
      thisSet_m->retrieve(ssname,oid);
      sset= new ObjectStorageSet(ssname,Restore);
      storageSets_m.push_back(sset);
    }
  	
    // Make sure the object maps are empty.
    objectMap_m.clear();
    objectMapInv_m.clear();
  	
    // Fetch each of the global object descriptors and restore
    // the object maps.
    subset= thisSet_m->selectLabel("ObjectSet::GlobalObjectDescriptor");
    GlobalObjectDescriptor desc;
    GlobalObjectRef ref;
    for(iter=subset.begin(); iter!=subset.end(); iter++){
      oid=*iter;
      thisSet_m->retrieve(desc,oid);
      ref.storageSetID= desc.storageSetID;
      ref.localID= desc.localID;
      objectMap_m[desc.globalID]= ref;
      objectMapInv_m[ref]= desc.globalID;
    }
  	
    openState_m= Open;
  }
  
  // Generates new global object IDs on demand.
  ObjectID_t newID(){
    // Based on the size of the object dictionary.
    return objectMap_m.size() +1;
  }
  
};

#endif // MULTISTORAGE_OBJECT_SETS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MultiStorageObjectSets.h,v $   $Author: julianc $
// $Revision: 1.2 $   $Date: 2000/04/22 22:41:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

