// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Fully specialized serializer functions
//-----------------------------------------------------------------------------
// include files
//-----------------------------------------------------------------------------
#include <string>
extern "C"{
#include <string.h>
}
#include "Serializers.h"
#include "Utilities/PAssert.h"
//-----------------------------------------------------------------------------
// Functions to standardize positioning of the stream pointer for
// iostreams and char* buffers.
//-----------------------------------------------------------------------------
std::iostream& seekIn(std::iostream& s, int index){
  // Precondition: index must be greater than or equal to zero
  PInsist((index>=0),
	  "seekIn(): Invalid index");
  s.seekg(index);
  return s;
}
//-----------------------------------------------------------------------------
std::iostream& seekOut(std::iostream& s, int index){
  // Precondition: index must be greater than or equal to zero
  PInsist((index>=0),
	  "seekOut(): Invalid index");
  s.seekp(index);
  return s;
}
//-----------------------------------------------------------------------------
char*& seekIn(char*& s, int index){
  // Precondition: index must be greater than or equal to zero
  PInsist((index>=0),
	  "seekIn(): Invalid index");
  s=(char*)&s[index];
  return s;
}
//-----------------------------------------------------------------------------
char*& seekOut(char*& s, int index){
  // Precondition: index must be greater than or equal to zero
  PInsist((index>=0),
	  "seekOut(): Invalid index");
  s=(char*)&s[index];
  return s;
}

//-----------------------------------------------------------------------------
// Special functions for standard strings
//-----------------------------------------------------------------------------
// Return the serial size of a standard string instance in bytes.
// Standard strings are encoded as an integer number of (valid) characters
// followed by a null-terminated character string.
//-----------------------------------------------------------------------------
int serialSizeof(const std::string& t){
  int len= t.size();
  return sizeof(int) + len +1;
}

//-----------------------------------------------------------------------------
// Special functions to encode and insert standard strings in specific streams.
// Standard strings are encoded as an integer length followed by a
// null-terminated character string. These special functions are needed
// because of the specific procedures used in inserting and extracting strings
// from/to the various source/target streams.
//-----------------------------------------------------------------------------
int streamStringOut(std::iostream& s, const std::string& t){
  int len= t.size();
  int nBytes= serialize(s,len);
  nBytes+= serializeN(s,t.c_str(),len+1);
  return nBytes;
}
//-----------------------------------------------------------------------------
int streamStringIn(std::string& t, std::iostream& s){
  int len;
  int nb= deserialize(len,s);
  std::getline(s,t,'\0');
  return nb+ len +1;
}
//-----------------------------------------------------------------------------
int streamStringOut(char*& s, const std::string& t){
  int len= t.size();
  int nBytes= serialize(s,len);
  nBytes+= serializeN(s,t.c_str(),len+1);
  return nBytes;
}
//-----------------------------------------------------------------------------
int streamStringIn(std::string& t, char*& s){
  int len;
  int nb= deserialize(len,s);
  t= s;
  s+= len+1;
  return nb+ len+ 1;
}
//-----------------------------------------------------------------------------

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Serializers.cmpl.cpp,v $   $Author: julianc $
// $Revision: 1.9 $   $Date: 2000/04/22 22:33:10 $
// ----------------------------------------------------------------------
// ACL:rcsinfo







