// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------

#ifndef POOMA_IO_SERIALIZERS_H
#define POOMA_IO_SERIALIZERS_H

//////////////////////////////////////////////////////////////////////
//-----------------------------------------------------------------------------
// Overview:
//
// The serialize() functions convert an object or array of objects
// into a series of bytes and insert the bytes into either an I/O stream or
// a character buffer treated as a stream. The complementary deserialize()
// functions perform the reverse transformation. Each of these functions
// returns the number of bytes inserted or extracted from the stream.
// The pointer to the next position in the stream is also updated.
// Note this means that the pointer to char*, if a char* buffer is the
// source or target, is incremented.
//
// A function called serialSizeof() will return the expected number of bytes
// for a given type.
//
// These functions are global in scope. They depend for their implementation
// on several utility functions to standardize certain operations so as
// to remove interface differences from the iostream and char* target streams
// allowing a generic implementation of most functions independent of the
// stream type.
//
// In principle, a user need only write a specialized version of the essential
// serialize() and deserialize() functions for a new type in terms of existing
// types, making use of the existing serializers for those types.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Serializer Functions: Various templated and non-templated version of:
// serialSizeof(), serialize(), deserialize(), serializeN(), deserializeN(),
// Also the utility functions: seekIn(), seekOut(), streamIn(), streamOut(),
// streamStringIn(), streamStringOut().
// 
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include <iostream>
#include <string>
#include "Utilities/PAssert.h"
#include "Utilities/ElementProperties.h"
//-----------------------------------------------------------------------------
// Functions to standardize positioning of the stream pointer for both
// iostreams and char* buffers.
//-----------------------------------------------------------------------------
std::iostream& seekIn(std::iostream& s, int index);

std::iostream& seekOut(std::iostream& s, int index);

char*& seekIn(char*& s, int index);

char*& seekOut(char*& s, int index);

//-----------------------------------------------------------------------------
// Functions to standardize read/write operations on generic type instances
// for specific streams. Returns number of bytes processed.
//-----------------------------------------------------------------------------
template <class T>
int streamOut(std::iostream& s, const T& t);

template <class T>
int streamIn(T& t, std::iostream& s);

template <class T>
int streamOut(char*& s, const T& t);

template <class T>
int streamIn(T& t, char*& s);

template <class T>
int streamNOut(std::iostream& s, const T* t, int num);

template <class T>
int streamNIn(T* t, std::iostream& s, int num);

template <class T>
int streamNOut(char*& s, const T* t, int num);

template <class T>
int streamNIn(T* t, char*& s, int num);

//////////////////////////////////////////////////////////////////////
//-----------------------------------------------------------------------------
// The generic serializer functions
//-----------------------------------------------------------------------------
// Returns the serial size of a given object instance in bytes.
// Must be specialized for some new types.
//-----------------------------------------------------------------------------
template <class T>
int serialSizeof(const T& t);

//-----------------------------------------------------------------------------
// The essential generic serializers. These must be specialized for some
// new types.
//-----------------------------------------------------------------------------
template <class Stream, class T>
int serialize(Stream& s, const T& t);

template <class Stream, class T>
int deserialize(T& t, Stream& s);

//-----------------------------------------------------------------------------
// Expanded generic interface for any types supported by the essential
// serializer implementations.
//-----------------------------------------------------------------------------
// Serialize/deserialize to generic stream target
// using an optional seek first to the given index.
// Returns number of bytes read/written.

template <class Stream, class T>
int serialize(Stream& s, const T& t, int index);

template <class Stream, class T>
int deserialize(T& t, Stream& s, int index);

// Serialize/deserialize simple arrays to input/output streams
// given a pointer and number of elements num.

template <class Stream, class T>
int serializeN(Stream& s, const T* t, int num);

template <class Stream, class T>
int deserializeN(T* t, Stream& s, int num);

// Serialize/deserialize simple arrays to input/output streams
// given a pointer and number of elements using a seek first.

template <class Stream, class T>
int serializeN(Stream& s, const T* t, int num, int index);

template <class Stream, class T>
int deserializeN(T* t, Stream& s, int num, int index);

//////////////////////////////////////////////////////////////////////
//-----------------------------------------------------------------------------
// Specializations for C++ standard strings and special functions needed
// for their implementation. 
//-----------------------------------------------------------------------------
int serialSizeof(const std::string& t);

template <class Stream>
int serialize(Stream& s, const std::string& t);

template <class Stream>
int deserialize(std::string& t, Stream& s);

int streamStringOut(std::iostream& s, const std::string& t);

int streamStringIn(std::string& t, std::iostream& s);

int streamStringOut(char*& s, const std::string& t);

int streamStringIn(std::string& t, char*& s);

/*
//////////////////////////////////////////////////////////////////////
//-----------------------------------------------------------------------------
// Specializations of serializeN()/deserializeN() for simple arrays of
// known fixed types. 
//-----------------------------------------------------------------------------
// For int:
int serializeN(std::iostream& s, const int* t, int num);
int deserializeN(int* t, std::iostream& s, int num);

// For long:
int serializeN(std::iostream& s, const long* t, int num);
int deserializeN(long* t, std::iostream& s, int num);

// For float:
int serializeN(std::iostream& s, const float* t, int num);
int deserializeN(float* t, std::iostream& s, int num);

// For double:
int serializeN(std::iostream& s, const double* t, int num);
int deserializeN(double* t, std::iostream& s, int num);
*/
//////////////////////////////////////////////////////////////////////
// Include templated definitions
//////////////////////////////////////////////////////////////////////

#include "IO/Serializers.cpp"

#endif // POOMA_IO_SERIALIZERS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Serializers.h,v $   $Author: ambro $
// $Revision: 1.12 $   $Date: 2000/07/10 17:04:35 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

