// -*- C++ -*-
//-----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include <fstream>
#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Layout/Node.h"
#include "Layout/UniformGridLayout.h"
#include "Layout/GridLayout.h"
#include "Domain/IndirectionList.h"
#include "IO/LayoutIO.h"
#include "IO/DomainIO.h"

//-----------------------------------------------------------------------------
// Test of Pooma Layout serializers
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m);
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m){
  tstr.check(cond);
  if(cond){
    tstr.out() << "PASSED "<<m<<std::endl;
  }
  else{
    tstr.out() << "FAILED "<<m<<std::endl;
  }
}
/*
template <int Dim, class DT>
bool equalDomains(Domain<Dim,DT>& d1, Domain<Dim,DT>& d2);
template <int Dim, class DT>
bool equalDomains(Domain<Dim,DT>& d1, Domain<Dim,DT>& d2){
  int dim=Dim;
  for(int i=0;i<dim;i++){
    if((d1[i].first()!=d2[i].first())||
       (d1[i].last()!=d2[i].last()))
      return false;
  }
  return true;
}
*/
template <class Dom>
bool equalNodes(Node<Dom>& n1, Node<Dom>& n2);
template <class Dom>
bool equalNodes(Node<Dom>& n1, Node<Dom>& n2){
  Dom owned1, alloc1, owned2, alloc2;
  owned1= n1.domain();
  alloc1= n1.allocated();
  owned2= n2.domain();
  alloc2= n2.allocated();
  if(!(owned1==owned2))
    return false;
  if(!(alloc1==alloc2))
    return false;

  Node<Dom>::ID_t loc1, loc2, glob1, glob2;
  loc1= n1.localID();
  glob1= n1.globalID();
  loc2= n2.localID();
  glob2= n2.globalID();
  if(loc1!=loc2)
    return false;
  if(glob1!=glob2)
    return false;

  Node<Dom>::Context_t context1, context2;
  context1= n1.context();
  context2= n2.context();
  if(context1!=context2)
    return false;

  int affinity1, affinity2;
  affinity1= n1.affinity();
  affinity2= n2.affinity();
  if(affinity1!=affinity2)
    return false;

  return true;
}


template <class L>
bool equalLayouts(L& lhs, L& rhs);
template <class L>
bool equalLayouts(L& l1, L& l2){
  typedef typename L::LayoutData_t LayoutData_t;
  typedef typename L::Domain_t Domain_t;

  Domain_t dom1= l1.domain();
  Domain_t dom2= l2.domain();
  if(!(dom1==dom2))
    return false;

  if(!(l1.blocks()==l2.blocks()))
    return false;

  typename LayoutData_t::GuardLayers_t g1,g2;

  g1= l1.internalGuards();
  g2= l2.internalGuards();
  if(!(g1==g2))
    return false;

  g1= l1.externalGuards();
  g2= l2.externalGuards();
  if(!(g1==g2))
    return false;

  int s1,s2;
  typedef typename L::List_t List_t;
  List_t lst1,lst2;
  s1= lst1.size();
  s2= lst2.size();
  if(s1!=s2)
    return false;

  Node<Domain_t> *node1, *node2;
  typename List_t::iterator iter1= lst1.begin(),
    iter2= lst2.begin();
  for(int i=0;i<s1;i++){
    node1= *iter1;
    node2= *iter2;
    if(!(equalNodes(*node1,*node2)))
      return false;
    ++iter1;
    ++iter2;
  }
  return true;
}

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  Pooma::Tester tester(argc, argv);

  tester.out() <<"Test Layout serializers:"<< std::endl;

  // Open a file for the test.
  std::fstream datafile("LayoutSerializerTest.dat",
                        std::ios::out | std::ios::binary);

  int ksize;
  // Create Interval domains.
  Interval<2> alloc, owned;
  alloc[0]= Interval<1>(0,7);
  alloc[1]= Interval<1>(0,3);
  owned[0]= Interval<1>(1,6);
  owned[1]= Interval<1>(1,2);

  // Create a node object based on the above domains.
  Node<Interval<2>,Interval<2> >
    mynode(owned, alloc, 1, 2, 3);

  // Serialize the node.
  ksize=serialize(datafile,mynode);
  itemCheck(tester,(ksize==serialSizeof(mynode)),
	    "Size computation on Node");

  // Create a uniform grid layout.
  Interval<2> dom;
  dom[0]= Interval<1>(0,31);
  dom[1]= Interval<1>(0,63);
  Loc<2> grid(2,4);
  UniformGridLayout<2>::GuardLayers_t guards(1);
  UniformGridLayout<2> layout(dom,grid,guards,ReplicatedTag());

  // Serialize
  ksize=serialize(datafile,layout);
  itemCheck(tester,(ksize==serialSizeof(layout)),
	    "Size computation on UniformGridLayout");

  // Create a grid layout.
  // Use the same guard layers as before.
  IndirectionList<int> igx(4), igy(4);
  int gx[4]= {0,4,16,32};
  int gy[4]= {0,8,32,64};
  for(int i=0;i<4;i++){
      igx(i)= gx[i];
  }
  for(int i=0;i<4;i++){
      igy(i)= gy[i];
  }
  Grid<2> lgrid(igx,igy);
  GridLayout<2> glayout(lgrid,guards,ReplicatedTag());

  // Serialize
  ksize=serialize(datafile,glayout);
  itemCheck(tester,(ksize==serialSizeof(glayout)),
	    "Size computation on GridLayout");


  // Close the file.
  datafile.close();

  // Reopen as another file object.
  std::fstream dfile("LayoutSerializerTest.dat",
                     std::ios::in | std::ios::binary);

  // Create anoter node object to instantiate.
  Node<Interval<2>,Interval<2> > mynode2;

  // Deserialize the object.
  deserialize(mynode2, dfile);

  // Check the results.
  bool checkResult;
  //  std::string typeString= PoomaCTTI(mynode2);
  tester.out()<<"Node object retrieved"<<std::endl;
  //  tester.out()<<"Type identifier: "<<typeString<<std::endl;
  checkResult= equalNodes(mynode,mynode2);
  itemCheck(tester,checkResult,"Check on stored Node");

  // Create a uniform grid layout target.
  Interval<2> dom2;
  Loc<2> grid2;
  UniformGridLayout<2>::GuardLayers_t iguards2, eguards2;
  UniformGridLayout<2> layout2;

  // Deserialize
  deserialize(layout2, dfile);
  checkResult= equalLayouts(layout,layout2);
  itemCheck(tester,checkResult,"Check on stored UniformGridLayout");

  // Print out the results.
  tester.out()<<"UniformGridLayout object retrieved"<<std::endl;
  //  typeString= PoomaCTTI(layout2);
  //  tester.out()<<"Type identifier: "<<typeString<<std::endl;
  tester.out()<<layout2<<std::endl;

  // Retrieve the grid layout object.
  GridLayout<2> glayout2;
  deserialize(glayout2,dfile);
  checkResult= equalLayouts(glayout,glayout2);
  itemCheck(tester,checkResult,"Check on stored GridLayout");

  // Print out the results.
  tester.out()<<"GridLayout object retrieved"<<std::endl;
  //  typeString= PoomaCTTI(glayout2);
  //  tester.out()<<"Type identifier: "<<typeString<<std::endl;
  tester.out()<<glayout2<<std::endl;

  dfile.close();

  int retval = tester.results("Test of Layout serializers");
  Pooma::finalize();
  return retval;
}

