//-----------------------------------------------------------------------------
// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include <fstream>
#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Pooma/Arrays.h"
//#include "Engine/BrickEngine.h"
//#include "Engine/CompressibleBrick.h"
//#include "Engine/MultiPatchEngine.h"
#include "IO/EngineIO.h"

//-----------------------------------------------------------------------------
// Test of Pooma Layout serializers
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m);
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m){
  tstr.check(cond);
  if(cond){
    tstr.out() << "PASSED "<<m<<std::endl;
  }
  else{
    tstr.out() << "FAILED "<<m<<std::endl;
  }
}

template <class Dom>
bool equalNodes(Node<Dom>& n1, Node<Dom>& n2);
template <class Dom>
bool equalNodes(Node<Dom>& n1, Node<Dom>& n2){
  Dom owned1, alloc1, owned2, alloc2;
  owned1= n1.domain();
  alloc1= n1.allocated();
  owned2= n2.domain();
  alloc2= n2.allocated();
  if(!(owned1==owned2))
    return false;
  if(!(alloc1==alloc2))
    return false;

  Node<Dom>::ID_t loc1, loc2, glob1, glob2;
  loc1= n1.localID();
  glob1= n1.globalID();
  loc2= n2.localID();
  glob2= n2.globalID();
  if(loc1!=loc2)
    return false;
  if(glob1!=glob2)
    return false;

  Node<Dom>::Context_t context1, context2;
  context1= n1.context();
  context2= n2.context();
  if(context1!=context2)
    return false;

  int affinity1, affinity2;
  affinity1= n1.affinity();
  affinity2= n2.affinity();
  if(affinity1!=affinity2)
    return false;

  return true;
}

template <class L>
bool equalLayouts(L& lhs, L& rhs);
template <class L>
bool equalLayouts(L& l1, L& l2){
  typedef typename L::LayoutData_t LayoutData_t;
  typedef typename L::Domain_t Domain_t;

  Domain_t dom1= l1.domain();
  Domain_t dom2= l2.domain();
  if(!(dom1==dom2))
    return false;

  if(!(l1.blocks()==l2.blocks()))
    return false;

  typename LayoutData_t::GuardLayers_t g1,g2;

  g1= l1.internalGuards();
  g2= l2.internalGuards();
  if(!(g1==g2))
    return false;

  g1= l1.externalGuards();
  g2= l2.externalGuards();
  if(!(g1==g2))
    return false;

  int s1,s2;
  typedef typename L::List_t List_t;
  List_t lst1,lst2;
  s1= lst1.size();
  s2= lst2.size();
  if(s1!=s2)
    return false;

  Node<Domain_t> *node1, *node2;
  typename List_t::iterator iter1= lst1.begin(),
    iter2= lst2.begin();
  for(int i=0;i<s1;i++){
    node1= *iter1;
    node2= *iter2;
    if(!(equalNodes(*node1,*node2)))
      return false;
    ++iter1;
    ++iter2;
  }
  return true;
}

template <class T, class EngineTag>
bool equalEngines(Engine<1,T,EngineTag>& e1,
		  Engine<1,T,EngineTag>& e2);
template <class T, class EngineTag>
bool equalEngines(Engine<1,T,EngineTag>& e1,
		  Engine<1,T,EngineTag>& e2){
  typename Engine<1,T,EngineTag>::Domain_t dom1, dom2;
  dom1= e1.domain();
  dom2= e2.domain();
  if(!(dom1==dom2))
    return false;
  for(int i=dom1[0].first();
      i<dom1[0].last(); i++){
    if(e1(i)!=e2(i))
      return false;
  }
  return true;
}

template <class T, class EngineTag>
bool equalEngines(Engine<2,T,EngineTag>& e1,
		  Engine<2,T,EngineTag>& e2);
template <class T, class EngineTag>
bool equalEngines(Engine<2,T,EngineTag>& e1,
		  Engine<2,T,EngineTag>& e2){
  typename Engine<2,T,EngineTag>::Domain_t dom1, dom2;
  dom1= e1.domain();
  dom2= e2.domain();
  if(!(dom1==dom2))
    return false;
  for(int j=dom1[1].first();
      j<dom1[1].last(); j++){
    for(int i=dom1[0].first();
	i<dom1[0].last(); i++){
      if(e1(i,j)!=e2(i,j))
	return false;
    }
  }
  return true;
}

template <class T, class EngineTag>
bool equalEngines(Engine<3,T,EngineTag>& e1,
		  Engine<3,T,EngineTag>& e2);
template <class T, class EngineTag>
bool equalEngines(Engine<3,T,EngineTag>& e1,
		  Engine<3,T,EngineTag>& e2){
  typename Engine<3,T,EngineTag>::Domain_t dom1, dom2;
  dom1= e1.domain();
  dom2= e2.domain();
  if(!(dom1==dom2))
    return false;
  for(int k=dom1[2].first();
      k<dom1[2].last(); k++){
    for(int j=dom1[1].first();
	j<dom1[1].last(); j++){
      for(int i=dom1[0].first();
	  i<dom1[0].last(); i++){
	if(e1(i,j,k)!=e2(i,j,k))
	  return false;
      }
    }
  }
  return true;
}

template <int Dim, class T, class LayoutTag, class PatchTag>
bool equalMPEngines(Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >& mpe1,
		    Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >& mpe2);
template <int Dim, class T, class LayoutTag, class PatchTag>
bool equalMPEngines(Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >& mpe1,
		    Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> >& mpe2){

  typedef Engine<Dim,T,MultiPatch<LayoutTag,PatchTag> > Engine_t;
  typedef typename Engine_t::Layout_t Layout_t;
  typedef typename Layout_t::List_t List_t;
  typedef typename Layout_t::Value_t Node_t;
  typedef typename Engine_t::PatchID_t ID_t;
  typedef typename Engine_t::PatchEngine_t PatchEngine_t;

  Layout_t l1,l2;
  l1= mpe1.layout();
  l2= mpe2.layout();
  if(!equalLayouts(l1,l2))
    return false;
  
  bool result= true;
  List_t list1,list2;
  list1= l1.nodeListLocal();
  list2= l2.nodeListLocal();
  typename List_t::iterator iter1,iter2;
  Node_t *n1, *n2;
  ID_t id1, id2;
  PatchEngine_t p1, p2;
  iter1= list1.begin();
  iter2= list2.begin();
  int np= list1.size();
  for(int i=0;i<np;i++){
    n1= *iter1;
    n2= *iter2;
    id1= n1->localID();
    id2= n2->localID();
    p1= mpe1.localPatch(id1);
    p2= mpe2.localPatch(id2);
    result= result&& equalEngines(p1,p2);
    ++iter1; ++iter2;
  }
  return result;
}

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  Pooma::Tester tester(argc, argv);
  bool checkResult;

  tester.out() <<"Test MultiPatch engine serializers:"<< std::endl;

  // Open a file for the test.
  std::fstream datafile("MPEngineSerializerTest.dat",
                        std::ios::out | std::ios::binary);

  int ksize;
  // Create a uniform grid layout.
  Interval<2> mpdom;
  mpdom[0]= Interval<1>(0,31);
  mpdom[1]= Interval<1>(0,63);
  Loc<2> grid(2,4);
  UniformGridLayout<2>::GuardLayers_t guards(1);
  UniformGridLayout<2> layout(mpdom,grid,guards,ReplicatedTag());

  // Create a multipatch engine.
  Engine<2,int,MultiPatch<UniformTag,Brick> > mpe(layout);

  int i,j;
  // Assign values.
  for(j=0;j<64;j++){
    for(i=0;i<32;i++){
      mpe(i,j)= ((double)j)+
	((double)i)*0.01;
    }
  }

  // Serialize
  ksize=serialize(datafile,mpe);
  itemCheck(tester,(ksize==serialSizeof(mpe)),
	    "Size computation on MultiPatch engine");

  // Close the file.
  datafile.close();

  // Reopen as another file object.
  std::fstream dfile("MPEngineSerializerTest.dat",
                     std::ios::in | std::ios::binary);

  // Create a MultiPatch engine target
  Engine<2,int,MultiPatch<UniformTag,Brick> > mpe2;

  // Deserialize
  deserialize(mpe2, dfile);
  checkResult= equalMPEngines(mpe,mpe2);
  itemCheck(tester,checkResult,"Check on stored MultiPatch engine");

  // Print out the results.
  tester.out()<<"MultiPatch engine object retrieved"<<std::endl;
  std::string typeString= PoomaCTTI(mpe2);
  tester.out()<<"Type identifier: "<<typeString<<std::endl;

  dfile.close();

  int retval = tester.results("Test of MultiPatch engine serializers");
  Pooma::finalize();
  return retval;
}











