// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
//
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without
// charge, provided that this Notice and any statement of authorship are
// reproduced on all copies.  Neither the Government nor the University
// makes any warranty, express or implied, or assumes any liability or
// responsibility for the use of this SOFTWARE.
//
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
//
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Test of DynamicLayout
//-----------------------------------------------------------------------------

// Include files

#include "Layout/DynamicLayout.h"
#include "Partition/GridPartition.h"
#include "Utilities/Tester.h"
#include "Pooma/Pooma.h"
#include "Domain/Range.h"
#include "Domain/Grid.h"
#if POOMA_NO_STRINGSTREAM
	#include <strstream>
#else
        #include <sstream>
#endif

//#define BARRIER

#ifndef BARRIER
#if POOMA_CHEETAH
# define BARRIER Pooma::controller()->barrier()
#else
# define BARRIER
#endif
#endif

void printLayout(Inform &out, const std::string msg, const DynamicLayout &l);

void printLayout(Inform &out, const std::string msg, const DynamicLayout &l)
{
  DynamicLayout::const_iterator pos;
  const int numContexts = Pooma::contexts();
  const int myContext   = Pooma::context();

  BARRIER;

  out.setOutputContext(0);
  out << msg << std::endl;
  out.setOutputContext(-1);

  BARRIER;

// This looks like a silly loop, but with the BARRIER it causes the contexts
// to print things in order.

  for (int i = 0; i < numContexts; ++i)
    {
      if (myContext == i)
	for (pos = l.beginLocal(); pos != l.endLocal(); ++pos)
	  out << pos->domain() << std::endl;
      BARRIER;
    }

  out.setOutputContext(0);
  out << "Total Domain = " << l.domain() << std::endl;

  BARRIER;
}

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  const int numContexts = Pooma::contexts();
  const int myContext   = Pooma::context();

  tester.out() << "Testing dynamic ops in DynamicLayout class . . .\n";
  tester.out() << "Running with " << numContexts << " contexts." << std::endl;

  Interval<1> domain;
  int numBlocks = numContexts * 5;

  tester.out() << "Creating DynamicLayout with domain " << domain
               << " and " << numBlocks << " blocks." << std::endl;
               
  BARRIER;

  Loc<1> foo(numBlocks);
  GridPartition<1> gp(foo);
  DistributedMapper<1> cmap(gp);
  DynamicLayout layout(domain, gp, cmap);

  DynamicLayout::iterator pos;

  std::string msg("Here are the patch domains for the initial partitioning:");
  printLayout(tester.out(), msg, layout);

  // Creates 35 elements in the first patch of each subdomain.

  for (int i = 0; i < numContexts; ++i)
    {
      if (i == myContext)
        {
          layout.create(35,0);
	  layout.create(10,1);
#if POOMA_NO_STRINGSTREAM
	  std::strstream s;
#else
          std::ostringstream s;
#endif
          s << "Here are the patch domains after adding elements\n"
            << "to the first two patches on context " << i 
	    << ", before syncing.";
          printLayout(tester.out(), s.str(), layout);
	  BARRIER;

          layout.sync();
#if POOMA_NO_STRINGSTREAM
	  std::strstream ss;
#else
          std::ostringstream ss;
#endif       
	  
          ss << "Here are the patch domains on context " << i 
	    << ", after syncing.";
          printLayout(tester.out(), ss.str(), layout);
          BARRIER;
        }
    }

  BARRIER;

  int ret = tester.results("DynamicLayout Test2");
  Pooma::finalize();

  return ret;
}

