// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Test of global ID database
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Layout/UniformGridLayout.h"
#include "Layout/GlobalIDDataBase.h"
#include "Utilities/Tester.h"

#include <iostream>

int main(int argc, char *argv[])
{
  // Initialize POOMA and output stream, using Tester class
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  tester.out() << argv[0] << ": Tests of global ID database." << std::endl;
  tester.out() << "------------------------------------------------" << std::endl;

  int size = 120;

  Interval<1> domain(size);

  UniformGridPartition<1> partition1(Loc<1>(10));
  UniformGridLayout<1> layout1(domain,partition1, ReplicatedTag());

  UniformGridPartition<1> partition2(Loc<1>(6));
  UniformGridLayout<1> layout2(domain,partition2, ReplicatedTag());

  std::vector<INode<1> > inodes;
  GlobalIDDataBase gidStore;

  UniformGridLayout<1>::const_iterator p = layout1.beginGlobal();
  while (p != layout1.endGlobal())
  {
    inodes.push_back(INode<1>(*p, layout1.ID(), &gidStore));
    ++p;
  }
  int i;
  int ni = inodes.size();
  for (i = 0; i < ni; ++i)
  {
    layout2.touches(inodes[i].domain(), std::back_inserter(inodes),
		    TouchesConstructINode<1>(layout2.ID(),inodes[i].key(),
					     &gidStore)
		    );
  }
  inodes.erase(inodes.begin(), inodes.begin() + ni);

  int gid1,gid2;
  int gidStore1,gidStore2;
  for (i = 0; i < inodes.size(); ++i)
  {
    gid1 = layout1.globalID(inodes[i].domain().firsts());
    gid2 = layout2.globalID(inodes[i].domain().firsts());
    gidStore1 = inodes[i].globalID(layout1.ID());
    gidStore2 = inodes[i].globalID(layout2.ID());

    tester.check(gid1 == gidStore1);
    tester.check(gid2 == gidStore2);

    tester.out() << "domain " << inodes[i].domain()
		 << ", key " << inodes[i].key()
		 << ", gid #1 - (" << gid1 << " == " << gidStore1 << ")"
		 << ", gid #2 - (" << gid2 << " == " << gidStore2 << ")"
		 << std::endl;
  }

  gidStore.print(tester.out());

  tester.out() << "------------------------------------------------"
	       << std::endl;

  int retval = tester.results("giddatabaseTest");
  Pooma::finalize();
  return retval;  
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: giddatabaseTest.cpp,v $   $Author: luchini $
// $Revision: 1.6 $   $Date: 2000/06/01 18:59:20 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
