// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Classes:
// CreateLeaf<Expr>
// MakeFieldReturn<T>
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDCREATELEAF_H
#define POOMA_NEWFIELD_FIELDCREATELEAF_H

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

struct FarLeftTag;

template<class G, class T, class E> class Field;


//-----------------------------------------------------------------------------
// These are the external traits classes that are used to build trees.
// CreateLeaf is used to convert arbitrary classes into expression objects.
// MakeFieldReturn is used to combine expressions together with operators.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Traits classes
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// CreateLeaf is an external functor class used to convert objects into the
// leaves of the expression tree.
//
// CreateLeaf<T> converts objects of type T to leaf objects and requires
// the following interface:
//
// typedef ... Leaf_t;        // The leaf object
// typedef ... Return_t;      // Type returned by make()
// Return_t make(const T&);   // make the leaf object from the T object
//
// Return_t should equivalent to Leaf_t. (Leaf_t needs to be able
// be constructed with a Return_t.)  We avoid making extra copies by building
// expression trees from references, so define Return_t to be a const ref to
// an Leaf_t.  (Returning by value would be bad, since we would create
// a temporary copy that won't survive until the whole expression is put
// together.)
//
// CreateLeaf is used to construct expression trees.  It should also be
// used when performing operations on the expression tree, such as forEach,
// in order to extract the expression.  For example:
// template<class G, class T, class E>
// void func(const Field<G,T,E>& f)
// {
//   forEach(CreateLeaf<Field<G,T,E >::make(f),...,...);
// }
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Fields are leaf objects, we just pass them through unless they have
// Expression engines. Then, we remove the expresion from the field and form
// leaves with tree-nodes. 

template<class GeometryTag, class T, class Expr>
struct CreateLeaf<Field<GeometryTag, T, ExpressionTag<Expr> > >
{
  typedef Field<GeometryTag, T, ExpressionTag<Expr> > Input_t;
  typedef Expr Leaf_t;
  typedef const Leaf_t &Return_t;
  inline static
  Return_t make(const Input_t &f)
    {
      return f.engine().expression();
    }
};
  
template<class GeometryTag, class T, class EngineTag>
struct CreateLeaf<Field<GeometryTag, T, EngineTag> >
{
  typedef Field<GeometryTag, T, EngineTag> Input_t;
  typedef Reference<Input_t> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &f)
    {
      return Leaf_t(f);
    }
};

//-----------------------------------------------------------------------------
// Special case for Scalar<Field> returns ErrorType to avoid
// hairy type computations.

template<class GeometryTag, class T, class EngineTag>
struct CreateLeaf<Scalar<Field<GeometryTag, T, EngineTag> > >
{
  typedef Scalar<Field<GeometryTag, T, EngineTag> > Input_t;
  typedef Scalar<ErrorType> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &)
    {
      return ErrorType();
    }
};


#endif // POOMA_NEWFIELD_FIELDCREATELEAF_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldCreateLeaf.h,v $   $Author: sa_smith $
// $Revision: 1.4 $   $Date: 2000/08/08 17:31:33 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
