// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   NoGeometry<Dim>
//   FieldEngine<NoGeometry, T, EngineTag>
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDENGINE_NOGEOMETRY_H
#define POOMA_NEWFIELD_FIELDENGINE_NOGEOMETRY_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// FieldEngine<NoGeometry, T, EngineTag>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Array/Array.h"
#include "Domain/Interval.h"
#include "Engine/ConstantFunctionEngine.h"
#include "Engine/IndexFunctionEngine.h"
#include "Layout/INode.h"
#include "NewField/Updater/UpdaterList.h"
#include "Tiny/Vector.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class Components> class ComponentWrapper;
template<int Dim, class T, class EngineTag> class Engine;
template<class GeometryTag, class T, class EngineTag> class FieldEngine;


//-----------------------------------------------------------------------------
// NoGeometry tag class.
//-----------------------------------------------------------------------------

template<int Dim>
struct NoGeometry;


//-----------------------------------------------------------------------------
// FieldEngine, specialized for no geometry. This is really just a field
// wrapper of a single engine.
//-----------------------------------------------------------------------------

template<int Dim, class T, class EngineTag>
class FieldEngine<NoGeometry<Dim>, T, EngineTag>
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef FieldEngine<NoGeometry<Dim>, T, EngineTag> This_t;
    
  // Our base class.
  
  typedef FieldEngineBase<Dim, T, EngineTag> Base_t;

  // The geometry tag.
  
  typedef NoGeometry<Dim> GeometryTag_t;
  
  // The engine tag.
  
  typedef EngineTag EngineTag_t;
    
  // The domain type.
  
  typedef typename Base_t::Domain_t Domain_t;
  
  // The number of indices required to select a point in this mesh.
  
  enum { dimensions = Dim };
  
  // Our engine type.
  
  typedef Engine<Dim, T, EngineTag> Engine_t;
  
  // Our layout type.
  
  typedef typename Engine_t::Layout_t Layout_t;
  
  // Our guard layers type.
  
  typedef GuardLayers<Dim> GuardLayers_t;


  //---------------------------------------------------------------------------
  // Constructors.
  //
  // There is no sub-field constructor because this field-engine can't have
  // subfields.

  // Copy constructor.
  
  FieldEngine(const This_t &model)
  : engine_m(model.engine_m),
    guards_m(model.guards_m),
    updaters_m(model.updaters_m)
    { }

  // Domain view constructor, including INodes and non-interval
  // views. 

  template<class G2, class T2, class ET2, class Domain>  
  FieldEngine(const FieldEngine<G2, T2, ET2> &model, 
    const Domain &d)
  : engine_m(NewEngineEngine<typename 
      FieldEngine<G2, T2, ET2>::Engine_t, Domain>::apply(model.engine(), d), 
             NewEngineDomain<typename
      FieldEngine<G2, T2, ET2>::Engine_t, Domain>::apply(model.engine(), d)),
    updaters_m(model.updaters())
    { }
    
  // Sub-field constructor. Needs to exist, but shouldn't be called since we
  // have no subfields.
  
  template<class G2, class T2, class ET2>  
  FieldEngine(const FieldEngine<G2, T2, ET2> &model, const int &)
  : engine_m(model.engine())
    {
      PError("Can't create a sub-field of a FieldEngine<NoGeometry,...>.");
    }

  // Patch constructor.    

  template<class ET2>
  FieldEngine(const FieldEngine<GeometryTag_t, T, ET2> &model, 
    const EnginePatch &p)
  : engine_m(engineFunctor(model.engine(), p)),
    updaters_m(model.updaters())
    { }
    
  // ComponentView constructor. The geometry for the model should be the
  // same since a component view simply mutates the elements.

  template<class T2, class ET2, class Components>  
  FieldEngine(const FieldEngine<GeometryTag_t, T2, ET2> &model, 
    const ComponentWrapper<Components> &c)
  : engine_m(model.engine(), c.components()),
    updaters_m(model.updaters())
    { }


  //---------------------------------------------------------------------------
  // Copy assignment operator (disabled).
  
  This_t &operator=(const This_t &rhs);


  //---------------------------------------------------------------------------
  // Empty destructor is fine.
  
  ~FieldEngine() { }
      

  //---------------------------------------------------------------------------
  // Accessors and modifiers.
    
  inline int numSubFields() const
    {
      return 0;
    }

  const Loc<Dim> offsets() const
    {
      // Always vert-centered...
      
      return Loc<Dim>(1); 
    }
    
  Engine_t &engine()
    {
      return engine_m;
    }
    
  const Engine_t &engine() const
    {
      return engine_m;
    }

  const UpdaterList &updaters() const
    {
      return updaters_m;
    }

  UpdaterList &updaters()
    {
      return updaters_m;
    }
    
  const GuardLayers_t &guardLayers() const
    {
      return guards_m;
    }


  //---------------------------------------------------------------------------
  // Domain accessor functions.
  //
  // This field-engine always has vert-centering and the total domain is
  // given by the engine.
        
  inline const Domain_t &physicalCellDomain() const
    {
      return shrink(physicalDomain(), 1);
    }
        
  inline Domain_t totalCellDomain() const
    {
      return shrink(engine_m.domain(), 1);
    }

  Domain_t physicalDomain() const
    {
      return shrink(engine_m.domain(), guards_m);
    }

  Domain_t totalDomain() const
    {
      return engine_m.domain();
    }

  
  //---------------------------------------------------------------------------
  // Make a distinct copy of this fieldEngineBase.   
 
  template<class Subject>
  void makeOwnCopy(const Subject &s)
    {
      // Deepen the copies of the engine & updaters list.
      
      engine().makeOwnCopy();
      updaters().makeOwnCopy(s);
    }
  
private:

  Engine_t engine_m;
  GuardLayers_t guards_m;
  UpdaterList updaters_m;
};

#endif // POOMA_NEWFIELD_FIELDENGINE_NOGEOMETRY_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldEngine.NoGeometry.h,v $   $Author: swhaney $
// $Revision: 1.1 $   $Date: 2000/07/24 23:36:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

