// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Class:
//   ConstantFaceBC
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_UPDATER_CONSTANTFACEBC_H
#define POOMA_NEWFIELD_UPDATER_CONSTANTFACEBC_H

//-----------------------------------------------------------------------------
// Overview: 
// ConstantFaceBC: BCondCategory class setting all guard layers beyond a
//                 specified (logically) rectilinear mesh face to a constant 
//                 value.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/NoInit.h"
#include "Utilities/PAssert.h"

#include "NewField/Updater/Updater.h"
#include "NewField/Updater/UpdaterBase.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag> class Field;


//-----------------------------------------------------------------------------
// Full Description:
// 
// ConstantFaceBC is a specialization of UpdaterCategory.
// 
// It represents a Dirichlet boundary condition on a logically rectilinear
// domain where the value on that face is a constant. The setConstant()
// function provides a means to have a time-dependent BC.  A constructor switch
// allows the BC to enforce that the mesh-boundary value is zero; this affects
// only vertex-centered Field values/components because the boundary is defined
// to be the last vertex.
// 
// The T template parameter is the type of the constant value.
//-----------------------------------------------------------------------------

template <class T>
class ConstantFaceBC : public UpdaterCategory<ConstantFaceBC<T> > {
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  ConstantFaceBC(int face, const T &constant, 
    bool enforceConstantBoundary = false) 
  : face_m(face), 
    constant_m(constant), 
    enforceConstantBoundary_m(enforceConstantBoundary)  
    { }
    
  ConstantFaceBC(const ConstantFaceBC &model) 
  : face_m(model.face_m), 
    constant_m(model.constant_m), 
    enforceConstantBoundary_m(model.enforceConstantBoundary_m)
    { }


  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ConstantFaceBC &operator=(const ConstantFaceBC &rhs)
  {
    face_m = rhs.face_m;
    constant_m = rhs.constant_m;
    enforceConstantBoundary_m = rhs.enforceConstantBoundary_m;

    return *this;
  }


  //---------------------------------------------------------------------------
  // Accessors/modifiers for data members.
  
  int face() const { return face_m; }
  T constant() const { return constant_m; }
  bool enforceConstantBoundary() const { return enforceConstantBoundary_m; }

  // User may want to change the constant's value, e.g., for time-dependence.

  void setConstant(T newConstant) { constant_m = newConstant; }
  
  
private:

  int face_m;
  T constant_m;
  bool enforceConstantBoundary_m;
};


// ----------------------------------------------------------------------------
// An actual boundary condition class for ConstantFaceBC
// ----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
class Updater<Field<GeometryTag, T, EngineTag>, ConstantFaceBC<T> > :
  public UpdaterBase<Field<GeometryTag, T, EngineTag>, ConstantFaceBC<T> >
{
public:

  typedef Updater<Field<GeometryTag, T, EngineTag>, ConstantFaceBC<T> > 
    This_t;
  typedef UpdaterBase<Field<GeometryTag, T, EngineTag>, ConstantFaceBC<T> > 
    Base_t;
  typedef Field<GeometryTag, T, EngineTag> Subject_t;
  
  
  //---------------------------------------------------------------------------
  // Constructors.

  Updater(const Subject_t &s, const ConstantFaceBC<T> &bc)
  : Base_t(s, bc),
    domain_m(Pooma::NoInit())
    {
      // This only makes sense if the subject has no sub-fields.
       
      PAssert(subject().numSubFields() == 0);
      
      // Set up the domain.
      
      domain_m = subject().totalDomain();
      
      // Get the direction.

      int d = bc.face() / 2;

      // The other directions span the subject's total domain. 
      // Therefore, we just chop out the guard layers, taking care to 
      // handle the case where we are enforcing a zero boundary 
      // (appropriate only for vert-centering).
      
      int adjust;
      if (bc.enforceConstantBoundary() && 
        subject().fieldEngine().offsets()[d] == 1) 
        adjust = 0;
      else
        adjust = 1;
  
      // Select the high or low face.

      if (bc.face() & 1) 
        {
          // High face.
 
          // Get the number of guard layers in the upper direction.

          int nGuards = subject().fieldEngine().guardLayers().upper(d);
          
          // Adjust the domain.
                   
          domain_m[d] = Interval<1>(domain_m[d].max() - nGuards + adjust, 
			                        domain_m[d].max());
        } 
      else 
        {
          // Low face.
        
          // Get the number of guard layers in the lower direction.

          int nGuards = subject().fieldEngine().guardLayers().lower(d);
          
          // Adjust the domain.
                   	
          domain_m[d] = Interval<1>(domain_m[d].min(), 
				                    domain_m[d].min() + nGuards - adjust);
        }
    }


  //---------------------------------------------------------------------------
  // Destructor.

  ~Updater() { }


  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void update()
    {
      subject()(domain_m) = category().constant();
    }
    

private:

  typename Subject_t::Domain_t domain_m;
};


//-----------------------------------------------------------------------------
// AllConstantFaceBC is a functor class that sets all of a Field's BCs to be 
// ConstantFaceBC.
//-----------------------------------------------------------------------------

template<class T>
class AllConstantFaceBC
{
public:

  AllConstantFaceBC(const T &constant, bool enforceConstantBoundary = false)
  : const_m(constant), ecb_m(enforceConstantBoundary) { }
  
  template<class GeometryTag, class EngineTag>
  void operator()(Field<GeometryTag, T, EngineTag> &f) const
    {
      typedef Field<GeometryTag, T, EngineTag> Subject_t;
      
      for (int i = 0; i < 2 * Subject_t::dimensions; i++)
        f.addUpdater(ConstantFaceBC<T>(i, const_m, ecb_m));
    }

private:

  T const_m;
  bool ecb_m;
};

#endif // POOMA_NEWFIELD_UPDATER_CONSTANTFACEBC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ConstantFaceBC.h,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/07/20 21:03:51 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
