// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// UpdaterListItem: the ultimate base class for all automatic update objects.
//-----------------------------------------------------------------------------

#ifndef POOMA_UPDATERS_UPDATERLISTITEM_H
#define POOMA_UPDATERS_UPDATERLISTITEM_H

#if defined(POOMA_DEBUG_UPDATERS)
#include <iostream>
#endif

//-----------------------------------------------------------------------------
// UpdaterListItem is an abstract base class for all updaters. It
// provides an interface for the most basic services: responding to pre-read 
// and post-write events and setting/clearing the dirty flag. 
//
// This class is not templated because we want to store these puppies in
// UpdaterList containers, which are also not templated. 
//-----------------------------------------------------------------------------

class UpdaterListItem {
public:

  //---------------------------------------------------------------------------
  // Default and copy constructors.

  UpdaterListItem() : dirty_m(true) { }
  UpdaterListItem(const UpdaterListItem &model)
  : dirty_m(model.dirty_m) { }
  
  
  //---------------------------------------------------------------------------
  // Trivial destructor, but virtual since we will be subclassing.

  virtual ~UpdaterListItem() { }
  

  //---------------------------------------------------------------------------
  // Update function function. Subclasses must override to make this updater
  // actually do something.
  
  virtual void update() = 0; 


  //---------------------------------------------------------------------------
  // Notification functions. 

  virtual void notifyPreRead()
    {
      // This function is called if somebody is getting ready to read somewhere 
      // and we need to update. 
    
      if (dirty())
        {
          update();
          clearDirty();
        }
    }
  
  virtual void notifyPostWrite()
    {
      // This function is called if somebody has already written somewhere.
      // We simply set the dirty flag.
    
      setDirty();
    }


  //---------------------------------------------------------------------------
  // Accessors for the dirty flag.

  inline bool dirty() const { return dirty_m; }
    

  //---------------------------------------------------------------------------
  // Modifiers used to set/clear the dirty flag.
  
  void setDirty()
  {
#if defined(POOMA_DEBUG_UPDATERS)
    std::cout << "Setting dirty flag for " << (void *) this 
              << std::endl;
#endif
    dirty_m = true;
  }  
  void clearDirty()
  {
#if defined(POOMA_DEBUG_UPDATERS)
    std::cout << "Clearing dirty flag for " << (void *) this 
              << std::endl;
#endif
    dirty_m = false;
  }  
    
private:

  bool dirty_m;
  
};

#endif // POOMA_BCONDS_UpdaterListItem_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UpdaterListItem.h,v $   $Author: swhaney $
// $Revision: 1.1 $   $Date: 2000/07/20 20:38:30 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
