// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No.  W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE, and to allow others to do so.
// The public may copy and use this SOFTWARE, FOR NONCOMMERCIAL USE ONLY,
// without charge, provided that this Notice and any statement of
// authorship are reproduced on all copies.  Neither the Government nor
// the University makes any warranty, express or implied, or assumes any
// liability or responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// ErrorType
// ForEachRef
//-----------------------------------------------------------------------------

#ifndef POOMA_PETE_POOMAPETE_H
#define POOMA_PETE_POOMAPETE_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// This file contains POOMA specific extensions to PETE.  Please refrain from
// changing other PETE files in this directory without making corresponding
// changes to the PETE repository.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/ErrorType.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Reference versions of ForEach.
//-----------------------------------------------------------------------------

template<class Expr, class FTag, class CTag>
struct ForEachRef
{
  typedef typename LeafFunctor<Expr, FTag>::Type_t Type_t;
  inline static
  const Type_t &apply(const Expr &expr, const FTag &f, const CTag &)
    {
      return LeafFunctor<Expr, FTag>::apply(expr, f);
    }
};

template<class Expr, class FTag, class CTag>
inline const typename ForEachRef<Expr,FTag,CTag>::Type_t &
forEachRef(const Expr &e, const FTag &f, const CTag &c)
{
  return ForEachRef<Expr, FTag, CTag>::apply(e, f, c);
}

template<class Op, class A, class FTag, class CTag>
struct ForEachRef<UnaryNode<Op, A>, FTag, CTag>
{
  typedef typename ForEachRef<A, FTag, CTag>::Type_t TypeA_t;
  typedef typename Combine1<TypeA_t, Op, CTag>::Type_t Type_t;
  inline static
  const Type_t &apply(const UnaryNode<Op, A> &expr, const FTag &f, 
    const CTag &c) 
    {
      return Combine1<TypeA_t, Op, CTag>::
        combine(ForEachRef<A, FTag, CTag>::apply(expr.child(), f, c),
                expr.operation(), c);
    }
};

template<class Op, class A, class B, class FTag, class CTag>
struct ForEachRef<BinaryNode<Op, A, B>, FTag, CTag >
{
  typedef typename ForEachRef<A, FTag, CTag>::Type_t TypeA_t;
  typedef typename ForEachRef<B, FTag, CTag>::Type_t TypeB_t;
  typedef typename Combine2<TypeA_t, TypeB_t, Op, CTag>::Type_t Type_t;
  inline static
  const Type_t &apply(const BinaryNode<Op, A, B> &expr, const FTag &f,
	                  const CTag &c) 
    {
      return Combine2<TypeA_t, TypeB_t, Op, CTag>::
        combine(ForEachRef<A, FTag, CTag>::apply(expr.left(), f, c),
                ForEachRef<B, FTag, CTag>::apply(expr.right(), f, c),
	            expr.operation(), c);
    }
};

template<class Op, class A, class B, class C, class FTag, class CTag>
struct ForEachRef<TrinaryNode<Op, A, B, C>, FTag, CTag >
{
  typedef typename ForEachRef<A, FTag, CTag>::Type_t TypeA_t;
  typedef typename ForEachRef<B, FTag, CTag>::Type_t TypeB_t;
  typedef typename ForEachRef<C, FTag, CTag>::Type_t TypeC_t;
  typedef typename Combine3<TypeA_t, TypeB_t, TypeC_t, Op, CTag>::Type_t 
    Type_t;
  inline static
  const Type_t &apply(const TrinaryNode<Op, A, B, C> &expr, const FTag &f,
	                  const CTag &c) 
    {
      return Combine3<TypeA_t, TypeB_t, TypeC_t, Op, CTag>::
        combine(ForEachRef<A, FTag, CTag>::apply(expr.left(), f, c),
	        ForEachRef<B, FTag, CTag>::apply(expr.middle(), f, c),
	        ForEachRef<C, FTag, CTag>::apply(expr.right(), f, c),
	        expr.operation(), c);
    }
};

template<class T, class FTag, class CTag>
struct ForEachRef<Expression<T>, FTag, CTag>
{
  typedef typename ForEachRef<T, FTag, CTag>::Type_t Type_t;
  inline static
  const Type_t &apply(const Expression<T> &expr, const FTag &f, 
	                  const CTag &c) 
    {
      return ForEachRef<T, FTag, CTag>::apply(expr.expression(), f, c);
    }
};

template<class T, class FTag, class CTag>
struct ForEachRef<Reference<T>, FTag, CTag>
{
  typedef typename ForEachRef<T, FTag, CTag>::Type_t Type_t;
  inline static
  const Type_t &apply(const Reference<T> &ref, const FTag &f, 
	                  const CTag &c) 
    {
      return ForEachRef<T, FTag, CTag>::apply(ref.reference(), f, c);
    }
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_PETE_POOMAPETE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PoomaPETE.h,v $   $Author: sa_smith $
// $Revision: 1.1 $   $Date: 2000/08/08 17:34:09 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
