// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
//   ParticleBCList
//-----------------------------------------------------------------------------

#ifndef POOMA_PARTICLES_PARTICLEBCLIST_H
#define POOMA_PARTICLES_PARTICLEBCLIST_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//-----------------------------------------------------------------------------
//
// ParticleBCList holds a list of ParticleBCItems for the Particles class,
// storing them in an STL vector of ParticleBCItem pointers.
//
//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Particles/ParticleBCItem.h"
#include "Utilities/PAssert.h"

#include <vector>
#include <iosfwd>

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
//
//-----------------------------------------------------------------------------

class ParticleBCList
{
public:

  // Typedefs.
  typedef std::vector<ParticleBCItem*>  BCContainer_t;
  typedef BCContainer_t::size_type      Size_t;

  // Constructors.
  ParticleBCList();
  ParticleBCList(const ParticleBCList&);

  // Destructor.
  ~ParticleBCList();

  // Methods.

  // Retrun number of boundary conditions.
  Size_t size() const { return bcList_m.size(); }

  // Access the ith boundary condition.
  const ParticleBCItem* operator()(Size_t i) const
  {
    PAssert(i < size());
    return bcList_m[i];
  }
  ParticleBCItem* operator()(Size_t i)
  {
    PAssert(i < size());
    return bcList_m[i];
  }

  // Add a boundary condition to the end of the list.
  // Returns the index number of the newly added ParticleBC.
  template <class Subject, class Object, class BCType>
  Size_t addBC(const Subject& s, const Object& o, const BCType& bc)
  {
    // BCType creates ParticleBC, which is added to list.
    bcList_m.push_back( bc.create(s, o) );
    return (size() - 1);
  }
  // Same, except one argument acts as both subject and object.
  template <class Subject, class BCType>
  Size_t addBC(const Subject& s, const BCType& bc)
  {
    bcList_m.push_back( bc.create(s) );
    return (size() - 1);
  }

  // Remove boundary condition with given index number from the list.
  void removeBC(Size_t);

  // Print each ParticleBC in the list to an ostream.
  void print(std::ostream&) const;

private:

  // The actual vector of ParticleBCItem pointers.
  BCContainer_t bcList_m;
};


// declare operator<< for a std::ostream and a ParticleBCList

std::ostream&
operator<<(std::ostream&, const ParticleBCList&);


#endif     // POOMA_PARTICLES_PARTICLEBCLIST_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ParticleBCList.h,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/03/07 13:17:49 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
