// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_OBSERVER_EVENT_H
#define POOMA_UTILITIES_OBSERVER_EVENT_H

//-----------------------------------------------------------------------------
// Classes:
// ObserverEvent
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
// ObserverEvent class - a base class for all events that will be passed
// on to observers from observables.  It includes one integer data member
// used to indicate to observer subclasses what kind of event it is.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Utilities/Unique.h"


//-----------------------------------------------------------------------------
//
// Full Description:
//
// The Observer class, along with the Observable class, are used to implement
// the observer pattern.  In this pattern, there are two sets of objects:
//   1. Observable<T> objects, which contain a list of Observer<T>  pointers.
//   2. Observer<T> objects, which check in as observers of any number of
//      Observable objects.
// When the Observer<T> is initialized, it should call the 'attach' method of
// all Observable<T> objects it needs to watch.  When the Observable changes
// in some way, for example when it changes state is or is deleted, the
// Observable will call the 'notify' method of all the Obserers registered
// with it.  An Observer<T> can stop watching an object, by calling the
// 'detach' method of that Observable.
//
// ObserverEvent is the type of object passed to the notify method.
// It contains an integer indicating the event "code", that classes can
// examine and use to downcast the event if necessary.  There is also a
// version of notify that just takes an integer; this is wrapped in
// an ObserverEvent, and passed on.
//
// If you have an event that requires some more information beyond just
// an event code, make a subclass of ObserverEvent and have the Observer's
// that get that event cast the event object to the proper type.
//
// ObserverEvent's also have a unique ID value, obtained via the
// ID() method, with type ObserverEvent::ID_t
//
//-----------------------------------------------------------------------------


///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

class ObserverEvent
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  typedef Unique::Value_t ID_t;


  //============================================================
  // Constructors
  //============================================================

  // Constructor: Must specify the integer code

  ObserverEvent(int event)
    : event_m(event), ID_m(Unique::lockedGet())
    {
    }

  // Copy constructor

  ObserverEvent(const ObserverEvent &oe)
    : event_m(oe.event_m), ID_m(oe.ID_m)
    {
    }

  // Assignment operator

  ObserverEvent &operator=(const ObserverEvent &oe)
    {
      event_m = oe.event();
      ID_m = oe.ID();
      return *this;
    }


  //============================================================
  // Destructors
  //============================================================

  // Nothing to do here.  We make this virtual so that we can do
  // dynamic casts.

  virtual ~ObserverEvent()
    {
    }


  //============================================================
  // Accessors
  //============================================================

  // Return our current event code

  inline int event() const
    {
      return event_m;
    }

  // Return our ID value

  inline ID_t ID() const
    {
      return ID_m;
    }

  // Return a value which indicates a "null ID", meaning one that does
  // not refer to any particular event.  This is useful for initializing
  // event values in constructors, etc.  It is static so that you do
  // not need to create a particular ObserverEvent instance to get this
  // value.

  static inline ID_t nullID()
    {
      return (-1);
    }

private:
  // The integer event code

  int event_m;

  // The unique ID vlaue

  ID_t ID_m;
};


//-----------------------------------------------------------------------------
//
// Full Description:
//
// checkDynamicID(obj, ID) is a specializable function that is used
// by some classes to check the dynamic ID value stored in the first
// argument by some means.  If it is the same as the given ID, this
// returns false.  If it is not the same, it should return true and
// change the state of obj to indicate that it has "seen" the given ID.
//
// The default version of this just returns true, generally meaning,
// "this ID has not been seen, proceed".
//
//-----------------------------------------------------------------------------

template<class Obj>
inline bool checkDynamicID(Obj &, ObserverEvent::ID_t)
{
  return true;
}


// } // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_UTILITIES_OBSERVER_EVENT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ObserverEvent.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/07 13:18:25 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

