// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_POOLED_H
#define POOMA_UTILITIES_POOLED_H

// Classes:
// Pooled
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
// Overview:
//
// Pooled: A mixin class for providing fast new and delete.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Utilities/Pool.h"
#include "Pooma/Configuration.h"


//-----------------------------------------------------------------------------
//
// Full Description:
//
// This mixin class provides two functions: new and delete.
// It has a static Pool from which it gets and returns memory.
//
// You use this class by inheriting from it like so:
//
// class A : public Pooled<A>
// {
//   ...
// };
//
// Pooled is templated on the class that inherits from it so that it
// will know the size of the blocks to request from the pool. 
//
// This technique will not be correct for a class B which inherits
// from A, so Pooled can only be used for classes which are leaves in
// the inheritance heirarchy.
//
//-----------------------------------------------------------------------------

template<class T>
class Pooled
{
#if POOMA_POOLED
public:

  // Allocate memory by getting it from the pool.
  inline void* operator new(size_t) { return pool_s.alloc(); }

  // Return memory to the pool.
  inline void operator delete(void *p, size_t) { if (p) pool_s.free(p); }

  // Provide a placement version that works.
  inline void* operator new(size_t, void* ptr) { return ptr; }

  // Need a matching delete
#if ! POOMA_NO_PLACEMENT_DELETE
  inline void operator delete(void *, void *) { }
#endif

private:

  // The pool which all objects of type T use.
  static Pool pool_s;
#endif
};


// Declare the pool for objects of type T.
#if POOMA_POOLED
template<class T>
Pool Pooled<T>::pool_s(sizeof(T));
#endif

//////////////////////////////////////////////////////////////////////

#endif

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Pooled.h,v $   $Author: swhaney $
// $Revision: 1.14 $   $Date: 2000/03/07 13:18:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
