// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// RefCountedPtr test code.
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/RefCountedPtr.h"
#include "Utilities/RefCounted.h"
#include "Utilities/PAssert.h"
#include "Utilities/Tester.h"

#include <iostream>

typedef RefCountedPtr<Shared<int> > RCIntPtr_t;

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

#if POOMA_EXCEPTIONS
  try {
#endif
    tester.out() << "\n\nTesting RefCountedPtr with Shared<int>.\n" 
		 << std::endl;

    RCIntPtr_t pn;
    pn = new Shared<int>(2);

    tester.out() << "pn->data() = " << pn->data() << std::endl;

    *pn = 5;

    tester.out() << "pn->data() = " << pn->data() << std::endl;

    RCIntPtr_t p1(new Shared<int>(1));
    RCIntPtr_t p2(new Shared<int>(2));
    RCIntPtr_t p3(new Shared<int>(3));

    tester.out() << p1->data() << " " 
         << p2->data() << " "
         << p3->data() << " "
         << std::endl;

    *p1 = *p2 = *p3 = -777;

    tester.out() << p1->data() << " " 
         << p2->data() << " "
         << p3->data() << " "
         << std::endl;

    PAssert(*p1 == *p2 && *p1 == *p3 && *p2 == *p3);
    PAssert(p1 != p2 && p1 != p3 && p2 != p3);

    PAssert(!p1.isShared());
    PAssert(!p2.isShared());
    PAssert(!p3.isShared());
    PAssert(!pn.isShared());

    RCIntPtr_t pc = pn;

    PAssert(pn.isShared());
    PAssert(pc.isShared());

    PAssert(pn == pc);
    PAssert(*pn == *pc);

    pn.invalidate();

    PAssert(!pc.isShared());

    tester.out() << "Value = " << pc->data() << std::endl;

    {
      RCIntPtr_t pn = pc;
      PAssert(pn == pc);
      PAssert(*pn == *pc);
      PAssert(pn.isShared());
      PAssert(pc.isShared());

      pn.makeOwnCopy();
      
      PAssert(pn != pc);
      PAssert(*pn == *pc);
      PAssert(!pn.isShared());
      PAssert(!pc.isShared());
      
      tester.out() << "Making copy and modifying. "
	   << "Next two shouldn't be the same" << std::endl;
      RCIntPtr_t pv = pn;
      pv.makeOwnCopy();
      
      *pv = pv->data() - 999;
      tester.out() << "Value = " << pv->data() << std::endl;
      tester.out() << "Value = " << pn->data() << std::endl;
      PAssert(*pv != *pn);
      tester.out() << std::endl;

      RCIntPtr_t p1 = pc;
      PAssert(p1.isShared());
      PAssert(pc.isShared());
      PAssert(*p1 == *pn);
    }

    PAssert(!pc.isShared());

    pc.invalidate();
#if POOMA_EXCEPTIONS
  }
  catch(const char *err) 
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
  catch(const Pooma::Assertion &err)
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
#endif    
  tester.out() << "All Done!" << std::endl;
  int retval = tester.results("rcptr_test2");
  Pooma::finalize();
  return retval;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: rcptr_test2.cpp,v $   $Author: swhaney $
// $Revision: 1.11 $   $Date: 2000/03/07 13:18:33 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
