/* Process declarations and variables for C compiler.
   Copyright (C) 1988 Free Software Foundation, Inc.
   Hacked by Michael Tiemann (tiemann@mcc.com)

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


/* Process declarations and symbol lookup for C front end.
   Also constructs types; the standard scalar types at initialization,
   and structure, union, array and enum types when they are declared.  */

/* ??? not all decl nodes are given the most useful possible
   line numbers.  For example, the CONST_DECLs for enum values.  */

#include "config.h"
#include "tree.h"
#include "flags.h"
#include "cplus-tree.h"
#include "cplus-parse.h"
#include "assert.h"

/* In grokdeclarator, distinguish syntactic contexts of declarators.  */
enum decl_context
{ NORMAL,			/* Ordinary declaration */
  FUNCDEF,			/* Function definition */
  PARM,				/* Declaration of parm before function body */
  FIELD,			/* Declaration inside struct or union */
  TYPENAME,			/* Typename (inside cast or sizeof)  */
  MEMFUNCDEF,			/* Member function definition */
};

#define NULL 0
#define MIN(X,Y) ((X) < (Y) ? (X) : (Y))
#define MAX(X,Y) ((X) > (Y) ? (X) : (Y))

#ifndef CHAR_TYPE_SIZE
#define CHAR_TYPE_SIZE BITS_PER_UNIT
#endif

#ifndef SHORT_TYPE_SIZE
#define SHORT_TYPE_SIZE (BITS_PER_UNIT * MIN (UNITS_PER_WORD / 2, 2))
#endif

#ifndef INT_TYPE_SIZE
#define INT_TYPE_SIZE BITS_PER_WORD
#endif

#ifndef LONG_TYPE_SIZE
#define LONG_TYPE_SIZE BITS_PER_WORD
#endif

#ifndef LONG_LONG_TYPE_SIZE
#define LONG_LONG_TYPE_SIZE (BITS_PER_WORD * 2)
#endif

#ifndef FLOAT_TYPE_SIZE
#define FLOAT_TYPE_SIZE BITS_PER_WORD
#endif

#ifndef DOUBLE_TYPE_SIZE
#define DOUBLE_TYPE_SIZE (BITS_PER_WORD * 2)
#endif

static tree grokparms (), grokdeclarator ();
tree pushdecl ();
static void builtin_function ();
/* static */ void grokclassfn ();
/* static */ tree grokopexpr (), grokoptypename ();

static tree lookup_tag ();
static tree lookup_tag_reverse ();
static tree lookup_name_current_level ();
static char *redeclaration_error_message ();
static int parmlist_is_exprlist ();
static int parmlist_is_random ();
static void grok_ctor_properties ();
static void grok_op_properties ();

tree finish_table ();

/* a node which has tree code ERROR_MARK, and whose type is itself.
   All erroneous expressions are replaced with this node.  All functions
   that accept nodes as arguments should avoid generating error messages
   if this node is one of the arguments, since it is undesirable to get
   multiple error messages from one error in the input.  */

tree error_mark_node;

/* Erroneous argument lists can use this *IFF* they do not modify it.  */
tree error_mark_list;

/* INTEGER_TYPE and REAL_TYPE nodes for the standard data types */

tree short_integer_type_node;
tree integer_type_node;
tree long_integer_type_node;
tree long_long_integer_type_node;

tree short_unsigned_type_node;
tree unsigned_type_node;
tree long_unsigned_type_node;
tree long_long_unsigned_type_node;

tree unsigned_char_type_node;
tree signed_char_type_node;
tree char_type_node;

tree float_type_node;
tree double_type_node;
tree long_double_type_node;

/* a VOID_TYPE node, and the same, packaged in a TREE_LIST.  */

tree void_type_node, void_list_node;

/* A node for type `void *'.  */

tree ptr_type_node;

/* A node for type `char *'.  */

tree string_type_node;

/* Type `char[256]' or something like it.
   Used when an array of char is needed and the size is irrelevant.  */

tree char_array_type_node;

/* Type `int[256]' or something like it.
   Used when an array of int needed and the size is irrelevant.  */

tree int_array_type_node;

/* type `int ()' -- used for implicit declaration of functions.  */

tree default_function_type;

/* function types `double (double)' and `double (double, double)', etc.  */

tree double_ftype_double, double_ftype_double_double;
tree int_ftype_int, long_ftype_long;

/* Function type `void (void *, void *, int)' and similar ones.  */

tree void_ftype_ptr_ptr_int, int_ftype_ptr_ptr_int, void_ftype_ptr_int_int;

/* C++ extensions */
tree vtable_entry_type;
tree class_type_node, record_type_node, union_type_node, enum_type_node;
tree unknown_type_node;

/* Function type `void * (long)', 'void (void *)' */
tree ptr_ftype_long, void_ftype_ptr;
tree ptr_ftype_ptr_int_int_ptr, void_ftype_ptr_int_int_ptr_int_int;

/* Used for virtual function tables.  */
tree vtbl_mask;

/* Array type `(void *)[]' */
tree vtbl_type_node;

/* Used to help generate temporary names which are unique within
   a function.  Reset to 0 by start_function.  */

static int temp_name_counter = 0;

/* Same, but not reset.  Local temp variables and global temp variables
   can have the same name.  */
static int global_temp_name_counter = 0;

#ifdef SOS
/* SOS extensions.  */
tree zlink_type, zret_type;
tree zlink, zret;
#endif

/* -- end of C++ */

/* Two expressions that are constants with value zero.
   The first is of type `int', the second of type `void *'.  */

tree integer_zero_node;
tree null_pointer_node;

/* A node for the integer constant 1.  */

tree integer_one_node;

/* An identifier whose name is <value>.  This is used as the "name"
   of the RESULT_DECLs for values of functions.  */

tree value_identifier;

/* While defining an enum type, this is 1 plus the last enumerator
   constant value.  */

static tree enum_next_value;

/* Parsing a function declarator leaves a list of parameter names
   or a chain or parameter decls here.  */

static tree last_function_parms;

/* Parsing a function declarator leaves here a chain of structure
   and enum types declared in the parmlist.  */

static tree last_function_parm_tags;

/* After parsing the declarator that starts a function definition,
   `start_function' puts here the list of parameter names or chain of decls.
   `store_parm_decls' finds it here.  */

static tree current_function_parms;

/* Similar, for last_function_parm_tags.  */
static tree current_function_parm_tags;

/* A list (chain of TREE_LIST nodes) of all LABEL_STMTs in the function
   that have names.  Here so we can clear out their names' definitions
   at the end of the function.  */

static tree named_labels;

/* A list of objects which have constructors or destructors
   which reside in the global scope.  The decl is stored in
   the TREE_VALUE slot and the initializer is stored
   in the TREE_PURPOSE slot.  */
static tree global_aggregates;
/* A list of static class variables.  This is needed, because a
   static class variable can be declared inside the class without
   an initializer, and then initialized, staticly, outside the class.  */
static tree pending_statics;

/* A list of functions which were declared inline, but later had their
   address taken.  Used only for non-virtual member functions, since we can
   find other functions easily enough.  */
static tree pending_addressable_inlines;

/* A list of overloaded functions which we should forget ever
   existed, such as functions declared in function scope,
   once we leave function scope.  */
static tree overloads_to_forget;

/* The FUNCTION_DECL for the function currently being compiled,
   or 0 if between functions.  */
tree current_function_decl;

/* Set to 0 at beginning of a function definition, set to 1 if
   a return statement that specifies a return value is seen.  */

int current_function_returns_value;

/* Set to 0 at beginning of a function definition, set to 1 if
   a return statement with no argument is seen.  */

int current_function_returns_null;

/* Set to nonzero by `grokdeclarator' for a function
   whose return type is defaulted, if warnings for this are desired.  */

static int warn_about_return_type;

/* C (and C++) language-specific option variables.  */

/* Nonzero means `char' should be signed.  */

int flag_signed_char;

/* Nonzero means allow type mismatches in conditional expressions;
   just make their values `void'.   */

int flag_cond_mismatch;

/* Nonzero means don't recognize the keyword `asm'.  */

int flag_no_asm;

/* Nonzero means do some things the same way PCC does.  */

int flag_traditional;

/* Nonzero means warn about implicit declarations.  */

int warn_implicit = 1;

/* Nonzero means warn about function definitions that default the return type
   or that use a null return and have a return-type other than void.  */

int warn_return_type;

/* Nonzero means give string constants the type `const char *'
   to get extra warnings from them.  These warnings will be too numerous
   to be useful, except in thoroughly ANSIfied programs.  */

int warn_write_strings;

/* Nonzero means warn about pointer casts that can drop a type qualifier
   from the pointer target type.  */

int warn_cast_qual;

/* Nonzero means warn about sizeof(function) or addition/subtraction
   of function pointers.  */

int warn_pointer_arith;

/* Nonzero means warn for all old-style non-prototype function decls.  */

int warn_strict_prototypes;

/* Non-zero means warn in function declared in derived class has the
   same name as a virtual in the base class, but fails to match the
   type signature of any virtual function in the base class.  */
int warn_overloaded_virtual;

/* Nonzero means `$' can be in an identifier.
   See cccp.c for reasons why this breaks some obscure ANSI C programs.  */

#ifndef DOLLARS_IN_IDENTIFIERS
#define DOLLARS_IN_IDENTIFIERS 0
#endif
int dollars_in_ident = DOLLARS_IN_IDENTIFIERS;

char *language_string = "GNU C++";

/* Nonzero for -no-strict-prototype switch: do not consider empty
   argument prototype to mean function takes no arguments.  */

int strict_prototype = 1;
int strict_prototypes_lang_c, strict_prototypes_lang_cplusplus = 1;

/* Nonzero means that labels can be used as first-class objects */

int flag_labels_ok;

/* Non-zero means to collect statistics which might be expensive
   and to print them when we are done.  */
int flag_detailed_statistics;

/* C++ specific flags.  */   
/* Nonzero for -fall-virtual: make every member function (except
   constructors) lay down in the virtual function table.  Calls
   can then either go through the virtual function table or not,
   depending.  */

int flag_all_virtual;

/* Nonzero to make characters constants have the data type `char'.
   This data type is signed or unsigned depending on flag_signed_char.  */

int flag_char_charconst;

/* Nonzero means that we cannot make optimizing assumptions about `this'.
   Blow this off, and do it right later.  */

int flag_this_is_variable = 1;

/* Nonzero means memoize our member lookups.  */

int flag_memoize_lookups = 0; int flag_save_memoized_contexts = 0;

/* Nonzero means to implement `dynamic' features a la SOS.  */

int flag_dynamic = 0;

/* 1 means write out virtual function tables and give them
   (C) public visibility.
   0 means write out virtual function tables and give them
   (C) static visibility (default).
   -1 means declare virtual function tables extern.  */

int write_virtuals = 0;

/* Nonzero means we should attempt to elide constructors when possible.  */

int flag_elide_constructors;

/* Decode the string P as a language-specific option.
   Return 1 if it is recognized (and handle it);
   return 0 if not recognized.  */
   
lang_decode_option (p)
     char *p;
{
  if (!strcmp (p, "-ftraditional"))
    flag_traditional = 1, dollars_in_ident = 1;
  else if (!strcmp (p, "-fsigned-char"))
    flag_signed_char = 1;
  else if (!strcmp (p, "-funsigned-char"))
    flag_signed_char = 0;
  else if (!strcmp (p, "-fno-signed-char"))
    flag_signed_char = 0;
  else if (!strcmp (p, "-fno-unsigned-char"))
    flag_signed_char = 1;
  else if (!strcmp (p, "-fcond-mismatch"))
    flag_cond_mismatch = 1;
  else if (!strcmp (p, "-fno-cond-mismatch"))
    flag_cond_mismatch = 0;
  else if (!strcmp (p, "-fasm"))
    flag_no_asm = 0;
  else if (!strcmp (p, "-fno-asm"))
    flag_no_asm = 1;
  else if (!strcmp (p, "-flabels-ok"))
    {
#if 1
      error ("option -flabels-ok is not implemented for now");
#else
      flag_labels_ok = 1;
#endif
    }
  else if (!strcmp (p, "-fno-labels-ok"))
    flag_labels_ok = 0;
  else if (!strcmp (p, "-fstats"))
    flag_detailed_statistics = 1;
  else if (!strcmp (p, "-fno-stats"))
    flag_detailed_statistics = 0;
  else if (!strcmp (p, "-traditional"))
    flag_traditional = 1, dollars_in_ident = 1;
  else if (!strcmp (p, "-ansi"))
    flag_no_asm = 1, dollars_in_ident = 0;
  else if (!strcmp (p, "-Wimplicit"))
    warn_implicit = 1;
  else if (!strcmp (p, "-Wreturn-type"))
    warn_return_type = 1;
  else if (!strcmp (p, "-Wwrite-strings"))
    warn_write_strings = 1;
  else if (!strcmp (p, "-Wcast-qual"))
    warn_cast_qual = 1;
  else if (!strcmp (p, "-Wpointer-arith"))
    warn_pointer_arith = 1;
  else if (!strcmp (p, "-Wstrict-prototypes"))
    warn_strict_prototypes = 1;
  else if (!strcmp (p, "-Wcomment"))
    ; /* cpp handles this one.  */
  else if (!strcmp (p, "-Wcomments"))
    ; /* cpp handles this one.  */
  else if (!strcmp (p, "-Wtrigraphs"))
    ; /* cpp handles this one.  */
  else if (!strcmp (p, "-Wall"))
    {
      extra_warnings = 1;
      warn_implicit = 1;
      warn_return_type = 1;
      warn_unused = 1;
      warn_switch = 1;
    }

  /* C++ extensions */
  else if (!strcmp (p, "-fno-strict-prototype"))
    strict_prototypes_lang_cplusplus = 0;
  else if (!strcmp (p, "-fno-warn-implicit"))
    warn_implicit = 0;
  else if (!strcmp (p, "-fall-virtual"))
    flag_all_virtual = 1;
  else if (!strcmp (p, "-fchar-charconst"))
    flag_char_charconst = 1;
  else if (!strcmp (p, "-fmemoize-lookups"))
    flag_memoize_lookups = 1;
  else if (!strcmp (p, "-fsave-memoized"))
    {
      flag_memoize_lookups = 1;
      flag_save_memoized_contexts = 1;
    }
  else if (!strcmp (p, "-fSOS"))
    flag_all_virtual = 2;
  else if (!strcmp (p, "-felide-constructors"))
    flag_elide_constructors = 1;
  else if (!strcmp (p, "-Woverloaded-virtual"))
    warn_overloaded_virtual = 1;
  else
    return 0;

  return 1;
}

/* Set to 0 at beginning of a constructor, set to 1
   if that function does an allocation before referencing its
   instance variable.  */
int current_function_assigns_this;
int current_function_just_assigned_this;

/* Set to 0 at beginning of a function.  Set non-zero when
   store_parm_decls is called.  Don't call store_parm_decls
   if this flag is non-zero!  */
int current_function_parms_stored;

/* For each binding contour we allocate a binding_level structure
 * which records the names defined in that contour.
 * Contours include:
 *  0) the global one
 *  1) one for each function definition,
 *     where internal declarations of the parameters appear.
 *  2) one for each compound statement,
 *     to record its declarations.
 *
 * The current meaning of a name can be found by searching the levels from
 * the current one out to the global one.
 *
 * Off to the side, may be the class_binding_level.  This exists
 * only to catch class-local declarations.  It is otherwise
 * nonexistent.
 */

/* Note that the information in the `names' component of the global contour
   is duplicated in the IDENTIFIER_GLOBAL_VALUEs of all identifiers.  */

struct binding_level
  {
    /* A chain of _DECL nodes for all variables, constants, functions,
       and typedef types.  These are in the reverse of the order supplied.
     */
    tree names;

    /* A list of structure, union and enum definitions,
     * for looking up tag names.
     * It is a chain of TREE_LIST nodes, each of whose TREE_PURPOSE is a name,
     * or NULL_TREE; and whose TREE_VALUE is a RECORD_TYPE, UNION_TYPE,
     * or ENUMERAL_TYPE node.
     *
     * C++: the TREE_VALUE nodes can be simple types for component_bindings.
     *
     */
    tree tags;

    /* For each level, a list of shadowed outer-level local definitions
       to be restored when this level is popped.
       Each link is a TREE_LIST whose TREE_PURPOSE is an identifier and
       whose TREE_VALUE is its old definition (a kind of ..._DECL node).  */
    tree shadowed;

    /* For each level (except not the global one),
       a chain of LET_STMT nodes for all the levels
       that were entered and exited one level down.  */
    tree blocks;

    /* The binding level which this one is contained in (inherits from).  */
    struct binding_level *level_chain;

    /* > 0 for the level that holds the parameters of a function.
       < 0 for the level that holds a class declaration.  */
    char parm_flag;

    /* Nonzero if this level "doesn't exist" for tags.  */
    char tag_transparent;

    /* Nonzero if this level can safely have additional
       cleanup-needing variables added to it.  */
    char more_cleanups_ok;

    /* Nonzero for temporary binding contours created by the compiler.  */
    char temporary;

    /* Number of decls in `names' that have incomplete 
       structure or union types.  */
    int n_incomplete;
  };

#define NULL_BINDING_LEVEL (struct binding_level *) NULL
  
/* The binding level currently in effect.  */

static struct binding_level *current_binding_level;

/* The binding level of the current class, if any.  */

static struct binding_level *class_binding_level;

/* A chain of binding_level structures awaiting reuse.  */

static struct binding_level *free_binding_level;

/* The outermost binding level, for names of file scope.
   This is created when the compiler is started and exists
   through the entire run.  */

static struct binding_level *global_binding_level;

/* Binding level structures are initialized by copying this one.  */

static struct binding_level clear_binding_level
  = {NULL, NULL, NULL, NULL, NULL, 0, 0, 0, 0};

/* Create a new `struct binding_level'.  */

static
struct binding_level *
make_binding_level ()
{
  /* NOSTRICT */
  return (struct binding_level *) xmalloc (sizeof (struct binding_level));
}

/* Nonzero if we are currently in the global binding level.  */

int
global_bindings_p ()
{
  return current_binding_level == global_binding_level;
}

/* Enter a new binding level.
   If TAG_TRANSPARENT is nonzero, do so only for the name space of variables,
   not for that of tags.  */

void
pushlevel (tag_transparent)
     int tag_transparent;
{
  register struct binding_level *newlevel = NULL_BINDING_LEVEL;

  /* If this is the top level of a function,
     just make sure that NAMED_LABELS is 0.
     They should have been set to 0 at the end of the previous function.  */

  if (current_binding_level == global_binding_level)
    {
      if (named_labels)
	abort ();
    }

  /* Reuse or create a struct for this binding level.  */

  if (free_binding_level)
    {
      newlevel = free_binding_level;
      free_binding_level = free_binding_level->level_chain;
    }
  else
    {
      newlevel = make_binding_level ();
    }

  /* Add this level to the front of the chain (stack) of levels that
     are active.  */

  *newlevel = clear_binding_level;
  if (class_binding_level)
    {
      newlevel->level_chain = class_binding_level;
      class_binding_level = 0;
    }
  else
    {
      newlevel->level_chain = current_binding_level;
    }
  current_binding_level = newlevel;
  newlevel->tag_transparent = tag_transparent;
  newlevel->more_cleanups_ok = 1;
  newlevel->temporary = 0;
}

void
pushlevel_temporary (tag_transparent)
     int tag_transparent;
{
  pushlevel (tag_transparent);
  current_binding_level->temporary = 1;
  clear_last_expr ();
#if 0
  /* Don't call push_momentary here!  It will cause cleanups
     to be allocated on the momentary obstack, and they
     will be overwritten by the next statement.  */
  push_momentary ();
#endif
  expand_start_bindings (0);
}

/* Exit a binding level.
   Pop the level off, and restore the state of the identifier-decl mappings
   that were in effect when this level was entered.

   If KEEP == 1, this level had explicit declarations, so
   and create a "block" (a LET_STMT node) for the level
   to record its declarations and subblocks for symbol table output.

   If KEEP == 2, this levels subblocks go to the front,
   not the back of the current binding level.  This happens,
   for instance, when code for constructors and destructors
   need to generate code at the end of a function which must
   be moved up to the front of the function.

   If FUNCTIONBODY is nonzero, this level is the body of a function,
   so create a block as if KEEP were set and also clear out all
   label names.

   If REVERSE is nonzero, reverse the order of decls before putting
   them into the LET_STMT.  */

void
poplevel (keep, reverse, functionbody)
     int keep;
     int reverse;
     int functionbody;
{
  register tree link;
  /* The chain of decls was accumulated in reverse order.
     Put it into forward order, just for cleanliness.  */
  tree decls;
  int tmp = functionbody;
  int real_functionbody = current_binding_level->temporary
    ? ((functionbody = 0), tmp) : functionbody;
  tree tags = functionbody >= 0 ? current_binding_level->tags : 0;
  tree subblocks = functionbody >= 0 ? current_binding_level->blocks : 0;
  tree block = 0;

  /* This warning is turned off because it causes warnings for
     declarations like `extern struct foo *x'.  */
#if 0
  /* Warn about incomplete structure types in this level.  */
  for (link = tags; link; link = TREE_CHAIN (link))
    if (TYPE_SIZE (TREE_VALUE (link)) == 0)
      {
	tree type = TREE_VALUE (link);
	char *errmsg;
	switch (TREE_CODE (type))
	  {
	  case RECORD_TYPE:
	    errmsg = "`struct %s' incomplete in scope ending here";
	    break;
	  case UNION_TYPE:
	    errmsg = "`union %s' incomplete in scope ending here";
	    break;
	  case ENUMERAL_TYPE:
	    errmsg = "`enum %s' incomplete in scope ending here";
	    break;
	  }
	if (TREE_CODE (TYPE_NAME (type)) == IDENTIFIER_NODE)
	  error (errmsg, IDENTIFIER_POINTER (TYPE_NAME (type)));
	else
	  /* If this type has a typedef-name, the TYPE_NAME is a TYPE_DECL.  */
	  error (errmsg, IDENTIFIER_POINTER (DECL_NAME (TYPE_NAME (type))));
      }
#endif /* 0 */

  /* Get the decls in the order they were written.
     Usually current_binding_level->names is in reverse order.
     But parameter decls were previously put in forward order.  */

  if (reverse)
    current_binding_level->names
      = decls = nreverse (current_binding_level->names);
  else
    decls = current_binding_level->names;

  /* If there were any declarations or structure tags in that level,
     or if this level is a function body,
     create a LET_STMT to record them for the life of this function.  */

  if (keep == 1 || functionbody > 0)
    block = build_let (0, 0, keep ? decls : 0,
                       subblocks, 0, keep ? tags : 0);

  /* In each subblock, record that this is its superior.  */

  if (keep >= 0)
    for (link = subblocks; link; link = TREE_CHAIN (link))
      STMT_SUPERCONTEXT (link) = block;

  /* Clear out the meanings of the local variables of this level;
     also record in each decl which block it belongs to.  */

  for (link = decls; link; link = TREE_CHAIN (link))
    {
      if (DECL_NAME (link) != 0)
	IDENTIFIER_LOCAL_VALUE (DECL_NAME (link)) = 0;
      DECL_CONTEXT (link) = block;
    }

  /* Restore all name-meanings of the outer levels
     that were shadowed by this level.  */

  for (link = current_binding_level->shadowed; link; link = TREE_CHAIN (link))
    IDENTIFIER_LOCAL_VALUE (TREE_PURPOSE (link)) = TREE_VALUE (link);

  /* If the level being exited is the top level of a function,
     check over all the labels.  */

  if (functionbody)
    {
      /* Clear out the definitions of all label names,
	 since their scopes end here.  */

      for (link = named_labels; link; link = TREE_CHAIN (link))
	{
	  if (DECL_SOURCE_LINE (TREE_VALUE (link)) == 0)
	    {
	      error ("label `%s' used somewhere above but not defined",
		     IDENTIFIER_POINTER (DECL_NAME (TREE_VALUE (link))));
	      /* Avoid crashing later.  */
	      define_label (input_filename, 1, DECL_NAME (TREE_VALUE (link)));
	    }
	  else if (warn_unused && !TREE_USED (TREE_VALUE (link)))
	    warning_with_decl (TREE_VALUE (link), 
			       "label `%s' defined but not used");
	  IDENTIFIER_LABEL_VALUE (DECL_NAME (TREE_VALUE (link))) = 0;
	}

      named_labels = 0;
    }

  /* Pop the current level, and free the structure for reuse.  */
  tmp = current_binding_level->temporary;
  {
    register struct binding_level *level = current_binding_level;
    current_binding_level = current_binding_level->level_chain;

    level->level_chain = free_binding_level;
    free_binding_level = level;
    if (current_binding_level->parm_flag < 0)
      {
	class_binding_level = current_binding_level;
	do
	  {
	    current_binding_level = current_binding_level->level_chain;
	  }
	while (current_binding_level->parm_flag < 0);
      }
  }

  if (functionbody > 0)
    {
      DECL_INITIAL (current_function_decl) = block;
      /* If this is the top level block of a function,
	 the vars are the function's parameters.
	 Don't leave them in the LET_STMT because they are
	 found in the FUNCTION_DECL instead.  */
      STMT_VARS (block) = 0;
    }
  else if (block)
    current_binding_level->blocks
      = chainon (current_binding_level->blocks, block);
  /* If we did not make a block for the level just exited,
     any blocks made for inner levels
     (since they cannot be recorded as subblocks in that level)
     must be carried forward so they will later become subblocks
     of something else.  */
  else if (subblocks)
    if (keep == 2)
      current_binding_level->blocks = chainon (subblocks, current_binding_level->blocks);
    else
      current_binding_level->blocks
        = chainon (current_binding_level->blocks, subblocks);

  if (tmp)
    {
#if 0
      /* We did not call push_momentary for this
	 binding contour, so there is nothing to pop.  */
      pop_momentary ();
#endif
      expand_end_bindings (getdecls (), keep, 1);
      poplevel (keep, reverse, real_functionbody);
    }
}

/* Make a note that a fixup is needed for the label BODY.  */
void
note_fixup_needed (body)
     tree body;
{
  current_binding_level->more_cleanups_ok = 0;
}

/* Do a pushlevel for class declarations.  */
void
pushlevel_class ()
{
  pushlevel (0);
  class_binding_level = current_binding_level;
  class_binding_level->parm_flag = -1;
  do
    {
      current_binding_level = current_binding_level->level_chain;
    }
  while (current_binding_level->parm_flag < 0);
}

/* ...and a poplevel for class declarations.  */
void
poplevel_class ()
{
  register struct binding_level *level = class_binding_level;
  class_binding_level = class_binding_level->level_chain;

  level->level_chain = free_binding_level;
  free_binding_level = level;
  if (class_binding_level->parm_flag >= 0)
    class_binding_level = 0;
}

/* Push a definition of struct, union or enum tag "name".
   "type" should be the type node.
   We assume that the tag "name" is not already defined.

   Note that the definition may really be just a forward reference.
   In that case, the TYPE_SIZE will be zero.

   C++ gratuitously puts all these tags in the name space. */

void
pushtag (name, type)
     tree name, type;
{
  register struct binding_level *b;

  if (class_binding_level)
    b = class_binding_level;
  else
    {
      b = current_binding_level;
      while (b->tag_transparent) b = b->level_chain;
    }

  if (name)
    {
      /* Record the identifier as the type's name if it has none.  */

      if (TYPE_NAME (type) == 0)
        TYPE_NAME (type) = name;

      b->tags = saveable_tree_cons (name, type, b->tags);

      /* do C++ gratuitous typedefing.  Note that we put the
         TYPE_DECL in the TREE_TYPE of the IDENTIFIER_NODE.  */
      if (current_lang_name == lang_name_cplusplus
          && TREE_TYPE (name) != TYPE_NAME (type))
        {
          register tree t = pushdecl (build_decl (TYPE_DECL, name, type));
          TYPE_NAME (type) = t;
          TREE_TYPE (name) = t;
        }
      if (b->parm_flag < 0)
	{
	  TREE_NONLOCAL (type) = 1;
	  IDENTIFIER_CLASS_VALUE (name) = TYPE_NAME (type);
	  CLASSTYPE_TAGS (current_class_type) = b->tags;
	}
    }
}

/* This routine is called from the last rule in yyparse ().
   Its job is to create all the code needed to initialize and
   destroy the global aggregates.  We do the destruction
   first, since that way we only need to reverse the decls once.  */

void
finish_file ()
{
  extern int lineno;
  char *real_input_filename = (main_input_filename == 0) ? (main_input_filename=input_filename) : main_input_filename;

  char *buf = (char *)alloca (sizeof (FILE_FUNCTION_FORMAT) + strlen (real_input_filename));
  char *p;
  tree fnname;
  tree vars = global_aggregates;
  int needs_cleaning = 0, needs_messing_up = 0;

  if (flag_detailed_statistics)
    dump_tree_statistics ();

  /* Bad parse errors.  Just forget about it.  */
  if (current_binding_level != global_binding_level)
    return;

  /* Set up the name of the file-level functions we may need.  */
  sprintf (buf, FILE_FUNCTION_FORMAT, real_input_filename);

  for (p = buf+11; *p; p++)
    if (! ((*p >= '0' && *p <= '9')
           || (*p >= 'A' && *p <= 'Z')
           || (*p >= 'a' && *p < 'z')))
      *p = '_';

  /* See if we really need the hassle.  */
  while (vars && needs_cleaning == 0)
    {
      tree decl = TREE_VALUE (vars);
      tree type = TREE_TYPE (decl);
      if (TYPE_NEEDS_DESTRUCTOR (type))
        {
          needs_cleaning = 1;
          needs_messing_up = 1;
          break;
        }
      else
        needs_messing_up |= TYPE_NEEDS_CONSTRUCTING (type);
      vars = TREE_CHAIN (vars);
    }
  if (needs_cleaning == 0)
    goto mess_up;

  /* Otherwise, GDB can get confused, because in only knows
     about source for LINENO-1 lines.  */
  lineno -= 1;

  fnname = get_identifier (buf);
  start_function (void_list_node, build_nt (CALL_EXPR, fnname, void_list_node), 0);
  store_parm_decls ();

  pushlevel (0);
  clear_last_expr ();
  push_momentary ();
  expand_start_bindings (0);

  /* These must be done in backward order to destroy,
     in which they happen to be!  */
  while (vars)
    {
      tree decl = TREE_VALUE (vars);
      tree type = TREE_TYPE (decl);

      if (TYPE_NEEDS_DESTRUCTOR (type))
        expand_delete (type, decl, integer_zero_node);
      vars = TREE_CHAIN (vars);
    }

  expand_end_bindings (getdecls (), 1, 0);
  poplevel (1, 0, 1);
  pop_momentary ();

  finish_function (1);

#if 0 && !defined(SDB_DEBUGGING_INFO) && !defined(FASCIST_ASSEMBLER)
  /* Now tell GNU LD that this is part of the static destructor set.  */
  {
    extern struct _iob *asm_out_file;
    fprintf (asm_out_file, ".stabs \"___DTOR_LIST__\",22,0,0,");
    assemble_name (asm_out_file, IDENTIFIER_POINTER (fnname));
    fputc ('\n', asm_out_file);
  }
#endif

  /* if it needed cleaning, then it will need messing up: drop through  */

 mess_up:
  /* Must do this while we think we are at the top level.  */
  vars = nreverse (global_aggregates);
  if (vars != NULL_TREE)
    {
      buf[FILE_FUNCTION_PREFIX_LEN] = 'I';
      fnname = get_identifier (buf);
      start_function (void_list_node, build_nt (CALL_EXPR, fnname, void_list_node), 0);
      store_parm_decls ();

      pushlevel (0);
      clear_last_expr ();
      push_momentary ();
      expand_start_bindings (0);

#ifdef SOS
      if (flag_all_virtual == 2)
        {
          tree decl;
          char c = buf[FILE_FUNCTION_PREFIX_LEN];
          buf[FILE_FUNCTION_PREFIX_LEN] = 'Z';

          decl = pushdecl (build_lang_decl (FUNCTION_DECL, get_identifier (buf), default_function_type));
          finish_decl (decl, NULL_TREE, NULL_TREE);
          expand_expr_stmt (build_function_call (decl, NULL_TREE));
          buf[FILE_FUNCTION_PREFIX_LEN] = c;
        }
#endif

      while (vars)
        {
          tree decl = TREE_VALUE (vars);
          tree init = TREE_PURPOSE (vars);

          if (TYPE_NEEDS_CONSTRUCTING (TREE_TYPE (decl)))
            {
              if (init)
                {
                  if (TREE_CODE (init) == VAR_DECL)
                    {
                      /* This behavior results when there are
                         multiple declarations of an aggregate,
                         the last of which defines it.  */
                      if (DECL_RTL (init) == DECL_RTL (decl))
                        {
                          assert (DECL_INITIAL (decl) == error_mark_node);
                          init = DECL_INITIAL (init);
                        }
                      else if (TREE_TYPE (decl) == TREE_TYPE (init))
                        {
#if 0
                          assert (0);
#else
                          /* point to real decl's rtl anyway.  */
                          DECL_RTL (init) = DECL_RTL (decl);
                          assert (DECL_INITIAL (decl) == error_mark_node);
                          init = DECL_INITIAL (init);
#endif
                        }
                    }
                }
              /* Set these global variables so that GDB at least puts
                 us near the declaration which required the initialization.  */
              input_filename = DECL_SOURCE_FILE (decl);
              lineno = DECL_SOURCE_LINE (decl);
              emit_note (input_filename, lineno);
              expand_aggr_init (decl, init, 0);
            }
          vars = TREE_CHAIN (vars);
        }

      expand_end_bindings (getdecls (), 1, 0);
      poplevel (1, 0, 1);
      pop_momentary ();

      finish_function (1);
#if 0 && !defined(SDB_DEBUGGING_INFO) && !defined(FASCIST_ASSEMBLER)
      /* Now tell GNU LD that this is part of the static constructor set.  */
      {
        extern struct _iob *asm_out_file;
        fprintf (asm_out_file, ".stabs \"___CTOR_LIST__\",22,0,0,");
        assemble_name (asm_out_file, IDENTIFIER_POINTER (fnname));
        fputc ('\n', asm_out_file);
      }
#endif
    }

#ifdef SOS
  if (flag_all_virtual == 2)
    {
      tree __sosDynError = default_conversion (lookup_name (get_identifier ("sosDynError")));

      tree null_string = build (ADDR_EXPR, string_type_node, combine_strings (build_string (0, "")));
      tree tags = gettags ();
      tree decls = getdecls ();
      tree entry;
      int i;

      entry = NULL_TREE;
      for (i = 0; i < 3; i++)
        entry = tree_cons (NULL_TREE, integer_zero_node, entry);
      zlink = build_tree_list (NULL_TREE, build (CONSTRUCTOR, zlink_type, NULL_TREE, entry));
      TREE_LITERAL (TREE_VALUE (zlink)) = 1;
      TREE_STATIC (TREE_VALUE (zlink)) = 1;

      entry = NULL_TREE;
      for (i = 0; i < 5; i++)
        entry = tree_cons (NULL_TREE, integer_zero_node, entry);
      zret = build_tree_list (NULL_TREE, build (CONSTRUCTOR, zret_type, NULL_TREE, entry));
      TREE_LITERAL (TREE_VALUE (zret)) = 1;
      TREE_STATIC (TREE_VALUE (zret)) = 1;

      /* Symbols with external visibility (except globally visible
         dynamic member functions) into the `zlink' table.  */
      while (decls)
        {
          if (TREE_PUBLIC (decls)
              && TREE_ASM_WRITTEN (decls)
              && (TREE_CODE (decls) != FUNCTION_DECL
                  || TREE_CODE (TREE_TYPE (decls)) != METHOD_TYPE
                  || TYPE_DYNAMIC (TYPE_METHOD_BASETYPE (TREE_TYPE (decls))) == 0))
            {
              entry = build (CONSTRUCTOR, zlink_type, NULL_TREE,
                        tree_cons (NULL_TREE,
                          build (ADDR_EXPR, string_type_node,
                                 combine_strings (build_string (IDENTIFIER_LENGTH (DECL_NAME (decls)),
                                                                IDENTIFIER_POINTER (DECL_NAME (decls))))),
                            tree_cons (NULL_TREE, integer_one_node,
                              build_tree_list (NULL_TREE, build_unary_op (ADDR_EXPR, decls, 0)))));
              TREE_LITERAL (entry) = 1;
              TREE_STATIC (entry) = 1;
              zlink = tree_cons (NULL_TREE, entry, zlink);
            }
          decls = TREE_CHAIN (decls);
        }

      buf[FILE_FUNCTION_PREFIX_LEN] = 'Z';
  
      fnname = get_identifier (buf);
      start_function (void_list_node, build_nt (CALL_EXPR, fnname, void_list_node), 0);
      store_parm_decls ();

      pushlevel (0);
      clear_last_expr ();
      push_momentary ();
      expand_start_bindings (0);

      {
        tree decl, type;
        /* Lay out a table static to this function
           with information about all text and data that
           this file provides.  */
        tree zlink_table = finish_table (get_identifier ("__ZLINK_tbl"), zlink_type, zlink, 0);
        decl = pushdecl (build_decl (VAR_DECL, get_identifier ("_ZLINK_once"), integer_type_node));
        TREE_STATIC (decl) = 1;
        finish_decl (decl, NULL_TREE, NULL_TREE);
        expand_start_cond (truthvalue_conversion (decl), 0);
        expand_null_return ();
        expand_end_cond ();
        finish_stmt ();
        expand_expr_stmt (build_modify_expr (decl, NOP_EXPR, integer_one_node));
        type = build_function_type (void_type_node, NULL_TREE);
        decl = pushdecl (build_lang_decl (FUNCTION_DECL, get_identifier ("_Ztable_from_cfront"), type));
        TREE_EXTERNAL (decl) = 1;
        finish_decl (decl, NULL_TREE, NULL_TREE);
        expand_expr_stmt (build_function_call (decl, build_tree_list (NULL_TREE, zlink_table)));
        expand_null_return ();
      }
      expand_end_bindings (0, 1, 0);
      poplevel (1, 0, 1);
      pop_momentary ();
      finish_function (1);

      buf[FILE_FUNCTION_PREFIX_LEN] = 'Y';
      fnname = get_identifier (buf);
      start_function (build_tree_list (NULL_TREE, get_identifier ("int")),
                      build_nt (INDIRECT_REF, build_nt (CALL_EXPR, fnname, void_list_node)), 0);
      store_parm_decls ();

      pushlevel (0);
      clear_last_expr ();
      push_momentary ();
      expand_start_bindings (0);

      {
#define SOS_VERSION 2
        tree sosVersionNumber = build_int_2 (SOS_VERSION, 0);
        tree zret_table;

        /* For each type defined, if is a dynamic type, write out
           an entry linking its member functions with their names.  */

        while (tags)
          {
            tree type = TREE_VALUE (tags);
            if (TYPE_DYNAMIC (type))
              {
                /* SOS currently only implements single inheritance,
                   so we just pick up one string, if this class
                   has a base class.  */
                tree base_name = CLASSTYPE_N_BASECLASSES (type) > 0 && TYPE_DYNAMIC (CLASSTYPE_BASECLASS (type, 1))
                  ? build (ADDR_EXPR, string_type_node, CLASSTYPE_TYPENAME_AS_STRING (CLASSTYPE_BASECLASS (type, 1)))
                    : null_string;
                tree dyn_table, dyn_entry = NULL_TREE, fns = CLASSTYPE_VIRTUALS (type);
                while (fns)
                  {
                    if (TREE_ASM_WRITTEN (TREE_OPERAND (TREE_VALUE (fns), 0)))
                      dyn_entry = tree_cons (NULL_TREE, TREE_VALUE (fns), dyn_entry);
                    else
                      dyn_entry = tree_cons (NULL_TREE, __sosDynError, dyn_entry);
                    fns = TREE_CHAIN (fns);
                  }
                dyn_entry = nreverse (dyn_entry);
                dyn_entry = tree_cons (NULL_TREE, build (NOP_EXPR, TYPE_POINTER_TO (default_function_type), sosVersionNumber),
                                       tree_cons (NULL_TREE, integer_zero_node,
                                                  tree_cons (NULL_TREE, integer_zero_node,
                                                             dyn_entry)));
                dyn_table = finish_table (DECL_NAME (TYPE_NAME (type)), TYPE_POINTER_TO (default_function_type), dyn_entry, 0);
                entry = build (CONSTRUCTOR, zret_type, NULL_TREE,
                               tree_cons (NULL_TREE, build (ADDR_EXPR, string_type_node, CLASSTYPE_TYPENAME_AS_STRING (type)),
                                          tree_cons (NULL_TREE, default_conversion (dyn_table),
                                                     tree_cons (NULL_TREE, build_int_2 (CLASSTYPE_VSIZE (type), 0),
                                                                tree_cons (NULL_TREE, base_name,
                                                                           build_tree_list (NULL_TREE, integer_zero_node))))));
                TREE_LITERAL (entry) = 1;
                TREE_STATIC (entry) = 1;
                zret = tree_cons (NULL_TREE, entry, zret);
              }
            tags = TREE_CHAIN (tags);
          }
        zret_table = finish_table (get_identifier ("__Zret"), zret_type, zret, 0);
        c_expand_return (convert (build_pointer_type (integer_type_node), default_conversion (zret_table)));
      }
      expand_end_bindings (0, 1, 0);
      poplevel (1, 0, 1);
      pop_momentary ();
      finish_function (1);

#if 0 && !defined(SDB_DEBUGGING_INFO) && !defined(FASCIST_ASSEMBLER)
      {
        extern struct _iob *asm_out_file;
        fprintf (asm_out_file, ".stabs \"___ZTOR_LIST__\",22,0,0,");
        assemble_name (asm_out_file, IDENTIFIER_POINTER (fnname));
        fputc ('\n', asm_out_file);
      }
#endif
    }
#endif

  /* Now write out any static class variables (which may have since
     learned how to be initialized).  */
  while (pending_statics)
    {
      tree decl = TREE_VALUE (pending_statics);
      if (TREE_USED (decl) == 1
	  || TREE_READONLY (decl) == 0
	  || DECL_INITIAL (decl) == 0)
	rest_of_decl_compilation (decl, DECL_RTL_NAME (decl), 1, 1);
      pending_statics = TREE_CHAIN (pending_statics);
    }

  /* Now write out inline functions which had their addresses taken
     and which were not declared virtual.  */
  while (pending_addressable_inlines)
    {
      tree decl = TREE_VALUE (pending_addressable_inlines);
      if (! TREE_ASM_WRITTEN (decl)
	  && DECL_SAVED_INSNS (decl))
        output_inline_function (decl);
      pending_addressable_inlines = TREE_CHAIN (pending_addressable_inlines);
    }

  /* Now delete from the chain of variables all virtual function tables
     which were never used during compilation.  */
  {
    tree prev;
    for (prev = 0, vars = global_binding_level->names;
         vars;
         prev = vars, vars = TREE_CHAIN (vars))
      {
        if (TREE_CODE (vars) == TYPE_DECL
            && IS_AGGR_TYPE (TREE_TYPE (vars))
            && TYPE_VIRTUAL_P (TREE_TYPE (vars)))
          {
            /* If this type has inline virtual functions, then
               write those functions out now.  */
            tree decl = CLASSTYPE_VTABLE (TREE_TYPE (vars));
            if (! TREE_EXTERNAL (decl) && TREE_USED (decl))
              {
                tree entries;

                for (entries = TREE_CHAIN (CONSTRUCTOR_ELTS (DECL_INITIAL (decl)));
                     entries; entries = TREE_CHAIN (entries))
                  {
                    tree fnaddr = FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (entries));
                    tree fn = TREE_OPERAND (fnaddr, 0);
                    if (! TREE_ASM_WRITTEN (fn)
                        && DECL_PENDING_INLINE_INFO (fn)
                        && TREE_ADDRESSABLE (fn))
                      output_inline_function (fn);
                  }
              }
          }
        if (TREE_CODE (vars) == VAR_DECL
            && DECL_VIRTUAL_P (vars)
            && ! TREE_USED (vars))
          {
            if (prev)
              TREE_CHAIN (prev) = TREE_CHAIN (vars);
            else
              global_binding_level->names = TREE_CHAIN (global_binding_level->names);
          }
      }
  }
}

/* Subroutine of duplicate_decls: return truthvalue of whether
   or not types of these decls match.  */
static int
decls_match (newdecl, olddecl)
     tree newdecl, olddecl;
{
  int types_match;

  if (TREE_CODE (newdecl) == FUNCTION_DECL && TREE_CODE (olddecl) == FUNCTION_DECL)
    {
      tree f1 = TREE_TYPE (newdecl);
      tree f2 = TREE_TYPE (olddecl);
      tree p1 = TYPE_ARG_TYPES (f1);
      tree p2 = TYPE_ARG_TYPES (f2);

      /* When we parse a static member function definition,
	 we put together a FUNCTION_DECL which thinks its type
	 is METHOD_TYPE.  Change that to FUNCTION_TYPE, and
	 proceed.  */
      if (TREE_CODE (f1) == METHOD_TYPE
	  && DECL_STATIC_FUNCTION_P (olddecl))
	{
	  p1 = TREE_CHAIN (p1);
	  f1 = build_function_type (TREE_TYPE (f1), p1);
	  TREE_TYPE (newdecl) = f1;
	  DECL_STATIC_FUNCTION_P (newdecl) = 1;
	}
      /* Here we must take care of the case where new default
	 parameters are specified.  Also, warn if an old
	 declaration becomes ambiguous because default
	 parameters may cause the two to be ambiguous.  */
      if (TREE_CODE (f1) != TREE_CODE (f2))
	{
	  if (TREE_CODE (f1) == OFFSET_TYPE)
	    compiler_error_with_decl (newdecl, "`%s' redeclared as member function");
	  else
	    compiler_error_with_decl (newdecl, "`%s' redeclared as non-member function");
	  return 0;
	}

      if (comptypes (TYPE_MAIN_VARIANT (TREE_TYPE (f1)),
		     TYPE_MAIN_VARIANT (TREE_TYPE (f2)), 1))
	{
	  types_match = compparms (p1, p2, 1);
	  /* C++: copy friendlist *before* we get smooshed.  */
	  if (DECL_FRIENDLIST (olddecl) && !DECL_FRIENDLIST (newdecl))
	    DECL_FRIENDLIST (newdecl) = DECL_FRIENDLIST (olddecl);
	}
      else types_match = 0;
    }
  else
    types_match = comptypes (TREE_TYPE (newdecl), TREE_TYPE (olddecl), 1);
  return types_match;
}

/* Handle when a new declaration NEWDECL has the same name as an old
   one OLDDECL in the same binding contour.  Prints an error message
   if appropriate.

   If safely possible, alter OLDDECL to look like NEWDECL, and return 1.
   Otherwise, return 0.  */

static int
duplicate_decls (newdecl, olddecl)
     register tree newdecl, olddecl;
{
  int types_match;

  if (TREE_CODE (olddecl) == TREE_LIST)
    {
      /* If a new decl finds a list of old decls, then
	 we assume that the new decl has C linkage, and
	 that the old decls have C++ linkage.  In this case,
	 we must look through the list to see whether
	 there is an ambiguity or not.  */
      tree olddecls = olddecl;

      /* If the overload list is empty, just install the decl.  */
      if (TREE_VALUE (olddecls) == NULL_TREE)
	{
	  TREE_VALUE (olddecls) = newdecl;
	  return 1;
	}

      while (olddecls)
	{
	  if (decls_match (newdecl, TREE_VALUE (olddecls)))
	    {
	      if (DECL_LANGUAGE (newdecl)
		  != DECL_LANGUAGE (TREE_VALUE (olddecls)))
		{
		  error_with_decl (newdecl, "declaration of `%s' with different language linkage");
		  error_with_decl (TREE_VALUE (olddecls), "previous declaration here");
		}
	      types_match = 1;
	      break;
	    }
	  olddecls = TREE_CHAIN (olddecls);
	}
      if (olddecls)
	olddecl = TREE_VALUE (olddecl);
      else
	return 1;
    }
  else
    types_match = decls_match (newdecl, olddecl);

  if (TREE_CODE (newdecl) == FUNCTION_DECL
      && TREE_CODE (olddecl) == FUNCTION_DECL)
    {
      /* Now that functions must hold information normally held
	 by field decls, there is extra work to do so that
	 declaration information does not get destroyed during
	 definition.  */
      if (DECL_VINDEX (olddecl) && ! DECL_VINDEX (newdecl))
	DECL_VINDEX (newdecl) = DECL_VINDEX (olddecl);
      if (DECL_VCONTEXT (olddecl) && ! DECL_VCONTEXT (newdecl))
	DECL_VCONTEXT (newdecl) = DECL_VCONTEXT (olddecl);
      if (DECL_FIELD_CONTEXT (olddecl) && ! DECL_FIELD_CONTEXT (newdecl))
	DECL_FIELD_CONTEXT (newdecl) = DECL_FIELD_CONTEXT (olddecl);
#ifdef SOS
      if (DECL_DINDEX (olddecl) && ! DECL_DINDEX (newdecl))
	DECL_DINDEX (newdecl) = DECL_DINDEX (newdecl);
#endif
      if (DECL_PENDING_INLINE_INFO (newdecl) == 0)
	DECL_PENDING_INLINE_INFO (newdecl) = DECL_PENDING_INLINE_INFO (olddecl);
    }

  /* If this decl has linkage, and the old one does too, maybe no error.  */
  if (TREE_CODE (olddecl) != TREE_CODE (newdecl))
    {
      error_with_decl (newdecl, "`%s' redeclared as different kind of symbol");
      error_with_decl (olddecl, "previous declaration of `%s'");
    }
  else
    {
      if (flag_traditional && TREE_CODE (newdecl) == FUNCTION_DECL
	  && IDENTIFIER_IMPLICIT_DECL (DECL_NAME (newdecl)) == olddecl)
	/* If -traditional, avoid error for redeclaring fcn
	   after implicit decl.  */
	;
      else if (TREE_CODE (olddecl) == FUNCTION_DECL
	       && DECL_FUNCTION_CODE (olddecl) != NOT_BUILT_IN)
	warning_with_decl (newdecl, "built-in function `%s' redeclared");
      else if (!types_match)
	{
	  /* Already complained about this, so don't do so again.  */
	  if (current_class_type == NULL_TREE
	      || IDENTIFIER_ERROR_LOCUS (DECL_NAME (newdecl)) != current_class_type)
	    error_with_decl (newdecl, "conflicting types for `%s'");

	  /* Check for function type mismatch
	     involving an empty arglist vs a nonempty one.  */
	  if (TREE_CODE (olddecl) == FUNCTION_DECL
	      && comptypes (TREE_TYPE (TREE_TYPE (olddecl)),
			    TREE_TYPE (TREE_TYPE (newdecl)), 1)
	      && ((TYPE_ARG_TYPES (TREE_TYPE (olddecl)) == 0
		   && DECL_INITIAL (olddecl) == 0)
		  || (TYPE_ARG_TYPES (TREE_TYPE (newdecl)) == 0
		      && DECL_INITIAL (newdecl) == 0)))
	    {
	      /* Classify the problem further.  */
	      register tree t = TYPE_ARG_TYPES (TREE_TYPE (olddecl));
	      if (t == 0)
		t = TYPE_ARG_TYPES (TREE_TYPE (newdecl));
	      for (; t; t = TREE_CHAIN (t))
		{
		  register tree type = TREE_VALUE (t);

		  if (TREE_CHAIN (t) == 0 && type != void_type_node)
		    {
		      error ("A parameter list with an ellipsis can't match");
		      error ("an empty parameter name list declaration.");
		      break;
		    }

		  if (type == float_type_node
		      || (TREE_CODE (type) == INTEGER_TYPE
			  && (TYPE_PRECISION (type)
			      < TYPE_PRECISION (integer_type_node))))
		    {
		      error ("An argument type that has a default promotion");
		      error ("can't match an empty parameter name list declaration.");
		      break;
		    }
		}
	    }
	  if (current_class_type != NULL_TREE
	      && IDENTIFIER_ERROR_LOCUS (DECL_NAME (newdecl)) != current_class_type)
	    error_with_decl (olddecl, "previous declaration of `%s'");

	  /* There is one thing GNU C++ cannot tolerate: a constructor
	     which takes the type of object being constructed.
	     Farm that case out here.  */
	  if (TREE_CODE (newdecl) == FUNCTION_DECL
	      && DECL_CONSTRUCTOR_P (newdecl))
	    {
	      tree type = TREE_TYPE (newdecl);
	      tree tmp = TREE_CHAIN (TYPE_ARG_TYPES (type));

	      if (tmp != NULL_TREE
		  && (TYPE_MAIN_VARIANT (TREE_VALUE (tmp))
		      == TYPE_METHOD_BASETYPE (type)))
		{
		  tree parm = TREE_CHAIN (DECL_ARGUMENTS (newdecl));
		  tree argtypes
		    = tree_cons (NULL_TREE,
				 build_reference_type (TREE_VALUE (tmp)),
				 TREE_CHAIN (tmp));

		  DECL_ARG_TYPE (parm)
		    = TREE_TYPE (parm)
		      = TYPE_REFERENCE_TO (TREE_VALUE (tmp));

		  TREE_TYPE (newdecl)
		    = build_method_type (TYPE_METHOD_BASETYPE (type),
					 build_function_type (TREE_TYPE (type),
							      argtypes));
		  error ("constructor cannot take as argument the type being constructed");
		  IDENTIFIER_ERROR_LOCUS (DECL_NAME (newdecl)) = current_class_type;
		}
	    }
	}
      else
	{
	  char *errmsg = redeclaration_error_message (newdecl, olddecl);
	  if (errmsg)
	    {
	      error_with_decl (newdecl, errmsg);
	      error_with_decl (olddecl,
			       "here is the previous declaration of `%s'");
	    }
	  else if (TREE_CODE (olddecl) == FUNCTION_DECL
		   && DECL_INITIAL (olddecl) != 0
		   && TYPE_ARG_TYPES (TREE_TYPE (olddecl)) == 0
		   && TYPE_ARG_TYPES (TREE_TYPE (newdecl)) != 0)
	    {
	      /* Prototype decl follows defn w/o prototype.  */
	      warning_with_decl (newdecl, "prototype for `%s'");
	      warning_with_decl (olddecl,
				 "follows non-prototype definition here");
	    }
	}
    }

  /* Deal with C++: must preserve virtual function table size.  */
  if (TREE_CODE (olddecl) == TYPE_DECL)
    {
      if (TYPE_LANG_SPECIFIC (TREE_TYPE (newdecl))
          && TYPE_LANG_SPECIFIC (TREE_TYPE (olddecl)))
        CLASSTYPE_VSIZE (TREE_TYPE (newdecl)) = CLASSTYPE_VSIZE (TREE_TYPE (olddecl));
      DECL_FRIEND_CLASSES (newdecl) = DECL_FRIEND_CLASSES (olddecl);
    }

  if (TREE_CODE (olddecl) == TREE_CODE (newdecl))
    {
      /* Copy all the DECL_... slots specified in the new decl
	 except for any that we copy here from the old type.  */

      if (types_match)
	{
	  /* Automatically handles default parameters.  */
	  tree oldtype = TREE_TYPE (olddecl);
	  /* Merge the data types specified in the two decls.  */
	  TREE_TYPE (newdecl)
	    = TREE_TYPE (olddecl)
	      = commontype (TREE_TYPE (newdecl), TREE_TYPE (olddecl));

	  /* Lay the type out, unless already done.  */
	  if (oldtype != TREE_TYPE (newdecl))
	    {
	      if (TREE_TYPE (newdecl) != error_mark_node)
		layout_type (TREE_TYPE (newdecl));
	      if (TREE_CODE (newdecl) != FUNCTION_DECL
		  && TREE_CODE (newdecl) != TYPE_DECL
		  && TREE_CODE (newdecl) != CONST_DECL)
		layout_decl (newdecl, 0);
	    }
	  else
	    {
	      /* Since the type is OLDDECL's, make OLDDECL's size go with.  */
	      DECL_SIZE (newdecl) = DECL_SIZE (olddecl);
	      DECL_SIZE_UNIT (newdecl) = DECL_SIZE_UNIT (olddecl);
	    }

	  /* Merge the initialization information.  */
	  if (DECL_INITIAL (newdecl) == 0)
	    DECL_INITIAL (newdecl) = DECL_INITIAL (olddecl);
	  /* Keep the old rtl since we can safely use it.  */
	  DECL_RTL (newdecl) = DECL_RTL (olddecl);
	}
      /* If cannot merge, then use the new type
	 and discard the old rtl.  */
      else
	TREE_TYPE (olddecl) = TREE_TYPE (newdecl);

      /* Merge the storage class information.  */
      if (TREE_EXTERNAL (newdecl))
	{
	  TREE_STATIC (newdecl) = TREE_STATIC (olddecl);
	  TREE_EXTERNAL (newdecl) = TREE_EXTERNAL (olddecl);

	  /* For functions, static overrides non-static.  */
	  if (TREE_CODE (newdecl) == FUNCTION_DECL)
	    {
	      TREE_PUBLIC (newdecl) &= TREE_PUBLIC (olddecl);
	      /* This is since we don't automatically
		 copy the attributes of NEWDECL into OLDDECL.  */
	      TREE_PUBLIC (olddecl) = TREE_PUBLIC (newdecl);
	      /* If this clears `static', clear it in the identifier too.  */
	      if (! TREE_PUBLIC (olddecl))
		TREE_PUBLIC (DECL_NAME (olddecl)) = 0;
	    }
	  else
	    TREE_PUBLIC (newdecl) = TREE_PUBLIC (olddecl);
	}
      else
	{
	  TREE_STATIC (olddecl) = TREE_STATIC (newdecl);
	  TREE_EXTERNAL (olddecl) = 0;
	  TREE_PUBLIC (olddecl) = TREE_INLINE (olddecl) ? 0 : TREE_PUBLIC (newdecl);
	}

      if (TREE_INLINE (newdecl))
        if (DECL_PENDING_INLINE_INFO (olddecl) == 0)
          {
	    /* Only makes sense to pretend it is inline if it
	       has not been written out yet.  */
#if 0
	    warning_with_decl (olddecl, "`%s' redeclared as `inline'");
#endif
	    TREE_PUBLIC (olddecl) = 0;
	    TREE_INLINE (olddecl) = 1;
	  }
	else if (TREE_PUBLIC (olddecl) == 1)
	  {
	    error_with_decl (olddecl, "`%s' redeclared with non-public visibility");
	  }

      /* If redeclaring a builtin function, and not a definition,
	 it stays built in.  */
      if (TREE_CODE (newdecl) == FUNCTION_DECL
	  && DECL_INITIAL (newdecl) == 0)
	DECL_SET_FUNCTION_CODE (newdecl, DECL_FUNCTION_CODE (olddecl));

      /* Don't lose track of having output OLDDECL as GDB symbol.  */
      DECL_BLOCK_SYMTAB_ADDRESS (newdecl)
	= DECL_BLOCK_SYMTAB_ADDRESS (olddecl);

      /* If fn was already defined, don't lose its DECL_RESULT.  */
      if (TREE_CODE (newdecl) == FUNCTION_DECL
          && DECL_RESULT (newdecl) == 0)
	DECL_RESULT (newdecl) = DECL_RESULT (olddecl);

      /* If fn was already defined, don't lose its DECL_SAVED_INSNS.  */
      if (TREE_CODE (newdecl) == FUNCTION_DECL
          && DECL_SAVED_INSNS (newdecl) == 0)
	{
	  DECL_SAVED_INSNS (newdecl) = DECL_SAVED_INSNS (olddecl);
	  DECL_RESULT_TYPE (newdecl) = DECL_RESULT_TYPE (olddecl);
	  DECL_ARGUMENTS (newdecl) = DECL_ARGUMENTS (olddecl);
	}

      TREE_ADDRESSABLE (newdecl) = TREE_ADDRESSABLE (olddecl);
      TREE_ASM_WRITTEN (newdecl) = TREE_ASM_WRITTEN (olddecl);

      /* Don't really know how much of the language-specific
         values we should copy from old to new.  */
      if (DECL_LANG_SPECIFIC (olddecl))
	DECL_IN_AGGR_P (newdecl) = DECL_IN_AGGR_P (olddecl);

      bcopy ((char *) newdecl + sizeof (struct tree_common),
	     (char *) olddecl + sizeof (struct tree_common),
	     sizeof (struct tree_decl) - sizeof (struct tree_common));

      return 1;
    }

  /* New decl is completely inconsistent with the old one =>
     tell caller to replace the old one.  */
  return 0;
}

/* Record a decl-node X as belonging to the current lexical scope.
   Check for errors (such as an incompatible declaration for the same
   name already seen in the same scope).

   Returns either X or an old decl for the same name.
   If an old decl is returned, it may have been smashed
   to agree with what X says.  */

tree
pushdecl (x)
     tree x;
{
  register tree t;
  register tree name = DECL_NAME (x);
  register struct binding_level *b = current_binding_level;

  if (name)
    {
      char *file;
      int line;

      t = lookup_name_current_level (name);
      if (t != 0 && t == error_mark_node)
	/* error_mark_node is 0 for a while during initialization!  */
	{
	  t = 0;
	  error_with_decl (x, "`%s' used prior to declaration");
	}

      if (t != 0)
	{
	  file = DECL_SOURCE_FILE (t);
	  line = DECL_SOURCE_LINE (t);
	}

      if (t != 0 && TREE_CODE (t) != TREE_CODE (x))
	{
	  if (TREE_CODE (t) == TYPE_DECL)
	    warning ("type declaration of %s shadowed",
		     IDENTIFIER_POINTER (name));
	  else if (TREE_CODE (x) == TYPE_DECL)
	    warning ("type declaration of %s shadows previous declaration",
		     IDENTIFIER_POINTER (name));
	  else if (duplicate_decls (x, t))
	    return t;
	}
      else if (t != 0 && duplicate_decls (x, t))
	{
	  /* If this decl is `static' and an `extern' was seen previously,
	     that is erroneous.  But don't complain if -traditional,
	     since traditional compilers don't complain.  */
	  if (!flag_traditional && TREE_PUBLIC (name)
	      && ! TREE_PUBLIC (x) && ! TREE_EXTERNAL (x) && ! TREE_INLINE (x))
	    {
	      if (IDENTIFIER_IMPLICIT_DECL (name))
		warning ("`%s' was declared implicitly `extern' and later `static'",
			 IDENTIFIER_POINTER (name));
	      else
		warning ("`%s' was declared `extern' and later `static'",
			 IDENTIFIER_POINTER (name));
	      warning_with_file_and_line (file, line,
					  "previous declaration of `%s'",
					  IDENTIFIER_POINTER (name));
	    }
	  return t;
	}

      /* If declaring a type as a typedef, and the type has no known
	 typedef name, install this TYPE_DECL as its typedef name.

	 C++: If it had an anonymous aggregate or enum name,
	 give it a `better' one.  */
      if (TREE_CODE (x) == TYPE_DECL)
	{
	  if (TYPE_NAME (TREE_TYPE (x)) == 0
	      || TREE_CODE (TYPE_NAME (TREE_TYPE (x))) != TYPE_DECL)
	    TYPE_NAME (TREE_TYPE (x)) = x;
	  else if (current_lang_name == lang_name_cplusplus
		   && IS_AGGR_TYPE (TREE_TYPE (x))
		   && ANON_AGGRNAME_P (DECL_NAME (TYPE_NAME (TREE_TYPE (x)))))
	    {
	      /* do gratuitous C++ typedefing, and make sure that
		 we access this type either through TREE_TYPE field
		 or via the tags list.  */
	      TYPE_NAME (TREE_TYPE (x)) = x;
	      pushtag (DECL_NAME (x), TREE_TYPE (x));
	    }
	}

      /* Multiple external decls of the same identifier ought to match.  */

      if (TREE_EXTERNAL (x) && IDENTIFIER_GLOBAL_VALUE (name) != 0
	  && (TREE_EXTERNAL (IDENTIFIER_GLOBAL_VALUE (name))
	      || TREE_PUBLIC (IDENTIFIER_GLOBAL_VALUE (name))))
	{
	  if (! comptypes (TREE_TYPE (x),
			   TREE_TYPE (IDENTIFIER_GLOBAL_VALUE (name)), 1))
	    {
	      warning_with_decl (x,
				 "type mismatch with previous external decl");
	      warning_with_decl (IDENTIFIER_GLOBAL_VALUE (name),
				 "previous external decl of `%s'");
	    }
	}

      /* In PCC-compatibility mode, extern decls of vars with no current decl
	 take effect at top level no matter where they are.  */
      if (flag_traditional && TREE_EXTERNAL (x)
	  && lookup_name (name) == 0)
	b = global_binding_level;

      /* This name is new in its binding level.
	 Install the new declaration and return it.  */
      if (b == global_binding_level)
	{
	  /* Install a global value.  */

	  /* If the first global decl has external linkage,
	     warn if we later see static one.  */
	  if (IDENTIFIER_GLOBAL_VALUE (name) == 0 && TREE_PUBLIC (x))
	    TREE_PUBLIC (name) = 1;

	  IDENTIFIER_GLOBAL_VALUE (name) = x;

	  if (IDENTIFIER_IMPLICIT_DECL (name) != 0
	      /* If this real decl matches the implicit, don't complain.  */
	      && ! (TREE_CODE (x) == FUNCTION_DECL
		    && TREE_TYPE (TREE_TYPE (x)) == integer_type_node))
	    warning ("`%s' was previously implicitly declared to return `int'",
		     IDENTIFIER_POINTER (name));

	  /* If this decl is `static' and an `extern' was seen previously,
	     that is erroneous.  */
	  if (TREE_PUBLIC (name)
	      && ! TREE_PUBLIC (x) && ! TREE_EXTERNAL (x))
	    {
	      if (IDENTIFIER_IMPLICIT_DECL (name))
		warning ("`%s' was declared implicitly `extern' and later `static'",
			 IDENTIFIER_POINTER (name));
	      else
		warning ("`%s' was declared `extern' and later `static'",
			 IDENTIFIER_POINTER (name));
	    }
	}
      else
	{
	  /* Here to install a non-global value.  */
	  tree oldlocal = IDENTIFIER_LOCAL_VALUE (name);
	  IDENTIFIER_LOCAL_VALUE (name) = x;

	  /* If this is an extern function declaration, see if we
	     have a global definition for the function.  */
	  if (oldlocal == 0
	      && IDENTIFIER_GLOBAL_VALUE (name)
	      && TREE_CODE (x) == FUNCTION_DECL
	      && TREE_CODE (IDENTIFIER_GLOBAL_VALUE (name)) == FUNCTION_DECL
	      && TREE_INLINE (IDENTIFIER_GLOBAL_VALUE (name)))
	    {
	      /* We have one.  Their types must agree.  */
	      if (! comptypes (TREE_TYPE (x),
			       TREE_TYPE (IDENTIFIER_GLOBAL_VALUE (name)), 1))
		warning_with_decl (x, "local declaration of `%s' doesn't match global one");
	      /* If the global one is inline, make the local one inline.  */
	      else if (TREE_INLINE (IDENTIFIER_GLOBAL_VALUE (name)))
		IDENTIFIER_LOCAL_VALUE (name) = IDENTIFIER_GLOBAL_VALUE (name);
	    }
	  /* If we have a local external declaration,
	     and no file-scope declaration has yet been seen,
	     then if we later have a file-scope decl it must not be static.  */
	  if (oldlocal == 0
	      && IDENTIFIER_GLOBAL_VALUE (name) == 0
	      && TREE_EXTERNAL (x))
	    {
	      TREE_PUBLIC (name) = 1;
	    }

	  /* Warn if shadowing an argument at the top level of the body.  */
	  if (oldlocal != 0 && !TREE_EXTERNAL (x)
	      && TREE_CODE (oldlocal) == PARM_DECL
	      && TREE_CODE (x) != PARM_DECL
	      && current_binding_level->level_chain->parm_flag > 0)
	    warning ("declaration of `%s' shadows a parameter",
		     IDENTIFIER_POINTER (name));

	  /* Maybe warn if shadowing something else.  */
	  else if (warn_shadow && !TREE_EXTERNAL (x))
	    {
	      char *warnstring = 0;

	      if (oldlocal != 0 && TREE_CODE (oldlocal) == PARM_DECL)
		warnstring = "declaration of `%s' shadows a parameter";
	      else if (IDENTIFIER_CLASS_VALUE (name) != 0)
		warnstring = "declaration of `%s' shadows a member of `this'";
	      else if (oldlocal != 0)
		warnstring = "declaration of `%s' shadows previous local";
	      else if (IDENTIFIER_GLOBAL_VALUE (name) != 0)
		warnstring = "declaration of `%s' shadows global declaration";

	      if (warnstring)
		warning (warnstring, IDENTIFIER_POINTER (name));
	    }

	  /* If storing a local value, there may already be one (inherited).
	     If so, record it for restoration when this binding level ends.  */
	  if (oldlocal != 0)
	    b->shadowed = tree_cons (name, oldlocal, b->shadowed);
	}

      /* Keep count of variables in this level with incomplete type.  */
      if (TREE_CODE (x) != TYPE_DECL && TYPE_SIZE (TREE_TYPE (x)) == 0)
	++b->n_incomplete;
    }

  /* Put decls on list in reverse order.
     We will reverse them later if necessary.  */
  TREE_CHAIN (x) = b->names;
  b->names = x;

  return x;
}

/* Like pushdecl, only it places X in GLOBAL_BINDING_LEVEL,
   if appropriate.  */
tree
pushdecl_top_level (x)
     tree x;
{
  register tree t;
  register tree name = DECL_NAME (x);
  register struct binding_level *b = current_binding_level;

  current_binding_level = global_binding_level;
  t = pushdecl (x);
  current_binding_level = b;
  return t;
}

/* DECL is a FUNCTION_DECL which may have other definitions already in place.
   We get around this by making IDENTIFIER_GLOBAL_VALUE (DECL_ORIGINAL_NAME (DECL))
   point to a list of all the things that want to be referenced by that name.
   It is then up to the users of that name to decide what to do with that
   list.  */
void
push_overloaded_decl (decl)
     tree decl;
{
  tree name = DECL_ORIGINAL_NAME (decl);
  tree glob = IDENTIFIER_GLOBAL_VALUE (name);

  TREE_OVERLOADED (decl) = 1;
  if (glob && TREE_CODE (glob) != TREE_LIST)
    {
      if (DECL_LANGUAGE (decl) == lang_c)
	{
	  if (TREE_CODE (glob) == FUNCTION_DECL)
	    {
	      if (DECL_LANGUAGE (glob) == lang_c)
		{
		  error_with_decl (decl, "C-language function `%s' overloaded here");
		  error_with_decl (glob, "Previous C-language version of this function was `%s'");
		}
	    }
	  else abort ();
	}
      if (! flag_traditional
	  && TREE_PERMANENT (glob) == 1
	  && current_binding_level != global_binding_level)
	overloads_to_forget = tree_cons (name, glob, overloads_to_forget);
      glob = tree_cons (DECL_NAME (glob), glob, NULL_TREE);
      glob = tree_cons (TREE_PURPOSE (glob), decl, glob);
    }
  else if (glob && TREE_VALUE (glob) == NULL_TREE)
    {
      TREE_VALUE (glob) = decl;
      return;
    }
  else if (value_member (decl, glob) == 0)
    {
      if (DECL_LANGUAGE (decl) == lang_c)
	{
	  tree decls = glob;
	  while (decls && DECL_LANGUAGE (TREE_VALUE (decls)) == lang_cplusplus)
	    decls = TREE_CHAIN (decls);
	  if (decls)
	    {
	      error_with_decl (decl, "C-language function `%s' overloaded here");
	      error_with_decl (TREE_VALUE (decls), "Previous C-language version of this function was `%s'");
	    }
	}
      if (! flag_traditional
	  && (glob == 0 || TREE_PERMANENT (glob) == 1)
	  && current_binding_level != global_binding_level)
	overloads_to_forget = tree_cons (name, glob, overloads_to_forget);
      glob = tree_cons (name, decl, glob);
    }
  else
    return;

  IDENTIFIER_GLOBAL_VALUE (name) = glob;
  TREE_TYPE (glob) = unknown_type_node;
}

/* Generate an implicit declaration for identifier FUNCTIONID
   as a function of type int ().  Print a warning if appropriate.  */

tree
implicitly_declare (functionid)
     tree functionid;
{
  register tree decl;
  int temp = allocation_temporary_p ();

  /* Save the decl permanently so we can warn if definition follows.  */
  if (temp && (flag_traditional || !warn_implicit))
    end_temporary_allocation ();

  /* We used to reuse an old implicit decl here,
     but this loses with inline functions because it can clobber
     the saved decl chains.  */
/*  if (IDENTIFIER_IMPLICIT_DECL (functionid) != 0)
    decl = IDENTIFIER_IMPLICIT_DECL (functionid);
  else  */
    decl = build_lang_decl (FUNCTION_DECL, functionid, default_function_type);

  TREE_EXTERNAL (decl) = 1;
  TREE_PUBLIC (decl) = 1;

  /* ANSI standard says implicit declarations are in the innermost block.
     So we record the decl in the standard fashion.
     If flag_traditional is set, pushdecl does it top-level.  */
  pushdecl (decl);
  rest_of_decl_compilation (decl, 0, 0, 0);

  if (warn_implicit
      /* Only one warning per identifier.  */
      && IDENTIFIER_IMPLICIT_DECL (functionid) == 0)
    warning ("implicit declaration of function `%s'",
	     IDENTIFIER_POINTER (functionid));

  IDENTIFIER_IMPLICIT_DECL (functionid) = decl;

  if (temp && (flag_traditional || ! warn_implicit))
    resume_temporary_allocation ();

  return decl;
}

/* Return zero if the declaration NEW is valid
   when the declaration OLD (assumed to be for the same name)
   has already been seen.
   Otherwise return an error message format string with a %s
   where the identifier should go.  */

static char *
redeclaration_error_message (newdecl, olddecl)
     tree newdecl, olddecl;
{
  if (TREE_CODE (newdecl) == TYPE_DECL)
    {
      /* Because C++ can put things into name space for free,
	 constructs like "typedef struct foo { ... } foo"
	 would look like an erroneous redeclaration.  */
      if (TREE_TYPE (olddecl) == TREE_TYPE (newdecl))
	return 0;
      else
	return "redefinition of `%s'";
    }
  else if (TREE_CODE (newdecl) == FUNCTION_DECL)
    {
      /* Declarations of functions can insist on internal linkage
	 but they can't be inconsistent with internal linkage,
	 so there can be no error on that account.
	 However defining the same name twice is no good.  */
      if (DECL_INITIAL (olddecl) != 0 && DECL_INITIAL (newdecl) != 0)
	return "redefinition of `%s'";
      return 0;
    }
  else if (current_binding_level == global_binding_level)
    {
      /* Objects declared at top level:  */
      /* If at least one is a reference, it's ok.  */
      if (TREE_EXTERNAL (newdecl) || TREE_EXTERNAL (olddecl))
	return 0;
      /* Reject two definitions.  */
      if (DECL_INITIAL (olddecl) != 0 && DECL_INITIAL (newdecl) != 0)
	return "redefinition of `%s'";
      /* Now we have two tentative defs, or one tentative and one real def.  */
      /* Insist that the linkage match.  */
      if (TREE_PUBLIC (olddecl) != TREE_PUBLIC (newdecl))
	return "conflicting declarations of `%s'";
      return 0;
    }
  else
    {
      /* Objects declared with block scope:  */
      /* Reject two definitions, and reject a definition
	 together with an external reference.  */
      if (!(TREE_EXTERNAL (newdecl) && TREE_EXTERNAL (olddecl)))
	return "redeclaration of `%s'";
      return 0;
    }
}

/* Get the LABEL_DECL corresponding to identifier ID as a label.
   Create one if none exists so far for the current function.
   This function is called for both label definitions and label references.  */

tree
lookup_label (id)
     tree id;
{
  register tree decl = IDENTIFIER_LABEL_VALUE (id);

  if (decl != 0)
    return decl;

  /* By giving the label type `void *', we can use it as a value.  */
  decl = build_decl (LABEL_DECL, id, ptr_type_node);
  DECL_MODE (decl) = VOIDmode;
  /* Mark that the label's definition has not been seen.  */
  DECL_SOURCE_LINE (decl) = 0;

  IDENTIFIER_LABEL_VALUE (id) = decl;

  named_labels
    = tree_cons (NULL_TREE, decl, named_labels);

  return decl;
}

/* Define a label, specifying the location in the source file.
   Return the LABEL_DECL node for the label, if the definition is valid.
   Otherwise return 0.  */

tree
define_label (filename, line, name)
     char *filename;
     int line;
     tree name;
{
  tree decl = lookup_label (name);
  if (DECL_SOURCE_LINE (decl) != 0)
    {
      error_with_decl (decl, "duplicate label `%s'");
      return 0;
    }
  else
    {
      /* Mark label as having been defined.  */
      DECL_SOURCE_FILE (decl) = filename;
      DECL_SOURCE_LINE (decl) = line;
      return decl;
    }
}

/* Return the list of declarations of the current level.
   Note that this list is in reverse order unless/until
   you nreverse it; and when you do nreverse it, you must
   store the result back using `storedecls' or you will lose.  */

tree
getdecls ()
{
  return current_binding_level->names;
}

/* Return the list of type-tags (for structs, etc) of the current level.  */

tree
gettags ()
{
  return current_binding_level->tags;
}

/* Store the list of declarations of the current level.
   This is done for the parameter declarations of a function being defined,
   after they are modified in the light of any missing parameters.  */

static void
storedecls (decls)
     tree decls;
{
  current_binding_level->names = decls;
}

/* Similarly, store the list of tags of the current level.  */

static void
storetags (tags)
     tree tags;
{
  current_binding_level->tags = tags;
}

/* Given NAME, an IDENTIFIER_NODE,
   return the structure (or union or enum) definition for that name.
   Searches binding levels from BINDING_LEVEL up to the global level.
   If THISLEVEL_ONLY is nonzero, searches only the specified context
   (but skips any tag-transparent contexts to find one that is
   meaningful for tags).
   FORM says which kind of type the caller wants;
   it is RECORD_TYPE or UNION_TYPE or ENUMERAL_TYPE.
   If the wrong kind of type is found, an error is reported.  */

static tree
lookup_tag (form, name, binding_level, thislevel_only)
     enum tree_code form;
     struct binding_level *binding_level;
     tree name;
     int thislevel_only;
{
  register struct binding_level *level;

  for (level = binding_level; level; level = level->level_chain)
    {
      register tree tail;
      for (tail = level->tags; tail; tail = TREE_CHAIN (tail))
	{
	  if (TREE_PURPOSE (tail) == name)
	    {
	      if (TREE_CODE (TREE_VALUE (tail)) != form)
		{
		  /* Definition isn't the kind we were looking for.  */
		  error ("`%s' defined as wrong kind of tag",
			 IDENTIFIER_POINTER (name));
		}
	      return TREE_VALUE (tail);
	    }
	}
      if (thislevel_only && ! level->tag_transparent)
	return NULL_TREE;
      if (current_class_type && level->level_chain == global_binding_level)
	{
	  /* Try looking in this class's tags before heading into
	     global binding level.  */
	  tree these_tags = CLASSTYPE_TAGS (current_class_type);
	  while (these_tags)
	    {
	      if (TREE_PURPOSE (these_tags) == name)
		{
		  if (TREE_CODE (TREE_VALUE (these_tags)) != form)
		    {
		      error ("`%s' defined as wrong kind of tag in class scope",
			     IDENTIFIER_POINTER (name));
		    }
		  return TREE_VALUE (tail);
		}
	      these_tags = TREE_CHAIN (these_tags);
	    }
	}
    }
  return NULL_TREE;
}

/* Given a type, find the tag that was defined for it and return the tag name.
   Otherwise return 0.  However, the value can never be 0
   in the cases in which this is used.

   C++: If NAME is non-zero, this is the new name to install.  This is
   done when replacing anonymous tags with real tag names.  */

static tree
lookup_tag_reverse (type, name)
     tree type;
     tree name;
{
  register struct binding_level *level;

  for (level = current_binding_level; level; level = level->level_chain)
    {
      register tree tail;
      for (tail = level->tags; tail; tail = TREE_CHAIN (tail))
	{
	  if (TREE_VALUE (tail) == type)
	    {
	      if (name)
		TREE_PURPOSE (tail) = name;
	      return TREE_PURPOSE (tail);
	    }
	}
    }
  return NULL_TREE;
}

/* Look up NAME in the current binding level and its superiors
   in the namespace of variables, functions and typedefs.
   Return a ..._DECL node of some kind representing its definition,
   or return 0 if it is undefined.  */

tree
lookup_name (name)
     tree name;
{
  register tree val;
  if (current_binding_level != global_binding_level
      && IDENTIFIER_LOCAL_VALUE (name))
    val = IDENTIFIER_LOCAL_VALUE (name);
  /* In C++ class fields are between local and global scope,
     just before the global scope.  */
  else if (current_class_type && IDENTIFIER_CLASS_VALUE (name))
    val = IDENTIFIER_CLASS_VALUE (name);
  else
    val = IDENTIFIER_GLOBAL_VALUE (name);
  if (val && TREE_TYPE (val) == error_mark_node)
    return error_mark_node;
  return val;
}

/* Similar to `lookup_name' but look only at current binding level.  */

static tree
lookup_name_current_level (name)
     tree name;
{
  register tree t;

  if (current_binding_level == global_binding_level)
    return IDENTIFIER_GLOBAL_VALUE (name);

  if (IDENTIFIER_LOCAL_VALUE (name) == 0)
    return 0;

  for (t = current_binding_level->names; t; t = TREE_CHAIN (t))
    if (DECL_NAME (t) == name)
      break;

  return t;
}

/* Create the predefined scalar types of C,
   and some nodes representing standard constants (0, 1, (void *)0).
   Initialize the global binding level.
   Make definitions for built-in primitive functions.  */

void
init_decl_processing ()
{
  register tree endlink;

  current_function_decl = NULL;
  named_labels = NULL;
  current_binding_level = NULL_BINDING_LEVEL;
  free_binding_level = NULL_BINDING_LEVEL;

  /* Must lay these out before anything else gets laid out.  */
  error_mark_node = make_node (ERROR_MARK);
  TREE_TYPE (error_mark_node) = error_mark_node;
  error_mark_list = build_tree_list (error_mark_node, error_mark_node);
  TREE_TYPE (error_mark_list) = error_mark_node;

  pushlevel (0);        /* make the binding_level structure for global names.  */
  global_binding_level = current_binding_level;

  value_identifier = get_identifier ("<value>");

  /* Define `int' and `char' first so that dbx will output them first.  */

  integer_type_node = make_signed_type (INT_TYPE_SIZE);
  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_INT])
    = pushdecl (build_decl (TYPE_DECL, ridpointers[(int) RID_INT],
			    integer_type_node));

  /* Define `char', which is like either `signed char' or `unsigned char'
     but not the same as either.  */

  char_type_node =
    (flag_signed_char
     ? make_signed_type (CHAR_TYPE_SIZE)
     : make_unsigned_type (CHAR_TYPE_SIZE));

  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_CHAR])
    = pushdecl (build_decl (TYPE_DECL, get_identifier ("char"),
			    char_type_node));

  long_integer_type_node = make_signed_type (LONG_TYPE_SIZE);
  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_LONG])
    = pushdecl (build_decl (TYPE_DECL, get_identifier ("long int"),
			    long_integer_type_node));

  unsigned_type_node = make_unsigned_type (INT_TYPE_SIZE);
  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_UNSIGNED])
    = pushdecl (build_decl (TYPE_DECL, get_identifier ("unsigned int"),
			    unsigned_type_node));

  long_unsigned_type_node = make_unsigned_type (LONG_TYPE_SIZE);
  pushdecl (build_decl (TYPE_DECL, get_identifier ("long unsigned int"),
			long_unsigned_type_node));

  /* `unsigned long' or `unsigned int' is the standard type for sizeof.
     Traditionally, use a signed type.  */
  if (INT_TYPE_SIZE != BITS_PER_WORD)
    sizetype = flag_traditional ? long_integer_type_node : long_unsigned_type_node;
  else
    sizetype = flag_traditional ? integer_type_node : unsigned_type_node;

  TREE_TYPE (TYPE_SIZE (integer_type_node)) = sizetype;
  TREE_TYPE (TYPE_SIZE (char_type_node)) = sizetype;
  TREE_TYPE (TYPE_SIZE (unsigned_type_node)) = sizetype;
  TREE_TYPE (TYPE_SIZE (long_unsigned_type_node)) = sizetype;
  TREE_TYPE (TYPE_SIZE (long_integer_type_node)) = sizetype;

  short_integer_type_node = make_signed_type (SHORT_TYPE_SIZE);
  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_SHORT])
    = pushdecl (build_decl (TYPE_DECL, get_identifier ("short int"),
			    short_integer_type_node));

  long_long_integer_type_node = make_signed_type (LONG_LONG_TYPE_SIZE);
  pushdecl (build_decl (TYPE_DECL, get_identifier ("long long int"),
			long_long_integer_type_node));

  short_unsigned_type_node = make_unsigned_type (SHORT_TYPE_SIZE);
  pushdecl (build_decl (TYPE_DECL, get_identifier ("short unsigned int"),
			short_unsigned_type_node));

  long_long_unsigned_type_node = make_unsigned_type (LONG_LONG_TYPE_SIZE);
  pushdecl (build_decl (TYPE_DECL, get_identifier ("long long unsigned int"),
			long_long_unsigned_type_node));

  /* Define both `signed char' and `unsigned char'.  */
  signed_char_type_node = make_signed_type (CHAR_TYPE_SIZE);
  pushdecl (build_decl (TYPE_DECL, get_identifier ("signed char"),
			signed_char_type_node));

  unsigned_char_type_node = make_unsigned_type (CHAR_TYPE_SIZE);
  pushdecl (build_decl (TYPE_DECL, get_identifier ("unsigned char"),
			unsigned_char_type_node));

  float_type_node = make_node (REAL_TYPE);
  TYPE_PRECISION (float_type_node) = FLOAT_TYPE_SIZE;
  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_FLOAT])
    = pushdecl (build_decl (TYPE_DECL, ridpointers[(int) RID_FLOAT],
			    float_type_node));
  layout_type (float_type_node);

  double_type_node = make_node (REAL_TYPE);
  TYPE_PRECISION (double_type_node) = DOUBLE_TYPE_SIZE;
  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_DOUBLE])
    = pushdecl (build_decl (TYPE_DECL, ridpointers[(int) RID_DOUBLE],
			    double_type_node));
  layout_type (double_type_node);

  long_double_type_node = make_node (REAL_TYPE);
  TYPE_PRECISION (long_double_type_node) = DOUBLE_TYPE_SIZE;
  pushdecl (build_decl (TYPE_DECL, get_identifier ("long double"),
			long_double_type_node));
  layout_type (long_double_type_node);

  integer_zero_node = build_int_2 (0, 0);
  TREE_TYPE (integer_zero_node) = integer_type_node;
  integer_one_node = build_int_2 (1, 0);
  TREE_TYPE (integer_one_node) = integer_type_node;

  size_zero_node = build_int_2 (0, 0);
  TREE_TYPE (size_zero_node) = sizetype;
  size_one_node = build_int_2 (1, 0);
  TREE_TYPE (size_one_node) = sizetype;

  void_type_node = make_node (VOID_TYPE);
  IDENTIFIER_GLOBAL_VALUE (ridpointers[(int) RID_VOID])
    = pushdecl (build_decl (TYPE_DECL,
			    ridpointers[(int) RID_VOID], void_type_node));
  layout_type (void_type_node); /* Uses integer_zero_node.  */
  void_list_node = build_tree_list (NULL_TREE, void_type_node);

  null_pointer_node = build_int_2 (0, 0);
  TREE_TYPE (null_pointer_node) = build_pointer_type (void_type_node);
  layout_type (TREE_TYPE (null_pointer_node));

  string_type_node = build_pointer_type (char_type_node);

  /* make a type for arrays of 256 characters.
     256 is picked randomly because we have a type for integers from 0 to 255.
     With luck nothing will ever really depend on the length of this
     array type.  */
  char_array_type_node
    = build_array_type (char_type_node, unsigned_char_type_node);
  /* Likewise for arrays of ints.  */
  int_array_type_node
    = build_array_type (integer_type_node, unsigned_char_type_node);

  default_function_type
    = build_function_type (integer_type_node, NULL_TREE);
  TYPE_POINTER_TO (default_function_type);

  ptr_type_node = build_pointer_type (void_type_node);
  endlink = void_list_node;

  double_ftype_double
    = build_function_type (double_type_node,
			   tree_cons (NULL_TREE, double_type_node, endlink));

  double_ftype_double_double
    = build_function_type (double_type_node,
			   tree_cons (NULL_TREE, double_type_node,
				      tree_cons (NULL_TREE,
						 double_type_node, endlink)));

  int_ftype_int
    = build_function_type (integer_type_node,
			   tree_cons (NULL_TREE, integer_type_node, endlink));

  long_ftype_long
    = build_function_type (long_integer_type_node,
			   tree_cons (NULL_TREE,
				      long_integer_type_node, endlink));

  void_ftype_ptr_ptr_int
    = build_function_type (void_type_node,
			   tree_cons (NULL_TREE, ptr_type_node,
				      tree_cons (NULL_TREE, ptr_type_node,
                                                 tree_cons (NULL_TREE,
							    integer_type_node,
							    endlink))));

  int_ftype_ptr_ptr_int
    = build_function_type (integer_type_node,
			   tree_cons (NULL_TREE, ptr_type_node,
				      tree_cons (NULL_TREE, ptr_type_node,
						 tree_cons (NULL_TREE,
							    integer_type_node,
							    endlink))));

  void_ftype_ptr_int_int
    = build_function_type (void_type_node,
			   tree_cons (NULL_TREE, ptr_type_node,
				      tree_cons (NULL_TREE, integer_type_node,
						 tree_cons (NULL_TREE,
							    integer_type_node,
							    endlink))));

  ptr_ftype_long
    = build_function_type (ptr_type_node,
			   tree_cons (NULL_TREE,
				      long_integer_type_node,
				      endlink));

  ptr_ftype_ptr_int_int_ptr
    = build_function_type (ptr_type_node,
	   tree_cons (NULL_TREE, ptr_type_node,
	      tree_cons (NULL_TREE, integer_type_node,
		 tree_cons (NULL_TREE, integer_type_node,
		    tree_cons (NULL_TREE, ptr_type_node,
		       endlink)))));

  void_ftype_ptr
    = build_function_type (void_type_node,
			   tree_cons (NULL_TREE,
				      ptr_type_node,
				      endlink));
  void_ftype_ptr_int_int_ptr_int_int
    = build_function_type (void_type_node,
	   tree_cons (NULL_TREE, ptr_type_node,
	      tree_cons (NULL_TREE, integer_type_node,
		 tree_cons (NULL_TREE, integer_type_node,
		    tree_cons (NULL_TREE, ptr_type_node,
		       tree_cons (NULL_TREE, integer_type_node,
			  tree_cons (NULL_TREE, integer_type_node,
			     endlink)))))));

#ifdef VTABLE_USES_MASK
  /* This is primarily for virtual function definition.  We
     declare an array of `void *', which can later be
     converted to the appropriate function pointer type.
     To do pointers to members, we need a mask which can
     distinguish an index value into a virtual function table
     from an address.  */
  vtbl_mask = build_int_2 (~(VINDEX_MAX - 1), -1);
#endif

  vtbl_type_node
    = build_array_type (ptr_type_node, NULL_TREE);
  layout_type (vtbl_type_node);
  vtbl_type_node = build_type_variant (vtbl_type_node, 1, 0);

  builtin_function ("__builtin_alloca",
                    build_function_type (ptr_type_node,
                                         tree_cons (NULL_TREE,
                                                    integer_type_node,
                                                    endlink)),
                    BUILT_IN_ALLOCA);

  builtin_function ("__builtin_abs", int_ftype_int, BUILT_IN_ABS);
  builtin_function ("__builtin_fabs", double_ftype_double, BUILT_IN_FABS);
  builtin_function ("__builtin_labs", long_ftype_long, BUILT_IN_LABS);
  builtin_function ("__builtin_ffs", int_ftype_int, BUILT_IN_FFS);
#if 0
  /* Support for these has not been written in either expand_builtin
     or build_function_call.  */
  builtin_function ("__builtin_div", default_ftype, BUILT_IN_DIV);
  builtin_function ("__builtin_ldiv", default_ftype, BUILT_IN_LDIV);
  builtin_function ("__builtin_ffloor", double_ftype_double, BUILT_IN_FFLOOR);
  builtin_function ("__builtin_fceil", double_ftype_double, BUILT_IN_FCEIL);
  builtin_function ("__builtin_fmod", double_ftype_double_double, BUILT_IN_FMOD);
  builtin_function ("__builtin_frem", double_ftype_double_double, BUILT_IN_FREM);
  builtin_function ("__builtin_memcpy", void_ftype_ptr_ptr_int, BUILT_IN_MEMCPY);
  builtin_function ("__builtin_memcmp", int_ftype_ptr_ptr_int, BUILT_IN_MEMCMP);
  builtin_function ("__builtin_memset", void_ftype_ptr_int_int, BUILT_IN_MEMSET);
  builtin_function ("__builtin_fsqrt", double_ftype_double, BUILT_IN_FSQRT);
  builtin_function ("__builtin_getexp", double_ftype_double, BUILT_IN_GETEXP);
  builtin_function ("__builtin_getman", double_ftype_double, BUILT_IN_GETMAN);
#endif

  /* C++ extensions */

  unknown_type_node = make_node (UNKNOWN_TYPE);
  pushdecl (build_decl (TYPE_DECL,
			get_identifier ("unknown type"),
			unknown_type_node));
  TYPE_SIZE (unknown_type_node) = TYPE_SIZE (void_type_node);
  TYPE_SIZE_UNIT (unknown_type_node) = TYPE_SIZE_UNIT (void_type_node);
  TYPE_ALIGN (unknown_type_node) = 1;
  TYPE_MODE (unknown_type_node) = TYPE_MODE (void_type_node);
  /* Indirecting an UNKNOWN_TYPE node yields an UNKNOWN_TYPE node.  */
  TREE_TYPE (unknown_type_node) = unknown_type_node;
  /* Looking up TYPE_POINTER_TO and TYPE_REFERENCE_TO yield the same result.  */
  TYPE_POINTER_TO (unknown_type_node) = unknown_type_node;
  TYPE_REFERENCE_TO (unknown_type_node) = unknown_type_node;

  /* Define these now, but use 0 as their DECL_FUNCTION_CODE.  This
     will install them in the global binding level, but cause them
     to be expanded normally.  */
  builtin_function ("__builtin_new", ptr_ftype_long, NOT_BUILT_IN);
  builtin_function ("__builtin_vec_new", ptr_ftype_ptr_int_int_ptr, NOT_BUILT_IN);
  builtin_function ("__builtin_delete", void_ftype_ptr, NOT_BUILT_IN);
  builtin_function ("__builtin_vec_delete", void_ftype_ptr_int_int_ptr_int_int, BUILT_IN_VEC_DELETE);
  builtin_function ("__builtin_saveregs", default_function_type, NOT_BUILT_IN);

  {
    /* Simplify life by making a "vtable_entry_type".  Give its
       fields names so that the debugger can use them.  */
    tree fields[4];
    int i;

    vtable_entry_type = make_lang_type (RECORD_TYPE);
    CLASSTYPE_OFFSET (vtable_entry_type) = integer_zero_node;
    fields[0] = build_lang_decl (FIELD_DECL, get_identifier (VTABLE_DELTA_NAME), short_integer_type_node);
    fields[1] = build_lang_decl (FIELD_DECL, get_identifier (VTABLE_INDEX_NAME), short_integer_type_node);
    fields[2] = build_lang_decl (FIELD_DECL, get_identifier (VTABLE_PFN_NAME), ptr_type_node);
    TYPE_FIELDS (vtable_entry_type) = fields[0];
    for (i = 0; i < 2; i++)
      {
        DECL_FIELD_CONTEXT (fields[i]) = vtable_entry_type;
        TREE_CHAIN (fields[i]) = fields[i+1];
      }
    DECL_FIELD_CONTEXT (fields[i]) = vtable_entry_type;
    TYPE_ALIGN (vtable_entry_type) = TYPE_ALIGN (double_type_node);
    layout_type (vtable_entry_type);
    TYPE_NAME (vtable_entry_type) = build_decl (TYPE_DECL,
						get_identifier ("$vtbl_ptr_type"),
						vtable_entry_type);
    layout_decl (TYPE_NAME (vtable_entry_type));

    /* Make this part of an invisible union.  */
    fields[3] = copy_node (fields[2]);
    TREE_TYPE (fields[3]) = short_integer_type_node;
    DECL_NAME (fields[3]) = get_identifier (VTABLE_DELTA2_NAME);
    DECL_MODE (fields[3]) = TYPE_MODE (short_integer_type_node);
    DECL_SIZE (fields[3]) = TYPE_SIZE (short_integer_type_node);
    DECL_SIZE_UNIT (fields[3]) = TYPE_SIZE_UNIT (short_integer_type_node);
    TREE_UNSIGNED (fields[3]) = 0;
    TREE_CHAIN (fields[2]) = fields[3];

    vtable_entry_type = build_type_variant (vtable_entry_type, 1, 0);
  }

#ifdef SOS
  if (flag_all_virtual == 2)
    {
      tree fields[5];
      tree ptr_ftype_default
        = build_function_type (ptr_type_node, NULL_TREE);
      int i;

      builtin_function ("sosFindCode", ptr_ftype_default, NOT_BUILT_IN);
      builtin_function ("sosLookup", ptr_ftype_default, NOT_BUILT_IN);
      builtin_function ("sosImport", ptr_ftype_default, NOT_BUILT_IN);
      builtin_function ("sosDynError", ptr_ftype_default, NOT_BUILT_IN);

      zlink_type = make_lang_type (RECORD_TYPE);
      CLASSTYPE_OFFSET (zlink_type) = integer_zero_node;
      fields[0] = build_lang_decl (FIELD_DECL, get_identifier ("n"), string_type_node);
      fields[1] = build_lang_decl (FIELD_DECL, get_identifier ("t"), char_type_node);
      fields[2] = build_lang_decl (FIELD_DECL, get_identifier ("ptr"), TYPE_POINTER_TO (default_function_type));

      TYPE_FIELDS (zlink_type) = fields[0];
      for (i = 0; i < 2; i++)
        {
	  DECL_FIELD_CONTEXT (fields[i]) = zlink_type;
	  TREE_CHAIN (fields[i]) = fields[i+1];
        }
      DECL_FIELD_CONTEXT (fields[i]) = zlink_type;
      TYPE_ALIGN (zlink_type) = 1;
      layout_type (zlink_type);

      zret_type = make_lang_type (RECORD_TYPE);
      CLASSTYPE_OFFSET (zret_type) = integer_zero_node;
      fields[0] = build_lang_decl (FIELD_DECL, get_identifier ("cn"), string_type_node);
      fields[1] = build_lang_decl (FIELD_DECL, get_identifier ("ptr"), build_pointer_type (TYPE_POINTER_TO (default_function_type)));
      fields[2] = build_lang_decl (FIELD_DECL, get_identifier ("n"), integer_type_node);
      fields[3] = build_lang_decl (FIELD_DECL, get_identifier ("bcl"), string_type_node);
      fields[4] = build_lang_decl (FIELD_DECL, get_identifier ("f"), char_type_node);

      TYPE_FIELDS (zret_type) = fields[0];
      for (i = 0; i < 4; i++)
        {
	  TREE_CHAIN (fields[i]) = fields[i+1];
	  DECL_FIELD_CONTEXT (fields[i]) = zret_type;
        }
      DECL_FIELD_CONTEXT (fields[i]) = zret_type;
      TYPE_ALIGN (zret_type) = 1;
      layout_type (zret_type);
    }
#endif

  /* Now do other language dependent initializations.  */
  init_class_processing ();
  init_init_processing ();
  init_search_processing ();
}

/* Make a definition for a builtin function named NAME and whose data type
   is TYPE.  TYPE should be a function type with argument types.
   FUNCTION_CODE tells later passes how to compile calls to this function.
   See tree.h for its possible values.  */

static void
builtin_function (name, type, function_code)
     char *name;
     tree type;
     enum built_in_function function_code;
{
  tree decl = build_lang_decl (FUNCTION_DECL, get_identifier (name), type);
  TREE_EXTERNAL (decl) = 1;
  TREE_PUBLIC (decl) = 1;
  make_function_rtl (decl);
  pushdecl (decl);
  DECL_SET_FUNCTION_CODE (decl, function_code);
}

/* Called when a declaration is seen that contains no names to declare.
   If its type is a reference to a structure, union or enum inherited
   from a containing scope, shadow that tag name for the current scope
   with a forward reference.
   If its type defines a new named structure or union
   or defines an enum, it is valid but we need not do anything here.
   Otherwise, it is an error.

   C++: may have to grok the declspecs to learn about static,
   complain for anonymous unions.  */

void
shadow_tag (declspecs)
     tree declspecs;
{
  int found_tag = 0;
  int warned = 0;
  register tree link;
  register enum tree_code code;
  register tree t = NULL_TREE;

  for (link = declspecs; link; link = TREE_CHAIN (link))
    {
      register tree value = TREE_VALUE (link);
      int ok = 0;

      code = TREE_CODE (value);
      if (IS_AGGR_TYPE_CODE (code) || code == ENUMERAL_TYPE)
	/* Used to test also that TYPE_SIZE (value) != 0.
	   That caused warning for `struct foo;' at top level in the file.  */
	{
	  register tree name = TYPE_NAME (value);

	  if (name == NULL_TREE)
	    name = lookup_tag_reverse (value, NULL_TREE);

	  if (name && TREE_CODE (name) == TYPE_DECL)
	    name = DECL_NAME (name);

	  t = lookup_tag (code, name, current_binding_level, 1);

	  if (t == 0)
	    {
	      if (IS_AGGR_TYPE_CODE (code))
		t = make_lang_type (code);
	      else
		t = make_node (code);
	      pushtag (name, t);
	      ok = 1;
	      break;
	    }
	  else if (name != 0 || code == ENUMERAL_TYPE)
	    ok = 1;

	  if (ok)
	    found_tag++;
	  else
	    {
	      if (!warned)
		warning ("useless keyword or type name in declaration");
	      warned = 1;
	    }
	}
    }

  /* This is where the variables in an anonymous union are
     declared.  An anonymous union declaration looks like:
     union { ... } ;
     because there is no declarator after the union, the parser
     sends that declaration here.  */
  if (code == UNION_TYPE
      && t != NULL_TREE
      && ANON_AGGRNAME_P (DECL_NAME (TYPE_NAME (t)))
      && TYPE_FIELDS (t))
    {
      tree decl = grokdeclarator (NULL_TREE, declspecs, NORMAL, 0);
      finish_anon_union (decl);
    }
  else if (!warned)
    {
      if (found_tag > 1)
	warning ("multiple types in one declaration");
      if (found_tag == 0)
	warning ("empty declaration");
    }
}

/* Decode a "typename", such as "int **", returning a ..._TYPE node.  */

tree
groktypename (typename)
     tree typename;
{
  if (TREE_CODE (typename) != TREE_LIST)
    return typename;
  return grokdeclarator (TREE_VALUE (typename),
			 TREE_PURPOSE (typename),
			 TYPENAME, 0);
}

/* Decode a declarator in an ordinary declaration or data definition.
   This is called as soon as the type information and variable name
   have been parsed, before parsing the initializer if any.
   Here we create the ..._DECL node, fill in its type,
   and put it on the list of decls for the current context.
   The ..._DECL node is returned as the value.

   Exception: for arrays where the length is not specified,
   the type is left null, to be filled in by `finish_decl'.

   Function definitions do not come here; they go to start_function
   instead.  However, external and forward declarations of functions
   do go through here.  Structure field declarations are done by
   grokfield and not through here.  */

/* Set this to zero to debug not using the temporary obstack
   to parse initializers.  */
int debug_temp_inits = 1;

tree
start_decl (declarator, declspecs, initialized)
     tree declspecs, declarator;
     int initialized;
{
  register tree decl = grokdeclarator (declarator, declspecs,
				       NORMAL, initialized);
  register tree type, tem;
  int init_written = initialized;

  if (decl == NULL_TREE) return decl;
  type = TREE_TYPE (decl);

  /* If this type of object needs a cleanup, and control may
     jump past it, make a new binding level so that it is cleaned
     up only when it is initialized first.  */
  if (TYPE_NEEDS_DESTRUCTOR (type)
      && current_binding_level->more_cleanups_ok == 0)
    pushlevel_temporary (1);

  if (initialized)
    /* Is it valid for this decl to have an initializer at all?
       If not, set INITIALIZED to zero, which will indirectly
       tell `finish_decl' to ignore the initializer once it is parsed.  */
    switch (TREE_CODE (decl))
      {
      case TYPE_DECL:
	/* typedef foo = bar  means give foo the same type as bar.
	   We haven't parsed bar yet, so `finish_decl' will fix that up.
	   Any other case of an initialization in a TYPE_DECL is an error.  */
	if (pedantic || list_length (declspecs) > 1)
	  {
	    error ("typedef `%s' is initialized",
		   IDENTIFIER_POINTER (DECL_NAME (decl)));
	    initialized = 0;
	  }
	break;

      case FUNCTION_DECL:
	error ("function `%s' is initialized like a variable",
	       IDENTIFIER_POINTER (DECL_NAME (decl)));
	initialized = 0;
	break;

      default:
	/* Don't allow initializations for incomplete types
	   except for arrays which might be completed by the initialization.  */
	if (TYPE_SIZE (type) != 0)
	  ;                     /* A complete type is ok.  */
	else if (TREE_CODE (type) != ARRAY_TYPE)
	  {
	    error ("variable `%s' has initializer but incomplete type",
		   IDENTIFIER_POINTER (DECL_NAME (decl)));
	    initialized = 0;
	  }
	else if (TYPE_SIZE (TREE_TYPE (type)) == 0)
	  {
	    error ("elements of array `%s' have incomplete type",
		   IDENTIFIER_POINTER (DECL_NAME (decl)));
	    initialized = 0;
	  }
      }

  if (! (initialized || TREE_CODE (decl) == TYPE_DECL
	 || ! IS_AGGR_TYPE (type) || TREE_EXTERNAL (decl)))
    {
      tree basetype = type;
      if (TYPE_SIZE (type) == 0)
	{
	  error ("aggregate `%s' has incomplete type and cannot be initialized",
		 IDENTIFIER_POINTER (DECL_NAME (decl)));
	  /* Change the type so that assemble_variable will give
	     DECL an rtl we can live with: (mem (const_int 0)).  */
	  TREE_TYPE (decl) = error_mark_node;
	  type = error_mark_node;
	}
      else
	{
	  /* If any base type in the hierarchy of TYPE needs a constructor,
	     then we set initialized to 1.  This way any nodes which are
	     created for the purposes of initializing this aggregate
	     will live as long as it does.  This is necessary for global
	     aggregates which do not have their initializers processed until
	     the end of the file.  */
	  initialized = TYPE_NEEDS_CONSTRUCTING (type);
	}
    }

  if (initialized)
    {
      if (current_binding_level != global_binding_level
	  && TREE_EXTERNAL (decl))
	warning ("declaration of `%s' has `extern' and is initialized",
		 IDENTIFIER_POINTER (DECL_NAME (decl)));
      TREE_EXTERNAL (decl) = 0;
      if (current_binding_level == global_binding_level)
	TREE_STATIC (decl) = 1;

      /* Tell `pushdecl' this is an initialized decl
	 even though we don't yet have the initializer expression.
	 Also tell `finish_decl' it may store the real initializer.  */
      DECL_INITIAL (decl) = error_mark_node;
    }

  /* Add this decl to the current binding level, but not if it
     comes from another scope, e.g. a static member variable.
     TEM may equal DECL or it may be a previous decl of the same name.  */
  if (DECL_CONTEXT (decl) != NULL_TREE)
    tem = decl;
  else
    {
      tem = pushdecl (decl);
      if (TREE_CODE (tem) == TREE_LIST)
	{
	  tree tem2 = value_member (decl, tem);
	  if (tem2 != NULL_TREE)
	    tem = TREE_VALUE (tem2);
	  else
	    {
	      while (tem && ! decls_match (decl, TREE_VALUE (tem)))
		tem = TREE_CHAIN (tem);
	      if (tem == NULL_TREE)
		tem = decl;
	      else
		tem = TREE_VALUE (tem);
	    }
	}
    }

#if 0
  /* We don't do this yet for GNU C++.  */
  /* For a local variable, define the RTL now.  */
  if (current_binding_level != global_binding_level
      /* But not if this is a duplicate decl
	 and we preserved the rtl from the previous one
	 (which may or may not happen).  */
      && DECL_RTL (tem) == 0)
    {
      if (TYPE_SIZE (TREE_TYPE (tem)) != 0)
	expand_decl (tem, NULL_TREE);
      else if (TREE_CODE (TREE_TYPE (tem)) == ARRAY_TYPE
	       && DECL_INITIAL (tem) != 0)
	expand_decl (tem, NULL_TREE);
    }
#endif

    if (TREE_CODE (decl) == FUNCTION_DECL
	&& TREE_OVERLOADED (decl))
      /* @@ Also done in start_function.  */
      push_overloaded_decl (tem);

  if (init_written
      && ! (TREE_CODE (tem) == PARM_DECL
	    || (TREE_CODE (tem) == VAR_DECL
		&& (TREE_READONLY (tem)))))
    {
      /* When parsing and digesting the initializer,
	 use temporary storage.  Do this even if we will ignore the value.  */
      if (current_binding_level == global_binding_level && debug_temp_inits)
	{
	  if (TYPE_NEEDS_CONSTRUCTING (type)
	      || TREE_CODE (type) == REFERENCE_TYPE)
	    /* In this case, the initializer must lay down in permanent
	       storage, since it will be saved until `finish_file'
	       (in first case) can run.   */
	    ;
	  else
	    temporary_allocation ();
	}
    }

  return tem;
}

/* Finish processing of a declaration;
   install its line number and initial value.
   If the length of an array type is not known before,
   it must be determined now, from the initial value, or it is an error.

   For C++, `finish_decl' must be fairly evasive:  it must keep initializers
   for aggregates that have constructors alive on the permanent obstack,
   so that the global initializing functions can be written at the end.

   INIT0 holds the value of an initializer that should be allowed to escape
   the normal rules.

   For functions that take defualt parameters, DECL points to its
   "maximal" instantiation.  finish_decl must then also declared its
   subsequently lower and lower forms of instantiation, checking for
   ambiguity as it goes.  This can be sped up later.  */

void
finish_decl (decl, init, asmspec)
     tree decl, init;
     char *asmspec;
{
  register tree type;
  tree init0 = 0;
  int init_written = init != 0;
  tree cleanup = NULL_TREE;
  int was_incomplete;
  int temporary = allocation_temporary_p ();

  /* If this is 0, then we did not change obstacks.  */
  if (! decl)
    {
      if (init)
	error ("assignment (not initialization) in declaration");
      return;
    }

  /* If the type of the thing we are declaring either has
     a constructor, or has a virtual function table pointer,
     AND its initialization was accepted by `start_decl',
     then we stayed on the permanent obstack through the
     declaration, otherwise, changed obstacks as GCC would.  */

  type = TREE_TYPE (decl);

  was_incomplete = (DECL_SIZE (decl) == 0);

  /* Take care of TYPE_DECLs up front.  */
  if (TREE_CODE (decl) == TYPE_DECL)
    {
      if (init && DECL_INITIAL (decl))
	{
	  /* typedef foo = bar; store the type of bar as the type of foo.  */
	  TREE_TYPE (decl) = type = TREE_TYPE (init);
	  DECL_INITIAL (decl) = init = 0;
	}
      if (IS_AGGR_TYPE (TREE_TYPE (decl)))
	{
	  if (TREE_TYPE (DECL_NAME (decl)))
	    warning ("shadowing previous type declaration of `%s'",
		     IDENTIFIER_POINTER (DECL_NAME (decl)));
	  TREE_TYPE (DECL_NAME (decl)) = decl;
	}
      rest_of_decl_compilation (decl, 0,
				current_binding_level == global_binding_level, 0);
      goto finish_end;
    }

  if (! TREE_EXTERNAL (decl) && TREE_READONLY (decl))
    {
      if (TYPE_NEEDS_CONSTRUCTING (type))
	{
	  /* Currently, GNU C++ puts constants in text
	     space.  This makes them impossible to initialize.
	     In the future, one would hope for an operating system
	     which understood the difference between initialization
	     and the running of a program.  */
	  TREE_READONLY (decl) = 0;
	}
    }

  if (TREE_CODE (decl) == FIELD_DECL)
    {
      if (init && init != error_mark_node)
	assert (TREE_PERMANENT (init));

      if (asmspec)
	{
	  /* This must override the asm specifier which was placed
	     by grokclassfn.  Lay this out fresh.
	     
	     @@ Should emit an error if this redefines an asm-specified
	     @@ name, or if we have already used the function's name.  */
	  DECL_RTL (TREE_TYPE (decl)) = 0;
	  DECL_RTL_NAME (decl) = asmspec;
	  make_decl_rtl (decl, asmspec, 0);
	}

      goto finish_end;
    }

  /* If `start_decl' didn't like having an initialization, ignore it now.  */

  if (init != 0 && DECL_INITIAL (decl) == 0)
    init = 0;
  else if (TREE_EXTERNAL (decl))
    ;
  else if (init == 0 && TREE_CODE (decl) == VAR_DECL && DECL_VIRTUAL_P (decl))
    {
      if (write_virtuals >= 0)
	{
	  init = build_nt (CONSTRUCTOR, NULL_TREE, CLASSTYPE_VIRTUALS (DECL_VPARENT (decl)));
	  DECL_INITIAL (decl) = init;
	}
    }
  else if (TREE_CODE (type) == REFERENCE_TYPE)
    {
      if (! init)
	{
	  error ("variable declared as reference not initialized");
	  if (TREE_CODE (decl) == VAR_DECL)
	    SET_DECL_REFERENCE_SLOT (decl, error_mark_node);
	}
      else
	{
	  char *errstr = 0;
	  int is_reference = TREE_CODE (TREE_TYPE (init)) == REFERENCE_TYPE;
	  tree tmp = is_reference ? convert_from_reference (init) : init;

	  if (is_reference)
	    {
	      if (! comptypes (TREE_TYPE (type), TREE_TYPE (tmp), 0))
		errstr = "assignment between dissimilar reference types";
	    }
	  else if (! comptypes (TREE_TYPE (type), TREE_TYPE (init), 0))
	    errstr = "invalid type conversion for reference";

	  if (errstr)
	    {
	      /* Things did not go smoothly; look for type conversion.  */
	      if (IS_AGGR_TYPE (TREE_TYPE (tmp)))
		{
		  tmp = build_type_conversion (type, init, 0);
		  if (tmp != NULL_TREE)
		    {
		      init = tmp;
		      errstr = 0;
		      is_reference = 1;
		    }
		  else
		    {
		      tmp = build_type_conversion (TREE_TYPE (type), init, 0);
		      if (tmp != NULL_TREE)
			{
			  init = tmp;
			  errstr = 0;
			  is_reference = 0;
			}
		    }
		}
	    }

	  if (errstr)
	    {
	      error (errstr);
	      init = 0;
	      if (TREE_CODE (decl) == VAR_DECL)
		SET_DECL_REFERENCE_SLOT (decl, error_mark_node);
	    }

	  if (init)
	    {
	      /* In the case of initialization, it is permissable
		 to assign one reference to another.  */
	      tree this_ptr_type = build_pointer_type (TREE_TYPE (type));
	      tree actual_init;

	      if (is_reference)
		{
		  DECL_INITIAL (decl) = init;
		}
	      else if (lvalue_p (init))
		{
		  DECL_INITIAL (decl) = convert_to_nonzero_pointer (this_ptr_type, build_unary_op (ADDR_EXPR, init, 0));
		  DECL_INITIAL (decl) = save_expr (DECL_INITIAL (decl));
		  TREE_TYPE (DECL_INITIAL (decl)) = type;
		}
	      else if ((actual_init = unary_complex_lvalue (ADDR_EXPR, init)))
		{
		  /* The initializer for this decl goes into its
		     DECL_REFERENCE_SLOT.  Make sure that we can handle
		     multiple evaluations without ill effect.  */
		  if (TREE_CODE (actual_init) == ADDR_EXPR
		      && TREE_CODE (TREE_OPERAND (actual_init, 0)) == NEW_EXPR)
		    actual_init = save_expr (actual_init);
		  DECL_INITIAL (decl) = convert_to_nonzero_pointer (this_ptr_type, actual_init);
		  if (TREE_CODE (DECL_INITIAL (decl)) != SAVE_EXPR)
		    DECL_INITIAL (decl) = save_expr (DECL_INITIAL (decl));
		  TREE_TYPE (DECL_INITIAL (decl)) = type;
		}
	      else
		{
		  tree target_type = TREE_TYPE (type);
		  tree tmp_addr;
		  tmp = get_temp_name (target_type);
		  if (current_binding_level == global_binding_level)
		    TREE_STATIC (tmp) = 1;
		  tmp_addr = build_unary_op (ADDR_EXPR, tmp, 0);
		  TREE_TYPE (tmp_addr) = build_pointer_type (target_type);
		  DECL_INITIAL (decl) = convert_to_nonzero_pointer (this_ptr_type, tmp_addr);
		  TREE_TYPE (DECL_INITIAL (decl)) = type;
		  if (TYPE_NEEDS_CONSTRUCTING (target_type))
		    {
		      if (current_binding_level == global_binding_level)
			{
			  TREE_STATIC (tmp) = 1;
			  /* lay this variable out now.  Otherwise `output_addressed_constants'
			     gets confused by its initializer.  */
			  make_decl_rtl (tmp, 0, 1);
			  global_aggregates = perm_tree_cons (init, tmp, global_aggregates);
			}
		      else
			{
			  init = build_method_call (tmp, DECL_NAME (TYPE_NAME (target_type)), build_tree_list (NULL_TREE, init), NULL_TREE, LOOKUP_NORMAL);
			  DECL_INITIAL (decl) = build (COMPOUND_EXPR, type, init, DECL_INITIAL (decl));
			  cleanup = maybe_build_cleanup (tmp);
			}
		    }
		  else
		    {
		      DECL_INITIAL (tmp) = init;
		      TREE_STATIC (tmp) = current_binding_level == global_binding_level;
		      finish_decl (tmp, init, 0);
		    }
		}

	      /* ?? Can this be optimized in some cases to
		 hand back the DECL_INITIAL slot??  */
	      if (TYPE_SIZE (TREE_TYPE (type)))
		SET_DECL_REFERENCE_SLOT (decl, build (INDIRECT_REF, TREE_TYPE (type), decl));

	      init = 0;
	    }
	}
    }
  /* PARM_DECLs get cleanups, too.  */
  else if ((TREE_CODE (decl) == VAR_DECL || TREE_CODE (decl) == PARM_DECL)
	   && TYPE_NEEDS_DESTRUCTOR (type))
    {
      if (temporary)
	end_temporary_allocation ();
      cleanup = maybe_build_cleanup (decl);
      if (temporary)
	resume_temporary_allocation ();
    }

  if (TREE_CODE (decl) == CONST_DECL)
    {
      assert (TREE_CODE (decl) != REFERENCE_TYPE);

      DECL_INITIAL (decl) = init;

      /* This will keep us from needing to worry about our obstacks.  */
      assert (init != 0);
      init = 0;
    }
  else if (init)
    {
      if (TYPE_NEEDS_CONSTRUCTING (type))
	{
	  if (TREE_CODE (init) == CONSTRUCTOR)
	    {
	      error_with_decl (decl, "`%s' must be initialized by constructor, not by `{...}'");
	      init = error_mark_node;
	    }
	  /* We must hide the initializer so that expand_decl
	     won't try to do something it does not understand.
	     
	     @@ This may break if we have an array of things to
	     @@ construct, and the array size is not known ahead
	     @@ of time.  */
	  init0 = init;
	  init = 0;
	  DECL_INITIAL (decl) = error_mark_node;
	}
      else
	store_init_value (decl, init);
    }
  else if (IS_AGGR_TYPE (type))
    {
      if (CLASSTYPE_READONLY_FIELDS_NEED_INIT (type))
	error_with_decl (decl, "structure `%s' with uninitialized const members");
      if (CLASSTYPE_REF_FIELDS_NEED_INIT (type))
	error_with_decl (decl, "structure `%s' with uninitialized reference members");
    }
  else if (TREE_CODE (decl) == VAR_DECL && TREE_READONLY (type))
    error_with_decl (decl, "uninitialized const `%s'");

  /* For top-level declaration, the initial value was read in
     the temporary obstack.  MAXINDEX, rtl, etc. to be made below
     must go in the permanent obstack; but don't discard the
     temporary data yet.  */

  if (current_binding_level == global_binding_level && temporary)
    end_temporary_allocation ();

  /* Deduce size of array from initialization, if not already known.  */

  if (TREE_CODE (type) == ARRAY_TYPE
      && TYPE_DOMAIN (type) == 0
      && TREE_CODE (decl) != TYPE_DECL)
    {
      int do_default = ! ((!pedantic && TREE_STATIC (decl))
			  || TREE_EXTERNAL (decl));
      int failure
	= complete_array_type (type, DECL_INITIAL (decl), do_default);

      if (failure == 1)
	error_with_decl (decl, "initializer fails to determine size of `%s'");

      if (failure == 2)
	{
	  if (do_default)
	    error_with_decl (decl, "array size missing in `%s'");
	  else if (!pedantic && TREE_STATIC (decl))
	    TREE_EXTERNAL (decl) = 1;
	}

      if (pedantic && TYPE_DOMAIN (type) != 0
	  && tree_int_cst_lt (TYPE_MAX_VALUE (TYPE_DOMAIN (type)),
			      integer_zero_node))
	error_with_decl (decl, "zero-size array `%s'");

      layout_decl (decl, 0);
    }

  if (TREE_CODE (decl) == VAR_DECL)
    {
      if (TREE_STATIC (decl) && DECL_SIZE (decl) == 0)
	{
	  /* A static variable with an incomplete type:
	     that is an error if it is initialized or `static'.
	     Otherwise, let it through, but if it is not `extern'
	     then it may cause an error message later.  */
	  if (! (TREE_PUBLIC (decl) && DECL_INITIAL (decl) == 0))
	    error_with_decl (decl, "storage size of `%s' isn't known");
	}
      else if (!TREE_EXTERNAL (decl) && DECL_SIZE (decl) == 0)
	{
	  /* An automatic variable with an incomplete type:
	     that is an error.  */
	  error_with_decl (decl, "storage size of `%s' isn't known");
	  TREE_TYPE (decl) = error_mark_node;
	}

      if ((TREE_EXTERNAL (decl) || TREE_STATIC (decl))
	  && DECL_SIZE (decl) != 0 && ! TREE_LITERAL (DECL_SIZE (decl)))
	error_with_decl (decl, "storage size of `%s' isn't constant");
    }

  /* Output the assembler code and/or RTL code for variables and functions,
     unless the type is an undefined structure or union.
     If not, it will get done when the type is completed.  */

  if (TREE_CODE (decl) == VAR_DECL || TREE_CODE (decl) == FUNCTION_DECL
      || TREE_CODE (decl) == RESULT_DECL)
    {
      int toplev = current_binding_level == global_binding_level;
      int was_temp = flag_traditional && allocation_temporary_p ();

      if (was_temp)
	end_temporary_allocation ();

      if (TREE_CODE (decl) == VAR_DECL && DECL_VIRTUAL_P (decl))
	make_decl_rtl (decl, 0, toplev);
      else if (TREE_CODE (decl) == VAR_DECL
	       && TREE_READONLY (decl)
	       && DECL_INITIAL (decl) != 0
	       && DECL_INITIAL (decl) != error_mark_node)
	{
	  DECL_INITIAL (decl) = save_expr (DECL_INITIAL (decl));

	  DECL_RTL_NAME (decl) = asmspec;

	  if (! toplev
	      && TREE_STATIC (decl)
	      && ! TREE_VOLATILE (decl)
	      && ! TREE_PUBLIC (decl)
	      && ! TREE_EXTERNAL (decl)
	      && ! TYPE_NEEDS_DESTRUCTOR (type)
	      && DECL_MODE (decl) != BLKmode)
	    {
	      /* If this variable is really a constant, then fill its DECL_RTL
		 slot with something which won't take up storage.
		 If something later should take its address, we can always give
		 it legitimate RTL at that time.  */
	      DECL_RTL (decl) = (struct rtx_def *)gen_reg_rtx (DECL_MODE (decl));
	      store_expr (DECL_INITIAL (decl), DECL_RTL (decl), 0);
	      TREE_ASM_WRITTEN (decl) = 1;
	    }
	  else if (toplev)
	    make_decl_rtl (decl, DECL_RTL_NAME (decl), toplev);
	  else
	    rest_of_decl_compilation (decl, DECL_RTL_NAME (decl), toplev, 0);
	}
      else
	rest_of_decl_compilation (decl, asmspec, toplev, 0);

      if (was_temp)
	resume_temporary_allocation ();

      if (IS_AGGR_TYPE (type)
	  && CLASSTYPE_UNINHERITED_VIRTUALS (type))
	uninheritable_virtuals_error (decl, type);
      else if (TREE_CODE (type) == FUNCTION_TYPE
	       && IS_AGGR_TYPE (TREE_TYPE (type))
	       && CLASSTYPE_UNINHERITED_VIRTUALS (TREE_TYPE (type)))
	uninheritable_virtuals_error (decl, TREE_TYPE (type));

      /* C++: Handle overloaded functions with defualt parameters.  */
      if (TREE_CODE (type) == FUNCTION_TYPE
	  && TREE_OVERLOADED (decl))
	{
	  tree parmtypes = TYPE_ARG_TYPES (type);
	  tree prev = NULL_TREE;
	  char *original_name = IDENTIFIER_POINTER (DECL_ORIGINAL_NAME (decl));

	  while (parmtypes && TREE_VALUE (parmtypes) != void_type_node)
	    {
	      if (TREE_PURPOSE (parmtypes))
		{
		  tree fnname, fndecl;
		  tree *argp = prev
		    ? & TREE_CHAIN (prev)
		      : & TYPE_ARG_TYPES (type);

		  *argp = NULL_TREE;
		  fnname = build_decl_overload (original_name, TYPE_ARG_TYPES (type));
		  *argp = parmtypes;
		  fndecl = build_decl (FUNCTION_DECL, fnname, type);
		  TREE_EXTERNAL (fndecl) = TREE_EXTERNAL (decl);
		  TREE_PUBLIC (fndecl) = TREE_PUBLIC (decl);
		  TREE_INLINE (fndecl) = TREE_INLINE (decl);
		  /* Keep G++ from thinking this function is unused.
		     It is only used to speed up search in name space.  */
		  TREE_USED (fndecl) = 1;
		  DECL_LANG_SPECIFIC (fndecl) = DECL_LANG_SPECIFIC (decl);
		  fndecl = pushdecl (fndecl);
		  DECL_RTL (fndecl) = DECL_RTL (decl);
		}
	      prev = parmtypes;
	      parmtypes = TREE_CHAIN (parmtypes);
	    }
	}
      else if (current_binding_level == global_binding_level)
	{
	  if (TREE_EXTERNAL (decl) == 0 && TYPE_NEEDS_CONSTRUCTING (type))
	    global_aggregates = perm_tree_cons (init0, decl, global_aggregates);
	}
      else
	{
	  /* This is a declared decl which must live until the
	     end of the binding contour.  It may need a cleanup.  */

	  /* If this decl is static, and it needs a constructor,
	     then we are in trouble (until I implement somthing
	     more intelligent).  */

	  /* Recompute the RTL of a local array now
	     if it used to be an incomplete type.  */
	  if (was_incomplete
	      && ! TREE_STATIC (decl) && ! TREE_EXTERNAL (decl))
	    {
	      /* If we used it already as memory, it must stay in memory.  */
	      TREE_ADDRESSABLE (decl) = TREE_USED (decl);
	      /* If it's still incomplete now, no init will save it.  */
	      if (DECL_SIZE (decl) == 0)
		DECL_INITIAL (decl) = 0;
	      expand_decl (decl, NULL_TREE);
	    }
	  else if (! TREE_ASM_WRITTEN (decl)
		   && TYPE_SIZE (type) != 0 || TREE_CODE (type) == ARRAY_TYPE)
	    {
	      /* Do this here, because we did not expand this decl's
		 rtl in start_decl.  */
	      if (DECL_RTL (decl) == 0)
		expand_decl (decl, cleanup);
	    }

	  /* Compute and store the initial value.  */
	  expand_decl_init (decl);

	  if (init0)
	    {
	      /* Function return values have initialization semantics.
		 We call expand_assignment because it "does the right
		 thing"; namely, in the case of aggregates (which we
		 force to be BLKmode), it takes the address of the item
		 we want to initialize, and initializes it.  */
	      if (TREE_STATIC (decl))
		{
		  sorry ("initialization of local static objects with constructor");
		}
	      else
		expand_aggr_init (decl, init0, 0);
	    }
	  else if (TYPE_NEEDS_CONSTRUCTING (type))
	    if (TREE_STATIC (decl))
	      sorry ("initialization of local static object needing constructor");
	    else
	      expand_aggr_init (decl, 0, 0);
	  else if (cleanup && TREE_STATIC (decl))
	    sorry ("local static object needing destructor");
	  else if (IS_AGGR_TYPE (type) && CLASSTYPE_BASE_INIT_LIST (type))
	    expand_aggr_init (decl, 0, 0);

	  /* Set this to 0 so we can tell whether an aggregate
	     which was initialized was ever used.  */
	  if (TYPE_NEEDS_CONSTRUCTING (type))
	    TREE_USED (decl) = 0;
	}
    }

 finish_end:
  /* Resume permanent allocation, if not within a function.  */
  if (temporary && current_binding_level == global_binding_level)
    {
      permanent_allocation ();
#if 0
      /* @@ I don't know whether this is true for GNU C++.  */
      /* We need to remember that this array HAD an initialization,
	 but discard the actual nodes, since they are temporary anyway.  */
      if (DECL_INITIAL (decl) != 0)
	DECL_INITIAL (decl) = error_mark_node;
#endif
    }
}

/* If DECL is of a type which needs a cleanup, build that cleanup here.
   We don't build cleanups if just going for syntax checking, since
   fixup_cleanups does not know how to not handle them.

   Don't build these on the momentary obstack; they must live
   the life of the binding contour.  */
tree
maybe_build_cleanup (decl)
     tree decl;
{
  extern int flag_syntax_only;
  /* @@ Do we need a VEC_DELETE_EXPR?  */
  if (TYPE_NEEDS_DESTRUCTOR (TREE_TYPE (decl)) && flag_syntax_only == 0)
    {
      int old = suspend_momentary ();
      /* Build this cheaply now.  It will be expanded later, and
         visibility can be checked then.  */
      tree rval = build (DELETE_EXPR, void_type_node, decl, integer_zero_node);
      resume_momentary (old);
      return rval;
    }
  return 0;
}

/* Make TYPE a complete type based on INITIAL_VALUE.
   Return 0 if successful, 1 if INITIAL_VALUE can't be decyphered,
   2 if there was no information (in which case assume 1 if DO_DEFAULT).  */

int
complete_array_type (type, initial_value, do_default)
     tree type;
     tree initial_value;
     int do_default;
{
  register tree maxindex = NULL_TREE;
  int value = 0;

  if (initial_value)
    {
      /* Note MAXINDEX  is really the maximum index,
	 one less than the size.  */
      if (TREE_CODE (initial_value) == STRING_CST)
	maxindex = build_int_2 (TREE_STRING_LENGTH (initial_value) - 1, 0);
      else if (TREE_CODE (initial_value) == CONSTRUCTOR)
	{
	  register int nelts
	    = list_length (CONSTRUCTOR_ELTS (initial_value));
	  maxindex = build_int_2 (nelts - 1, 0);
	}
      else
	{
	  /* Make an error message unless that happened already.  */
	  if (initial_value != error_mark_node)
	    value = 1;

	  /* Prevent further error messages.  */
	  maxindex = build_int_2 (1, 0);
	}
    }

  if (!maxindex)
    {
      if (do_default)
	maxindex = build_int_2 (1, 0);
      value = 2;
    }

  if (maxindex)
    {
      TYPE_DOMAIN (type) = build_index_type (maxindex);
      if (!TREE_TYPE (maxindex))
	TREE_TYPE (maxindex) = TYPE_DOMAIN (type);
    }

  /* Lay out the type now that we can get the real answer.  */

  layout_type (type);

  return value;
}

/* Return zero if something is declared to be a member of type
   CTYPE when in the context of CUR_TYPE.  STRING is the error
   message to print in that case.  Otherwise, quietly return 1.  */
static int
member_function_or_else (ctype, cur_type, string)
     tree ctype, cur_type;
     char *string;
{
  if (ctype && ctype != cur_type)
    {
      error (string, TYPE_NAME_STRING (ctype));
      return 0;
    }
  return 1;
}

/* Given declspecs and a declarator,
   determine the name and type of the object declared
   and construct a ..._DECL node for it.
   (In one case we can return a ..._TYPE node instead.
    For invalid input we sometimes return 0.)

   DECLSPECS is a chain of tree_list nodes whose value fields
    are the storage classes and type specifiers.

   DECL_CONTEXT says which syntactic context this declaration is in:
     NORMAL for most contexts.  Make a VAR_DECL or FUNCTION_DECL or TYPE_DECL.
     FUNCDEF for a function definition.  Like NORMAL but a few different
      error messages in each case.  Return value may be zero meaning
      this definition is too screwy to try to parse.
     MEMFUNCDEF for a function definition.  Like FUNCDEF but prepares to
      handle member functions (which have FIELD context).
      Return value may be zero meaning this definition is too screwy to
      try to parse.
     PARM for a parameter declaration (either within a function prototype
      or before a function body).  Make a PARM_DECL, or return void_type_node.
     TYPENAME if for a typename (in a cast or sizeof).
      Don't make a DECL node; just return the ..._TYPE node.
     FIELD for a struct or union field; make a FIELD_DECL.
   INITIALIZED is 1 if the decl has an initializer.

   In the TYPENAME case, DECLARATOR is really an absolute declarator.
   It may also be so in the PARM case, for a prototype where the
   argument type is specified but not the name.

   This function is where the complicated C meanings of `static'
   and `extern' are intrepreted.

   For C++, if there is any monkey business to do, the function which
   calls this one must do it, i.e., prepending instance variables,
   renaming overloaded function names, etc.

   Note that for this C++, it is an error to define a method within a class
   which does not belong to that class.

   Execpt in the case where SCOPE_REFs are implicitly known (such as
   methods within a class being redundantly qualified),
   declarations which involve SCOPE_REFs are returned as SCOPE_REFs
   (class_name::decl_name).  The caller must also deal with this.

   If a constructor or destructor is seen, and the context is FIELD,
   then the type gains the attribtue TREE_HAS_x.  If such a declaration
   is erroneous, NULL_TREE is returned.

   May return void_type_node if the declarator turned out to be a friend.
   See grokfield for details.  */

enum return_types { return_normal, return_ctor, return_dtor, return_conversion, };

static tree
grokdeclarator (declarator, declspecs, decl_context, initialized)
     tree declspecs;
     tree declarator;
     enum decl_context decl_context;
     int initialized;
{
  int specbits = 0;
  int nclasses = 0;
  tree spec;
  tree type = NULL_TREE;
  int longlong = 0;
  int constp;
  int volatilep;
  int virtualp, friendp, inlinep, ctorp = 0;
  int explicit_int = 0;
  int explicit_char = 0;
  char *name;
  tree typedef_type = 0;
  int funcdef_flag = 0;
  int resume_temporary = 0;
  enum tree_code innermost_code = ERROR_MARK;
  /* Set this to error_mark_node for FIELD_DECLs we could not handle properly.
     All FIELD_DECLs we build here have `init' put into their DECL_INITIAL.  */
  tree init = 0;

  /* Keep track of what sort of function is being processed
     so that we can warn about default return values, or explicit
     return values which do not match prescribed defaults.  */
  enum return_types return_type = return_normal;

  tree dname = NULL_TREE;
  tree ctype = current_class_type;
  tree ctor_return_type = NULL_TREE;
  enum overload_flags flags = NO_SPECIAL;

  if (decl_context == FUNCDEF)
    funcdef_flag = 1, decl_context = NORMAL;
  else if (decl_context == MEMFUNCDEF)
    funcdef_flag = -1, decl_context = FIELD;

  if (flag_traditional && allocation_temporary_p ())
    {
      resume_temporary = 1;
      end_temporary_allocation ();
    }

  /* Look inside a declarator for the name being declared
     and get it as a string, for an error message.  */
  {
    tree type;
    register tree decl = declarator;
    name = 0;

    /* If we see something of the form `aggr_type xyzzy (a, b, c)'
       it is either an old-style function declaration or a call to
       a constructor.  The following conditional makes recognizes this
       case as being a call to a constructor.  Too bad if it is not.  */
    if (current_lang_name == lang_name_cplusplus
        && decl && declspecs
        && TREE_CODE (decl) == CALL_EXPR
        && TREE_OPERAND (decl, 0)
        && TREE_CODE (TREE_OPERAND (decl, 0)) == IDENTIFIER_NODE)
      {
        type = TREE_CODE (TREE_VALUE (declspecs)) == IDENTIFIER_NODE
          ? lookup_name (TREE_VALUE (declspecs)) :
        (IS_AGGR_TYPE (TREE_VALUE (declspecs))
         ? TYPE_NAME (TREE_VALUE (declspecs)) : NULL_TREE);

        if (type && TREE_CODE (type) == TYPE_DECL
            && IS_AGGR_TYPE (TREE_TYPE (type))
            && parmlist_is_exprlist (TREE_OPERAND (decl, 1)))
          {
            if (decl_context == FIELD
                && TREE_CHAIN (TREE_OPERAND (decl, 1)))
              {
                /* That was an initializer list.  */
                sorry ("initializer lists for field declarations");
                decl = TREE_OPERAND (decl, 0);
                declarator = decl;
                init = error_mark_node;
                goto bot;
              }
            else
              {
                tree d = start_decl (TREE_OPERAND (decl, 0), declspecs, 1);
                finish_decl (d, TREE_OPERAND (decl, 1), NULL_TREE);
                return 0;
              }
          }

        if (parmlist_is_random (TREE_OPERAND (decl, 1)))
          {
            error ("bad parameter list specification for function `%s'",
                   IDENTIFIER_POINTER (TREE_OPERAND (decl, 0)));
            return 0;
          }
      bot:
        ;
      }

    while (decl)
      switch (TREE_CODE (decl))
        {
	case WRAPPER_EXPR:      /* for C++ wrappers.  */
	  if (current_lang_name != lang_name_cplusplus)
	    error ("wrapper declared in \"%s\" language context",
		   IDENTIFIER_POINTER (current_lang_name));

	  ctype = NULL_TREE;
	  assert (flags == NO_SPECIAL);
	  flags = WRAPPER_FLAG;
	  decl = TREE_OPERAND (decl, 0);
	  break;

	case ANTI_WRAPPER_EXPR: /* for C++ wrappers.  */
	  if (current_lang_name != lang_name_cplusplus)
	    error ("anti-wrapper declared in \"%s\" language context",
		   IDENTIFIER_POINTER (current_lang_name));

	  ctype = NULL_TREE;
	  assert (flags == NO_SPECIAL);
	  flags = ANTI_WRAPPER_FLAG;
	  decl = TREE_OPERAND (decl, 0);
	  break;

	case COND_EXPR:
	  if (current_lang_name != lang_name_cplusplus)
	    error ("wrapper predicate declared in \"%s\" language context",
		   IDENTIFIER_POINTER (current_lang_name));

	  ctype = NULL_TREE;
	  assert (flags == WRAPPER_FLAG);
	  flags = WRAPPER_PRED_FLAG;
	  decl = TREE_OPERAND (decl, 0);
	  break;

	case BIT_NOT_EXPR:      /* for C++ destructors!  */
	  if (current_lang_name != lang_name_cplusplus)
	    error ("destructor declared in \"%s\" language context",
		   IDENTIFIER_POINTER (current_lang_name));

	  assert (flags == NO_SPECIAL);
	  flags = DTOR_FLAG;
	  return_type = return_dtor;
	  decl = TREE_OPERAND (decl, 0);
	  if (TREE_CODE (decl) != IDENTIFIER_NODE)
	    abort ();
	  if ((ctype && DECL_NAME (TYPE_NAME (ctype)) != decl)
	      || (current_class_name && current_class_name != decl))
	    error ("destructor `%s' must match class name", IDENTIFIER_POINTER (decl));
	  else if (ctype == NULL_TREE && current_class_type == NULL_TREE)
	    {
	      error ("destructors must be member functions");
	      flags = NO_SPECIAL;
	    }
	  break;

	case ADDR_EXPR:         /* C++ reference declaration */
	  /* fall through */
	case ARRAY_REF:
	case INDIRECT_REF:
	  ctype = NULL_TREE;
	  innermost_code = TREE_CODE (decl);
	  decl = TREE_OPERAND (decl, 0);
	  break;

	case CALL_EXPR:
	  innermost_code = TREE_CODE (decl);
	  decl = TREE_OPERAND (decl, 0);
	  if (decl_context == FIELD
	      && ctype == NULL_TREE)
	    ctype = current_class_type;
	  if (ctype != NULL_TREE
	      && decl != NULL_TREE && flags != DTOR_FLAG
	      && TREE_TYPE (decl) && TREE_TYPE (TREE_TYPE (decl)) == ctype)
	    {
	      return_type = return_ctor;
	      ctor_return_type = ctype;
	    }
	  ctype = NULL_TREE;
	  break;

	case IDENTIFIER_NODE:
	  dname = decl;
	  name = IDENTIFIER_POINTER (decl);
	  decl = 0;
	  break;

	case OP_IDENTIFIER:
	  if (current_lang_name != lang_name_cplusplus)
	    error ("operator declared in \"%s\" language context",
		   IDENTIFIER_POINTER (current_lang_name));

	  /* C++ operators: if these are member functions, then
	     they overload the same way normal methods do.  However,
	     if they are declared outside of a classes scope, then
	     they are implicitly treated like `friends', i.e.,
	     they do not take any unseen arguments.  */
	  assert (flags == NO_SPECIAL);
	  flags = OP_FLAG;
	  name = "operator name";
	  decl = 0;
	  break;

	case TYPE_EXPR:
	  if (current_lang_name != lang_name_cplusplus)
	    error ("type conversion operator declared in \"%s\" language context",
		   IDENTIFIER_POINTER (current_lang_name));

	  ctype = NULL_TREE;
	  assert (flags == NO_SPECIAL);
	  flags = TYPENAME_FLAG;
	  name = "operator <typename>";
	  /* Go to the absdcl.  */
	  decl = TREE_OPERAND (decl, 0);
	  return_type = return_conversion;
	  break;

	  /* C++ extension */
	case SCOPE_REF:
	  if (current_lang_name != lang_name_cplusplus)
	      error ("member function declared in \"%s\" language context",
		   IDENTIFIER_POINTER (current_lang_name));

	  {
	    /* Perform error checking, and convert class names to types.
	       We may call grokdeclarator multiple times for the same
	       tree structure, so only do the conversion once.  In this
	       case, we have exactly what we want for `ctype'.  */
	    tree cname = TREE_OPERAND (decl, 0);
	    if (IS_AGGR_TYPE (cname))
	      ctype = cname;
	    else if (! is_aggr_typedef_or_else (cname))
	      {
		TREE_OPERAND (decl, 0) = 0;
	      }
	    else if (TREE_CODE (TREE_OPERAND (decl, 1)) == INDIRECT_REF)
	      {
		TREE_OPERAND (decl, 0) = TREE_TYPE (TREE_TYPE (cname));
	      }
	    else if (ctype == NULL_TREE)
	      {
		ctype = TREE_TYPE (TREE_TYPE (cname));
		TREE_OPERAND (decl, 0) = ctype;
	      }
	    else
	      {
		tree new_type = get_base_type (TREE_TYPE (TREE_TYPE (cname)), ctype, 0);
		if (new_type == NULL_TREE)
		  {
		    error ("type `%s' is not derived from type `%s'",
			   IDENTIFIER_POINTER (cname),
			   TYPE_NAME_STRING (ctype));
		    TREE_OPERAND (decl, 0) = 0;
		  }
		else
		  {
		    ctype = new_type;
		    TREE_OPERAND (decl, 0) = ctype;
		  }
	      }
	    decl = TREE_OPERAND (decl, 1);
	    if (ctype != NULL_TREE && DECL_NAME (TYPE_NAME (ctype)) == decl)
	      {
		return_type = return_ctor;
		ctor_return_type = ctype;
	      }
	  }
	  break;

	default:
	  assert (0);
	}
    if (name == 0)
      name = "type name";
  }

  /* A function definition's declarator must have the form of
     a function declarator.  */

  if (funcdef_flag && innermost_code != CALL_EXPR)
    return 0;

  /* Anything declared one level down from the top level
     must be one of the parameters of a function
     (because the body is at least two levels down).  */

  /* This heuristic cannot be applied to C++ nodes! Fixed, however,
     by not allowing C++ class definitions to specify their parameters
     with xdecls (must be spec.d in the parmlist).

     Since we now wait to push a class scope until we are sure that
     we are in a legitimate method context, we must set oldcname
     explicitly (since current_class_name is not yet alive).  */

  if (decl_context == NORMAL
      && current_binding_level->level_chain == global_binding_level)
    decl_context = PARM;

  /* Look through the decl specs and record which ones appear.
     Some typespecs are defined as built-in typenames.
     Others, the ones that are modifiers of other types,
     are represented by bits in SPECBITS: set the bits for
     the modifiers that appear.  Storage class keywords are also in SPECBITS.

     If there is a typedef name or a type, store the type in TYPE.
     This includes builtin typedefs such as `int'.

     Set EXPLICIT_INT if the type is `int' or `char' and did not
     come from a user typedef.

     Set LONGLONG if `long' is mentioned twice.

     For C++, constructors and destructors have their own fast treatment.  */

  for (spec = declspecs; spec; spec = TREE_CHAIN (spec))
    {
      register int i;
      register tree id = TREE_VALUE (spec);

      /* Certain parse errors slip through.  For example,
	 `int class;' is not caught by the parser. Try
	 weakly to recover here.  */
      if (TREE_CODE (spec) != TREE_LIST)
	return 0;

      if (TREE_CODE (id) == IDENTIFIER_NODE)
	{
	  if (id == ridpointers[(int) RID_INT])
	    {
	      if (type)
		error ("extraneous `int' ignored");
	      else
		{
		  explicit_int = 1;
		  type = TREE_TYPE (IDENTIFIER_GLOBAL_VALUE (id));
		}
	      goto found;
	    }
	  if (id == ridpointers[(int) RID_CHAR])
	    {
	      if (type)
		error ("extraneous `char' ignored");
	      else
		{
		  explicit_char = 1;
		  type = TREE_TYPE (IDENTIFIER_GLOBAL_VALUE (id));
		}
	      goto found;
	    }
	  /* C++ aggregate types.  */
	  if (TREE_TYPE (id))
	    {
	      if (type)
		error ("multiple declarations `%s' and `%s'",
		       IDENTIFIER_POINTER (type),
		       IDENTIFIER_POINTER (id));
	      else
		type = TREE_TYPE (TREE_TYPE (id));
	      goto found;
	    }

	  for (i = (int) RID_FIRST_MODIFIER; i < (int) RID_MAX; i++)
	    {
	      if (ridpointers[i] == id)
		{
		  if (i == (int) RID_LONG && specbits & (1<<i))
		    {
		      if (pedantic)
			warning ("duplicate `%s'", IDENTIFIER_POINTER (id));
		    else if (longlong)
		      warning ("`long long long' is too long for GCC");
		      else
			longlong = 1;
		    }
		  else if (specbits & (1 << i))
		    warning ("duplicate `%s'", IDENTIFIER_POINTER (id));
		  specbits |= 1 << i;
		  goto found;
		}
	    }
	}
      if (type)
	error ("two or more data types in declaration of `%s'", name);
      else if (TREE_CODE (id) == IDENTIFIER_NODE)
	{
	  register tree t = lookup_name (id);
	  if (!t || TREE_CODE (t) != TYPE_DECL)
	    error ("`%s' fails to be a typedef or built in type",
		   IDENTIFIER_POINTER (id));
	  else type = TREE_TYPE (t);
	}
      else if (TREE_CODE (id) != ERROR_MARK)
	/* Can't change CLASS nodes into RECORD nodes here!  */
	type = id;

    found: {}
    }

  typedef_type = type;

  /* No type at all: default to `int', and set EXPLICIT_INT
     because it was not a user-defined typedef.  */

  if (type == 0)
    {
      explicit_int = -1;
      if (return_type == return_dtor)
	type = void_type_node;
      else if (return_type == return_ctor)
	type = TYPE_POINTER_TO (ctor_return_type);
      else
	{
	  if (funcdef_flag && warn_return_type
	      && return_type == return_normal
	      && ! (specbits & ((1 << (int) RID_LONG) | (1 << (int) RID_SHORT)
				| (1 << (int) RID_SIGNED) | (1 << (int) RID_UNSIGNED))))
	    warn_about_return_type = 1;
	  /* Save warning until we know what is really going on.  */
	  type = integer_type_node;
	}
    }
  ctype = NULL_TREE;

  /* Now process the modifiers that were specified
     and check for invalid combinations.  */

  /* Long double is a special combination.  */

  if ((specbits & 1 << (int) RID_LONG) && type == double_type_node)
    {
      specbits &= ~ (1 << (int) RID_LONG);
      type = long_double_type_node;
    }

  /* Check all other uses of type modifiers.  */

  if (specbits & ((1 << (int) RID_LONG) | (1 << (int) RID_SHORT)
		  | (1 << (int) RID_UNSIGNED) | (1 << (int) RID_SIGNED)))
    {
      if (!explicit_int && !explicit_char && !pedantic)
	error ("long, short, signed or unsigned used invalidly for `%s'", name);
      else if ((specbits & 1 << (int) RID_LONG) && (specbits & 1 << (int) RID_SHORT))
	error ("long and short specified together for `%s'", name);
      else if (((specbits & 1 << (int) RID_LONG) || (specbits & 1 << (int) RID_SHORT))
	       && explicit_char)
	error ("long or short specified with char for `%s'", name);
      else if ((specbits & 1 << (int) RID_SIGNED) && (specbits & 1 << (int) RID_UNSIGNED))
	error ("signed and unsigned given together for `%s'", name);
      else
	{
	  if (specbits & 1 << (int) RID_UNSIGNED)
	    {
	      if (longlong)
		type = long_long_unsigned_type_node;
	      else if (specbits & 1 << (int) RID_LONG)
		type = long_unsigned_type_node;
	      else if (specbits & 1 << (int) RID_SHORT)
		type = short_unsigned_type_node;
	      else if (type == char_type_node)
		type = unsigned_char_type_node;
	      else
		type = unsigned_type_node;
	    }
	  else if ((specbits & 1 << (int) RID_SIGNED)
		   && type == char_type_node)
	    type = signed_char_type_node;
	  else if (longlong)
	    type = long_long_integer_type_node;
	  else if (specbits & 1 << (int) RID_LONG)
	    type = long_integer_type_node;
	  else if (specbits & 1 << (int) RID_SHORT)
	    type = short_integer_type_node;
	}
    }

  /* Set CONSTP if this declaration is `const', whether by
     explicit specification or via a typedef.
     Likewise for VOLATILEP.  */

  constp = !! (specbits & 1 << (int) RID_CONST) + TREE_READONLY (type);
  volatilep = !! (specbits & 1 << (int) RID_VOLATILE) + TREE_VOLATILE (type);
  inlinep = !! (specbits & (1 << (int) RID_INLINE));
  if (constp > 1)
    warning ("duplicate `const'");
  if (volatilep > 1)
    warning ("duplicate `volatile'");
  virtualp = specbits & (1 << (int) RID_VIRTUAL);
  if (virtualp && (specbits & (1 << (int) RID_STATIC)))
    {
      error ("member function cannot be declared both virtual and static");
      specbits &= ~ (1 << (int) RID_STATIC);
    }
  friendp = specbits & (1 << (int) RID_FRIEND);
  specbits &= ~ ((1 << (int) RID_VIRTUAL) | (1 << (int) RID_FRIEND));

  /* Warn if two storage classes are given. Default to `auto'.  */

  if (specbits)
    {
      if (specbits & 1 << (int) RID_AUTO) nclasses++;
      if (specbits & ((1 << (int) RID_STATIC)|(1 << (int) RID_INLINE)))
	nclasses++;
      if (specbits & 1 << (int) RID_EXTERN) nclasses++;
      if (specbits & 1 << (int) RID_REGISTER) nclasses++;
      if (specbits & 1 << (int) RID_TYPEDEF) nclasses++;
    }

  /* Give error if `virtual' is used outside of class declaration.  */
  if (virtualp && current_class_name == NULL_TREE)
    {
      error ("virtual outside class declaration");
      virtualp = 0;
    }

  /* Warn about storage classes that are invalid for certain
     kinds of declarations (parameters, typenames, etc.).  */

  if (nclasses > 1)
    error ("multiple storage classes in declaration of `%s'", name);
  else if (decl_context != NORMAL && nclasses > 0)
    {
      if (decl_context == PARM && (specbits & (1 << (int) RID_REGISTER)|(1 << (int) RID_AUTO)))
	;
      else if (decl_context == FIELD
	       && (specbits & ((1 << (int) RID_STATIC)|(1 << (int) RID_INLINE))))
	;                       /* C++ allows static class elements */
      else if (decl_context == FIELD && (specbits == (1 << (int) RID_TYPEDEF)))
	{
	  /* A typedef which was made in a class's scope.  */
	  tree loc_typedecl;
	  /* keep `grokdeclarator' from thinking we are in PARM context.  */
	  pushlevel (0);
	  loc_typedecl = start_decl (declarator, declspecs, initialized);
	  poplevel (0, 0, 0);
	  if (TREE_CODE (TREE_TYPE (loc_typedecl)) == ENUMERAL_TYPE)
	    {
	      tree ref = lookup_tag (ENUMERAL_TYPE, DECL_NAME (loc_typedecl), current_binding_level, 0);
	      if (! ref)
		pushtag (DECL_NAME (loc_typedecl), TREE_TYPE (loc_typedecl));
	    }
	  if (IDENTIFIER_CLASS_VALUE (DECL_NAME (loc_typedecl)))
	    error_with_decl (loc_typedecl,
			     "typedef of `%s' in class scope hides previous declaration");
	  IDENTIFIER_CLASS_VALUE (DECL_NAME (loc_typedecl)) = loc_typedecl;
	  return loc_typedecl;
	}
      else
	{
	  error ((decl_context == FIELD
		  ? "storage class specified for structure field `%s'"
		  : (decl_context == PARM
		     ? "storage class specified for parameter `%s'"
		     : "storage class specified for typename")),
		 name);
	  specbits &= ~ ((1 << (int) RID_TYPEDEF) | (1 << (int) RID_REGISTER)
			 | (1 << (int) RID_AUTO) | (1 << (int) RID_STATIC)
			 | (1 << (int) RID_EXTERN) | (1 << (int) RID_INLINE));
	}
    }
  else if (current_binding_level == global_binding_level)
    {
      if (specbits & (1 << (int) RID_AUTO))
	error ("top-level declaration of `%s' specifies `auto'", name);
#if 0
      if (specbits & (1 << (int) RID_REGISTER))
	error ("top-level declaration of `%s' specifies `register'", name);
#endif
#if 0
      /* I'm not sure under what circumstances we should turn
	 on the extern bit, and under what circumstances we should
	 warn if other bits are turned on.  */
      if (decl_context == NORMAL
	  && (specbits & (1 << (int) RID_EXTERN)) == 0
	  && ! root_lang_context_p ())
	{
	  specbits |= (1 << (int) RID_EXTERN);
	}
#endif
    }

  /* Now figure out the structure of the declarator proper.
     Descend through it, creating more complex types, until we reach
     the declared identifier (or NULL_TREE, in an absolute declarator).  */

  while (declarator && TREE_CODE (declarator) != IDENTIFIER_NODE)
    {
      /* Each level of DECLARATOR is either an ARRAY_REF (for ...[..]),
	 an INDIRECT_REF (for *...),
	 a CALL_EXPR (for ...(...)),
	 an identifier (for the name being declared)
	 or a null pointer (for the place in an absolute declarator
	 where the name was omitted).
	 For the last two cases, we have just exited the loop.

	 For C++ it could also be
	 a SCOPE_REF (for class :: ...).  In this case, we have converted
	 sensible names to types, and those are the values we use to
	 qualify the member name.
	 an ADDR_EXPR (for &...),
	 a BIT_NOT_EXPR (for destructors)
	 a TYPE_EXPR (for operator typenames)
	 a WRAPPER_EXPR (for wrappers)
	 an ANTI_WRAPPER_EXPR (for averting wrappers)

	 At this point, TYPE is the type of elements of an array,
	 or for a function to return, or for a pointer to point to.
	 After this sequence of ifs, TYPE is the type of the
	 array or function or pointer, and DECLARATOR has had its
	 outermost layer removed.  */

      if (TREE_CODE (type) == ERROR_MARK
	  && TREE_CODE (declarator) != OP_IDENTIFIER)
	{
	  declarator = TREE_OPERAND (declarator, 0);
	  continue;
	}
      else switch (TREE_CODE (declarator))
	{
	case ARRAY_REF:
	  {
	    register tree itype = NULL_TREE;
	    register tree size = TREE_OPERAND (declarator, 1);

	    declarator = TREE_OPERAND (declarator, 0);

	    /* Check for some types that there cannot be arrays of.  */

	    if (type == void_type_node)
	      {
		error ("declaration of `%s' as array of voids", name);
		type = error_mark_node;
	      }

	    if (TREE_CODE (type) == FUNCTION_TYPE)
	      {
		error ("declaration of `%s' as array of functions", name);
		type = error_mark_node;
	      }

	    if (size == error_mark_node)
	      type = error_mark_node;

	    if (type == error_mark_node)
	      continue;

	    if (size)
	      {
		/* might be a cast */
		if (TREE_CODE (size) == NOP_EXPR
		    && TREE_TYPE (size) == TREE_TYPE (TREE_OPERAND (size, 0)))
		  size = TREE_OPERAND (size, 0);

		if (TREE_CODE (TREE_TYPE (size)) != INTEGER_TYPE
		    && TREE_CODE (TREE_TYPE (size)) != ENUMERAL_TYPE)
		  {
		    error ("size of array `%s' has non-integer type", name);
		    size = integer_one_node;
		  }
		if (pedantic && integer_zerop (size))
		  warning ("ANSI C forbids zero-size array `%s'", name);
		if (TREE_LITERAL (size))
		  {
		    if (INT_CST_LT (size, integer_zero_node))
		      {
			error ("size of array `%s' is negative", name);
			size = integer_one_node;
		      }
		    itype = build_index_type (build_int_2 (TREE_INT_CST_LOW (size) - 1, 0));
		  }
		else
		  {
		    if (pedantic)
		      warning ("ANSI C forbids variable-size array `%s'", name);
		    itype = build_binary_op (MINUS_EXPR, size, integer_one_node);
		    itype = build_index_type (itype);
		  }
	      }

	    /* Build the array type itself.
	       Merge any constancy or volatility into the target type.  */

	    if (constp || volatilep)
	      type = build_type_variant (type, constp, volatilep);

#if 0   /* don't clear these; leave them set so that the array type
	   or the variable is itself const or volatile.  */
	    constp = 0;
	    volatilep = 0;
#endif

	    type = build_array_type (type, itype);
	    ctype = NULL_TREE;
	  }
	  break;

	case CALL_EXPR:
	  {
	    tree arg_types;

	    /* Declaring a function type.
	       Make sure we have a valid type for the function to return.  */
	    /* Is this an error?  Should they be merged into TYPE here?  */
	    if (pedantic && (constp || volatilep))
	      warning ("function declared to return const or volatile result");

	    /* Warn about some types functions can't return.  */

	    if (TREE_CODE (type) == FUNCTION_TYPE)
	      {
		error ("`%s' declared as function returning a function", name);
		type = integer_type_node;
	      }
	    if (TREE_CODE (type) == ARRAY_TYPE)
	      {
		error ("`%s' declared as function returning an array", name);
		type = integer_type_node;
	      }

	    if (ctype == NULL_TREE
		&& (decl_context == FIELD
		    || decl_context == MEMFUNCDEF)
		&& (friendp == 0 || dname == current_class_name))
	      ctype = current_class_type;

	    if (ctype && DECL_NAME (TYPE_NAME (ctype)) == dname)
	      {
		/* We are within a class's scope. If our declarator name
		   is the same as the class name, and we are defining
		   a function, then it is a constructor/destructor, and
		   therefore returns a void type.  */

		if (flags == DTOR_FLAG)
		  {
		    if (specbits & (1 << RID_STATIC))
		      {
			error ("destructor cannot be static member function");
			specbits &= ~(1 << RID_STATIC);
		      }
		    if (type != void_type_node)
		      {
			error ("non-void return type specified for destructor");
			type = void_type_node;
		      }
		    else if (explicit_int != -1)
		      warning ("declaration of return type for destructor redundant");

		    if (decl_context == FIELD)
		      {
			if (! member_function_or_else (ctype, current_class_type,
						       "destructor for alien class `%s' cannot be a member"))
			  return NULL_TREE;
			if (TYPE_HAS_DESTRUCTOR (ctype))
			  {
			    error ("class `%s' already has destructor defined",
				   IDENTIFIER_POINTER (current_class_name));
			  }
			TYPE_HAS_DESTRUCTOR (ctype) = 1;
		      }
		  }
		else if (flags == WRAPPER_FLAG || flags == ANTI_WRAPPER_FLAG)
		  {
		    if (specbits & (1 << RID_STATIC))
		      {
			error ("wrapper cannot be static member function");
			specbits &= ~(1 << RID_STATIC);
		      }
		    if (decl_context == FIELD)
		      {
			if (! member_function_or_else (ctype, current_class_type,
						       "wrapper for alien class `%s' cannot be member"))
			  return NULL_TREE;
			TYPE_HAS_WRAPPER (ctype) = 1;
		      }
		  }
		else if (flags == WRAPPER_PRED_FLAG)
		  {
		    if (specbits & (1 << RID_STATIC))
		      {
			error ("wrapper predicate cannot be static member function");
			specbits &= ~(1 << RID_STATIC);
		      }
		    if (TREE_CODE (type) != INTEGER_TYPE)
		      {
			error ("wrapper predicated must return an integer type");
			type = integer_type_node;
		      }
		    if (decl_context == FIELD)
		      {
			if (! member_function_or_else (ctype, current_class_type,
						       "wrapper predicate for alien class `%s' cannot be member"))
			  return NULL_TREE;
			TYPE_HAS_WRAPPER_PRED (ctype) = 1;
		      }
		  }
		else            /* its a constructor. */
		  {
		    if (specbits & (1 << RID_STATIC))
		      {
			error ("constructor cannot be static member function");
			specbits &= ~(1 << RID_STATIC);
		      }
		    if (TYPE_MAIN_VARIANT (type) != TYPE_POINTER_TO (ctor_return_type))
		      {
			error ("invalid return type given for constructor");
			type = TYPE_POINTER_TO (ctor_return_type);
		      }
		    else if (explicit_int != -1)
		      warning ("declaration of return type for constructor redundant");
		    if (virtualp || friendp)
		      {
			error ("constructors cannot be declared virtual or friend");
			virtualp = 0;
			friendp = 0;
		      }
		    if (specbits & ~((1 << (int) RID_INLINE)|(1 << (int) RID_STATIC)))
		      error ("Return value type specifier for `%s' ignored",
			     flags == DTOR_FLAG ? "destructor" : "constructor");
		    type = TYPE_POINTER_TO (ctype);
		    if (decl_context == FIELD)
		      {
			if (! member_function_or_else (ctype, current_class_type,
						       "constructor for alien class `%s' cannot be member"))
			  return NULL_TREE;
			TYPE_HAS_CONSTRUCTOR (ctype) = 1;
			ctorp = 1;
		      }
		  }
	      }
	    else if (friendp && virtualp)
	      {
		/* Cannot be both friend and virtual.  */
		error ("virtual functions cannot be friends");
		specbits ^= (1 << (int) RID_FRIEND);
		friendp = 0;
	      }

	    /* Construct the function type and go to the next
	       inner layer of declarator.  */

	    {
	      int funcdef_p;
	      tree inner_parms = TREE_OPERAND (declarator, 1);
	      tree inner_decl = TREE_OPERAND (declarator, 0);

	      declarator = TREE_OPERAND (declarator, 0);

	      if (inner_decl && TREE_CODE (inner_decl) == SCOPE_REF)
		inner_decl = TREE_OPERAND (inner_decl, 1);

	      /* Say it's a definition only for the CALL_EXPR
		 closest to the identifier.  */
	      funcdef_p =
		(inner_decl &&
		 (TREE_CODE (inner_decl) == IDENTIFIER_NODE
		  || TREE_CODE (inner_decl) == OP_IDENTIFIER
		  || TREE_CODE (inner_decl) == TYPE_EXPR)) ? funcdef_flag : 0;

	      arg_types = grokparms (inner_parms, funcdef_p);
	    }

	    if (declarator)
	      {
		/* Get past destructors, wrappers, etc.
		   We know we have one because FLAGS will be non-zero.

		   Complain about improper parameter lists here.  */
		if (TREE_CODE (declarator) == BIT_NOT_EXPR)
		  {
		    declarator = TREE_OPERAND (declarator, 0);

		    if (strict_prototype == 0 && arg_types == NULL_TREE)
		      arg_types = void_list_node;
		    else if (arg_types == NULL_TREE
			     || TREE_VALUE (arg_types) != void_type_node)
		      {
			error ("destructors cannot be specified with parameters");
			arg_types = void_list_node;
		      }
		  }
		else if (TREE_CODE (declarator) == WRAPPER_EXPR)
		  {
		    /* Report misuse of wrappers and their associates.
		       Note that because wrappers may be invoked
		       quite a bit implicitly, if we give an error
		       message, we make an effort to fix that error
		       so that spurious errors do not show up.  */
		    if (TREE_CODE (TREE_OPERAND (declarator, 0)) == COND_EXPR)
		      {
			/* First parameter must be a pointer to a member function.
			   Rest of parameters must all be default parameters.  */
			if (arg_types == NULL_TREE
			    || TREE_VALUE (arg_types) == void_type_node
			    || TREE_CODE (TREE_VALUE (arg_types)) != POINTER_TYPE
			    || TREE_CODE (TREE_TYPE (TREE_VALUE (arg_types))) != METHOD_TYPE)
			  {
			    error ("wrapper predicate takes a pointer-to-member-function as first argument");
			    arg_types = NULL_TREE;
			  }
			else if (TREE_CHAIN (arg_types)
				 && TREE_VALUE (TREE_CHAIN (arg_types)) != void_type_node
				 && TREE_PURPOSE (TREE_CHAIN (arg_types)) == NULL_TREE)
			  {
			    error ("all arguments past first must be default for wrapper predicate");
			    TREE_CHAIN (arg_types) = NULL_TREE;
			  }
			declarator = wrapper_pred_name;
		      }
		    else
		      {
			/* First parameter must be an int.
			   Second parameter must be a pointer to a member function.  */
			if (arg_types == NULL_TREE || TREE_CHAIN (arg_types) == NULL_TREE)
			  {
			    error ("wrappers must have at least two arguments (int, pointer-to-member-function)");
			    arg_types = NULL_TREE;
			  }
			else
			  {
			    if (TREE_CODE (TREE_VALUE (arg_types)) != INTEGER_TYPE)
			      {
				error ("first argument to wrapper must be an integer");
				TREE_VALUE (arg_types) = integer_type_node;
			      }
			    if (TREE_CODE (TREE_VALUE (TREE_CHAIN (arg_types))) != POINTER_TYPE
				|| TREE_CODE (TREE_TYPE (TREE_VALUE (TREE_CHAIN (arg_types)))) != METHOD_TYPE)
			      {
				error ("second argument to wrapper must be a pointer-to-member-function type");
				TREE_CHAIN (arg_types) = NULL_TREE;
			      }
			  }
			declarator = wrapper_name;
		      }
		  }
		else if (TREE_CODE (declarator) == ANTI_WRAPPER_EXPR)
		  declarator = anti_wrapper_name;
	      }

	    type = build_function_type (type, flag_traditional ? 0 : arg_types);
	  }
	  break;

	case ADDR_EXPR:
	case INDIRECT_REF:
	  /* Filter out pointers-to-references and references-to-references.
	     We can get these if a TYPE_DECL is used.  */

	  if (TREE_CODE (type) == REFERENCE_TYPE)
	    {
	      error ("cannot declare %s to references",
		     TREE_CODE (declarator) == ADDR_EXPR
		     ? "references" : "pointers");
	      declarator = TREE_OPERAND (declarator, 0);
	      continue;
	    }

	  /* Merge any constancy or volatility into the target type
	     for the pointer.  */

	  if (constp || volatilep)
	    type = build_type_variant (type, constp, volatilep);
	  constp = 0;
	  volatilep = 0;

	  if (TREE_CODE (declarator) == ADDR_EXPR)
	    if (TREE_CODE (type) == FUNCTION_TYPE)
	      {
		error ("cannot declare references to functions; use pointer to function instead");
		type = build_pointer_type (type);
	      }
	    else
	      {
		if (TYPE_MAIN_VARIANT (type) == void_type_node)
		  error ("invalid type: `void &'");
		else
		  type = build_reference_type (type);
	      }
	  else
	    type = build_pointer_type (type);

	  /* Process a list of type modifier keywords (such as
	     const or volatile) that were given inside the `*' or `&'.  */

	  if (TREE_TYPE (declarator))
	    {
	      register tree typemodlist;
	      int erred = 0;
	      for (typemodlist = TREE_TYPE (declarator); typemodlist;
		   typemodlist = TREE_CHAIN (typemodlist))
		{
		  if (TREE_VALUE (typemodlist) == ridpointers[(int) RID_CONST])
		    constp++;
		  else if (TREE_VALUE (typemodlist) == ridpointers[(int) RID_VOLATILE])
		    volatilep++;
		  else if (!erred)
		    {
		      erred = 1;
		      error ("invalid type modifier within %s declarator",
			     TREE_CODE (declarator) == ADDR_EXPR
			     ? "reference" : "pointer");
		    }
		}
	      if (constp > 1)
		warning ("duplicate `const'");
	      if (volatilep > 1)
		warning ("duplicate `volatile'");
	    }
	  declarator = TREE_OPERAND (declarator, 0);
	  ctype = NULL_TREE;
	  break;

	case SCOPE_REF:
	  {
	    /* We have converted type names to NULL_TREE if the
	       name was bogus, or to a _TYPE node, if not.

	       The variable CTYPE holds the type we will ultimately
	       resolve to.  The code here just needs to build
	       up appropriate member types.  */
	    tree sname = TREE_OPERAND (declarator, 1);

	    if (TREE_OPERAND (declarator, 0) == NULL_TREE)
	      {
		/* If we were given a non-aggregate typedef,
		   then we cleared this out.  Just keep going
		   as though it wasn't there.  */
		declarator = sname;
		continue;
	      }
	    ctype = TREE_OPERAND (declarator, 0);

	    if (TREE_CODE (sname) == IDENTIFIER_NODE)
	      {
		/* This is the `standard' use of the scoping operator:
		   basetype :: member .  */

		if (ctype == current_class_type || friendp)
		  if (TREE_CODE (type) == FUNCTION_TYPE)
		    type = build_method_type (ctype, type);
		  else
		    type = build_member_type (ctype, type);
		else if (TYPE_SIZE (ctype) != 0
		    || (specbits & (1<<(int)RID_TYPEDEF)))
		  {
		    tree t;
		    /* have to move this code elsewhere in this function.
		       this code is used for i.e., typedef int A::M; M *pm; */

		    if (decl_context == FIELD)
		      {
			t = lookup_field (ctype, sname, 0);
			if (! t)
			  t = lookup_fnfields (CLASSTYPE_AS_LIST (ctype),
					       sname, 0);
			if (t)
			  {
			    t = build_lang_decl (FIELD_DECL, build_nt (SCOPE_REF, ctype, t), type);
			    DECL_INITIAL (t) = init;
			    return t;
			  }

			if (flags == TYPENAME_FLAG)
			  error_with_aggr_type (ctype, "type conversion is not a member of structure `%s'");
			else
			  error ("field `%s' is not a member of structure `%s'",
				 IDENTIFIER_POINTER (sname),
				 TYPE_NAME_STRING (ctype));
		      }
		    if (TREE_CODE (type) == FUNCTION_TYPE)
		      type = build_method_type (ctype, type);
		    else
		      type = build_member_type (ctype, type);
		  }
		else
		  sorry ("structure `%s' not yet defined",
			 TYPE_NAME_STRING (ctype));
		declarator = sname;
	      }
	    else if (TREE_CODE (sname) == TYPE_EXPR)
	      {
		/* A TYPE_EXPR will change types out from under us.
		   So do the TYPE_EXPR now, and make this SCOPE_REF
		   inner to the TYPE_EXPR's CALL_EXPR.

		   This does not work if we don't get a CALL_EXPR back.
		   I did not think about error recovery, hence the
		   assert (0).  */

		/* Get the CALL_EXPR.  */
		sname = grokoptypename (sname, 0);
		assert (TREE_CODE (sname) == CALL_EXPR);
		type = TREE_TYPE (TREE_OPERAND (sname, 0));
		/* Scope the CALL_EXPR's name.  */
		TREE_OPERAND (declarator, 1) = TREE_OPERAND (sname, 0);
		/* Put the SCOPE_EXPR in the CALL_EXPR's innermost position.  */
		TREE_OPERAND (sname, 0) = declarator;
		/* Now work from the CALL_EXPR.  */
		declarator = sname;
		continue;
	      }
	    else if (TREE_CODE (sname) == SCOPE_REF)
	      abort ();
	    else
	      {
		declarator = TREE_OPERAND (declarator, 1);
		if (TREE_CODE (declarator) == CALL_EXPR)
		  /* In this case, we will deal with it later.  */
		  ;
		else
		  {
		    if (TREE_CODE (type) == FUNCTION_TYPE)
		      type = build_method_type (ctype, type);
		    else
		      type = build_member_type (ctype, type);
		  }
	      }
	  }
	  break;

	case OP_IDENTIFIER:
	  /* This is exceptional, in that we must finalize a 
	     member type before calling grokopexpr, if we want
	     to use the declared type information to resolve
	     ambiguities.  Do not get fooled by friends,
	     which do not have a member type built for them
	     unless they were explicitly scoped (in which case that
	     will have been taken care of in the SCOPE_REF case.  */
	  if (TREE_CODE (type) == FUNCTION_TYPE
	      && ((specbits & (1 << (int) RID_STATIC)) == 0)
	      && (decl_context == FIELD && friendp == 0
		  || decl_context == NORMAL && ctype != 0))
	    type = build_method_type (ctype, type);
	  declarator = grokopexpr (declarator, type, 0, 1);
	  if (declarator == NULL_TREE)
	    return NULL_TREE;
	  name = IDENTIFIER_POINTER (declarator);
	  break;

	case BIT_NOT_EXPR:
	  declarator = TREE_OPERAND (declarator, 0);
	  break;

	case TYPE_EXPR:
	  declarator = grokoptypename (declarator, 0);
	  if (explicit_int != -1
	      && comp_target_types (type, TREE_TYPE (TREE_OPERAND (declarator, 0)), 1) == 0)
	    {
	      error ("type conversion function declared to return incongruent type");
	    }
	  type = TREE_TYPE (TREE_OPERAND (declarator, 0));
	  break;

	case WRAPPER_EXPR:
	  if (TREE_CODE (TREE_OPERAND (declarator, 0)) == COND_EXPR)
	    declarator = wrapper_pred_name;
	  else
	    declarator = wrapper_name;
	  break;

	case ANTI_WRAPPER_EXPR:
	  declarator = anti_wrapper_name;
	  break;

	default:
	  assert (0);
	}

      /* layout_type (type); */

      /* @@ Should perhaps replace the following code by changes in
       * @@ stor_layout.c. */
      if (TREE_CODE (type) == FUNCTION_DECL)
	{
	  /* A function variable in C should be Pmode rather than EPmode
	     because it has just the address of a function, no static chain.*/
	  TYPE_MODE (type) = Pmode;
	}
    }

  /* Now TYPE has the actual type.  */

  /* If this is declaring a typedef name, return a TYPE_DECL.  */

  if (specbits & (1 << (int) RID_TYPEDEF))
    {
      /* Note that the grammar rejects storage classes
	 in typenames, fields or parameters.  */
      if (constp || volatilep)
	type = build_type_variant (type, constp, volatilep);

      /* If the user declares "struct {...} foo" then `foo' will have
	 an anonymous name.  Fill that name in now.  Nothing can
	 refer to it, so nothing needs know about the name change.
	 The TYPE_NAME field was filled in by build_struct_xref.  */
      if (IS_AGGR_TYPE (type)
	  && TREE_CODE (TYPE_NAME (type)) == TYPE_DECL
	  && ANON_AGGRNAME_P (DECL_NAME (TYPE_NAME (type))))
	{
	  /* replace the anonymous name with the real name everywhere.  */
	  lookup_tag_reverse (type, declarator);
	  DECL_NAME (TYPE_NAME (type)) = declarator;
	}

      if (resume_temporary)
	resume_temporary_allocation ();
      return build_decl (TYPE_DECL, declarator, type);
    }

  /* Detect the case of an array type of unspecified size
     which came, as such, direct from a typedef name.
     We must copy the type, so that each identifier gets
     a distinct type, so that each identifier's size can be
     controlled separately by its own initializer.  */

  if (type == typedef_type && TREE_CODE (type) == ARRAY_TYPE
      && TYPE_DOMAIN (type) == 0)
    {
      type = build_array_type (TREE_TYPE (type), TYPE_DOMAIN (type));
    }

  /* If this is a type name (such as, in a cast or sizeof),
     compute the type and return it now.  */

  if (decl_context == TYPENAME)
    {
      /* Note that the grammar rejects storage classes
	 in typenames, fields or parameters.  */
      if (constp || volatilep)
	type = build_type_variant (type, constp, volatilep);

      /* Special case: "friend class foo" looks like a TYPENAME context.  */
      if (friendp)
	{
	  /* A friendly class?  */
	  make_friend_class (current_class_type, TYPE_MAIN_VARIANT (type));
	  type = void_type_node;
	}

      if (resume_temporary)
	resume_temporary_allocation ();
      return type;
    }

  /* `void' at top level (not within pointer)
     is allowed only in typedefs or type names.
     We don't complain about parms either, but that is because
     a better error message can be made later.  */

  if (type == void_type_node && decl_context != PARM)
    {
      if (declarator != NULL_TREE
	  && TREE_CODE (declarator) == IDENTIFIER_NODE)
	error ("variable or field `%s' declared void",
	       IDENTIFIER_POINTER (declarator));
      else
	error ("variable or field declared void");
      type = integer_type_node;
    }

  /* Now create the decl, which may be a VAR_DECL, a PARM_DECL
     or a FUNCTION_DECL, depending on DECL_CONTEXT and TYPE.  */

  {
    register tree decl;

    if (decl_context == PARM)
      {
	if (ctype)
	  error ("cannot use `::' in parameter declaration");

	/* A parameter declared as an array of T is really a pointer to T.
	   One declared as a function is really a pointer to a function.
	   One declared as a member is really a pointer to member.  */

	if (TREE_CODE (type) == ARRAY_TYPE)
	  {
	    /* Transfer const-ness of array into that of type pointed to.  */
	    type = build_pointer_type
		    (build_type_variant (TREE_TYPE (type), constp, volatilep));
	    volatilep = constp = 0;
	  }
	else if (TREE_CODE (type) == FUNCTION_TYPE)
	  type = build_pointer_type (type);
	else if (TREE_CODE (type) == OFFSET_TYPE)
	  type = build_pointer_type (type);

	decl = build_decl (PARM_DECL, declarator, type);

	/* Compute the type actually passed in the parmlist,
	   for the case where there is no prototype.
	   (For example, shorts and chars are passed as ints.)
	   When there is a prototype, this is overridden later.  */

	DECL_ARG_TYPE (decl) = type;
	if (type == float_type_node)
	  DECL_ARG_TYPE (decl) = double_type_node;
	else if (TREE_CODE (type) == INTEGER_TYPE
		 && TYPE_PRECISION (type) < TYPE_PRECISION (integer_type_node))
	  DECL_ARG_TYPE (decl) = integer_type_node;

	if (virtualp)
	  {
	    error ("virtual non function `%s'", IDENTIFIER_POINTER (declarator));
	    virtualp = 0;
	  }
      }
    else if (decl_context == FIELD)
      {
	if (TREE_CODE (type) == FUNCTION_TYPE)
	  {
	    if (current_lang_name == lang_name_c)
	      {
		error ("field `%s' declared as a function in %s language context",
		       IDENTIFIER_POINTER (declarator),
		       IDENTIFIER_POINTER (current_lang_name));
		type = build_pointer_type (type);
	      }
	    else
	      {
		if (friendp == 0)
		  {
		    if (ctype == NULL_TREE)
		      ctype = current_class_type;
		    if ((specbits & (1 << (int) RID_STATIC)) == 0)
		      type = build_method_type (ctype, type);
		  }
		decl = build_lang_decl (FUNCTION_DECL, declarator, type);
		if (ctype != NULL_TREE && TREE_CODE (type) == FUNCTION_TYPE)
		  {
		    DECL_STATIC_FUNCTION_P (decl) = 1;
		    DECL_STATIC_CONTEXT (decl) = ctype;
		  }
		TREE_INLINE (decl) = inlinep;
	      }
	  }
	else if (TREE_CODE (type) == METHOD_TYPE)
	  {
	    decl = build_lang_decl (FUNCTION_DECL, declarator, type);
	    TREE_INLINE (decl) = inlinep;
	  }
	else if (TYPE_SIZE (type) == 0)
	  {
	    if (declarator)
	      error ("field `%s' has incomplete type",
		     IDENTIFIER_POINTER (declarator));
	    else
	      error ("field has incomplete type");
	    type = error_mark_node;
	    decl = NULL_TREE;
	  }
	else
	  {
	    if (friendp)
	      {
		error ("`%s' is neither function nor method; cannot be declared friend",
		       IDENTIFIER_POINTER (declarator));
		friendp = 0;
	      }
	    decl = NULL_TREE;
	  }

	if (decl)
	  TREE_EXTERNAL (decl) = 1;

	if (friendp)
	  {
	    /* Friends are treated specially.  */
	    if (ctype == current_class_type)
	      warning ("member functions are implicitly friends of their class");
	    else
	      {
		tree cname = ctype ? DECL_NAME (TYPE_NAME (ctype)) : NULL_TREE;
		if (OPERATOR_NAME_P (DECL_NAME (decl)))
		  {
		    TREE_OPERATOR (decl) = 1;
		    grok_op_properties (decl);
		  }

		return do_friend (cname, declarator, decl, last_function_parms, flags);
	      }
	  }

	/* Structure field.  It may not be a function, except for C++ */

	if (ctorp)
	  {
	    tree tmp;
	    /* Just handle constructors here.  We could do this
	       inside the following if stmt, but I think
	       that the code is more legible by breaking this
	       case out.  See comments below for what each of
	       the following calls is supposed to do.  */
	    DECL_CONSTRUCTOR_P (decl) = 1;
	    grokclassfn (current_class_name, decl, flags, 0);
	    grok_ctor_properties (decl);
	    tmp = lookup_name (DECL_NAME (decl));
	    if (tmp == 0)
	      IDENTIFIER_GLOBAL_VALUE (DECL_NAME (decl)) = decl;
	    else if (TREE_CODE (tmp) != TREE_CODE (decl))
	      error_with_decl (decl, "inconsistant declarations for `%s'");
	    else
	      {
		duplicate_decls (decl, tmp);
		decl = tmp;
	      }
	    make_decl_rtl (decl, NULL_TREE, 1);
	  }
	else if (decl)
	  {
	    tree tmp;
	    int i;

	    /* Function gets the ugly name, field gets the nice one.
	       This call may change the type of the function (because
	       of default parameters)!

	       Wrappers get field names which will not conflict
	       with constructors and destructors.  */
	    grokclassfn (current_class_name, decl, flags, 0);

	    if (OPERATOR_NAME_P (DECL_NAME (decl)))
	      {
		TREE_OPERATOR (decl) = 1;
		grok_op_properties (decl);
	      }

	    /* Now install the declaration of this function so that
	       others may find it (esp. its DECL_FRIENDLIST).
	       Pretend we are at top level, we will get true
	       reference later, perhaps.  */
	    tmp = lookup_name (DECL_NAME (decl));
	    if (tmp == 0)
	      IDENTIFIER_GLOBAL_VALUE (DECL_NAME (decl)) = decl;
	    else if (TREE_CODE (tmp) != TREE_CODE (decl))
	      error_with_decl (decl, "inconsistant declarations for `%s'");
	    else
	      {
		duplicate_decls (decl, tmp);
		decl = tmp;
	      }
	    make_decl_rtl (decl, NULL_TREE, 1);

	    /* If this declaration supersedes the declaration of
	       a method declared virtual in the base class, then
	       mark this field as being virtual as well.  */
	    for (i = 1; i <= CLASSTYPE_N_BASECLASSES (ctype); i++)
	      if (TYPE_VIRTUAL_P (CLASSTYPE_BASECLASS (ctype, i))
		  || flag_all_virtual == 1)
		{
		  tmp = get_first_matching_virtual (CLASSTYPE_BASECLASS (ctype, i),
						    decl,
						    flags == DTOR_FLAG);
		  if (tmp)
		    {
		      virtualp = 1;
		      SET_DECL_VINDEX (decl, tmp);
		      DECL_VOFFSET (decl) = TREE_TYPE (tmp);
		      break;
		    }
		}
	    if (DECL_VINDEX (decl) == NULL_TREE && ctorp == 0
		&& (virtualp || flag_all_virtual))
	      CLASSTYPE_VSIZE (ctype) += 1;
	  }
	else if (specbits & (1 << (int) RID_STATIC))
	  {
	    /* C++ allows static class members.
	       All other work for this is done by grokfield.
	       This VAR_DECL is built by build_lang_decl.
	       All other VAR_DECLs are built by build_decl.  */
	    if (current_lang_name == lang_name_c)
	      error ("structure field `%s' declared static in %s language context",
		     IDENTIFIER_POINTER (declarator),
		     IDENTIFIER_POINTER (current_lang_name));
	    decl = build_lang_decl (VAR_DECL, declarator, type);
	    TREE_STATIC (decl) = 1;
	    if (virtualp)
	      {
		error_with_decl (decl, "virtual non function declaration for static field `%s'");
		virtualp = 0;
	      }
	    if (friendp)
	      {
		error_with_decl (decl, "static field `%s' cannot be declared friend");
		friendp = 0;
	      }
	  }
	else
	  {
	    decl = build_lang_decl (FIELD_DECL, declarator, type);
	    if (virtualp)
	      {
		error_with_decl (decl, "virtual non function declaration for field `%s'");
		virtualp = 0;
	      }
	  }

	if (specbits & (1 << (int) RID_STATIC)
	    && TREE_CODE (decl) != FUNCTION_DECL)
	  TREE_PUBLIC (type) = 0;
      }
    else if (TREE_CODE (type) == FUNCTION_TYPE || TREE_CODE (type) == METHOD_TYPE)
      {
	int was_overloaded = 0;
	tree original_name = declarator;

	if (! declarator) return NULL_TREE;

	if (specbits & ((1 << (int) RID_AUTO) | (1 << (int) RID_REGISTER)))
	  error ("invalid storage class for function `%s'",
		 IDENTIFIER_POINTER (declarator));
	/* Function declaration not at top level.
	   Storage classes other than `extern' are not allowed
	   and `extern' makes no difference.  */
	if (current_binding_level != global_binding_level
	    && (specbits & ((1 << (int) RID_STATIC) | (1 << (int) RID_INLINE)))
	    && pedantic)
	  warning ("invalid storage class for function `%s'",
		   IDENTIFIER_POINTER (declarator));

	if (ctype == NULL_TREE)
	  {
	    if (virtualp)
	      {
		error ("virtual non-class function `%s'",
		       IDENTIFIER_POINTER (declarator));
		virtualp = 0;
	      }
#ifdef NO_AUTO_OVERLOAD
	    if (is_overloaded (declarator))
	      {
		/* Plain overloading: will not be grok'd by grokclassfn.  */
		if (current_lang_name == lang_name_cplusplus)
		  declarator = build_decl_overload (name, TYPE_ARG_TYPES (type));
		was_overloaded = 1;
	      }
#else
	    if (current_lang_name == lang_name_cplusplus
		&& ! (IDENTIFIER_LENGTH (original_name) == 4
		      && IDENTIFIER_POINTER (original_name)[0] == 'm'
		      && strcmp (IDENTIFIER_POINTER (original_name), "main") == 0)
		&& ! (IDENTIFIER_LENGTH (original_name) > 10
		      && IDENTIFIER_POINTER (original_name)[0] == '_'
		      && IDENTIFIER_POINTER (original_name)[1] == '_'
		      && strncmp (IDENTIFIER_POINTER (original_name)+2, "builtin_", 8) == 0))
	      {
		/* Plain overloading: will not be grok'd by grokclassfn.  */
		declarator = build_decl_overload (name, TYPE_ARG_TYPES (type));
		was_overloaded = 1;
	      }
#endif
	  }
	else if (TREE_CODE (type) == FUNCTION_TYPE
		 && ((specbits & (1 << (int) RID_STATIC)) == 0))
	  type = build_method_type (ctype, type);

	decl = build_lang_decl (FUNCTION_DECL, declarator, type);
	if (ctype != NULL_TREE && TREE_CODE (type) == FUNCTION_TYPE)
	  {
	    DECL_STATIC_FUNCTION_P (decl) = 1;
	    DECL_STATIC_CONTEXT (decl) = ctype;
	  }
	TREE_EXTERNAL (decl) = 1;
	/* Record presence of `static'.  In C++, `inline' is like `static'.  */
	TREE_PUBLIC (decl) = !(specbits & ((1 << (int) RID_STATIC) | (1 << (int) RID_INLINE)));
	/* Record presence of `inline', if it is reasonable.  */
	if (inlinep)
	  {
	    tree last = tree_last (TYPE_ARG_TYPES (type));

	    if (! was_overloaded
		&& ! ctype
		&& ! strcmp (IDENTIFIER_POINTER (original_name), "main"))
	      warning ("cannot inline function `main'");
	    else if (last && TREE_VALUE (last) != void_type_node)
	      warning ("inline declaration ignored for function with `...'");
	    else
	      /* Assume that otherwise the function can be inlined.  */
	      TREE_INLINE (decl) = 1;
	  }
	if (was_overloaded)
	  {
	    TREE_OVERLOADED (decl) = 1;
	    DECL_ORIGINAL_NAME (decl) = original_name;
	  }

	/* must do this because start_function won't.  */
	if (ctype)
	  {
	    tree cname = DECL_NAME (TYPE_NAME (ctype));
	    /* This is a top level definition of a method.
	       Must set flag if this is a constructor.  */
	    if (cname == DECL_ORIGINAL_NAME (decl)
		&& flags == NO_SPECIAL)
	      {
		DECL_CONSTRUCTOR_P (decl) = 1;
		grok_ctor_properties (decl);
	      }

	    grokclassfn (cname, decl, flags, 1);
	  }

	if (OPERATOR_NAME_P (DECL_NAME (decl)))
	  {
	    TREE_OPERATOR (decl) = 1;
	    grok_op_properties (decl);
	  }
      }
    else
      {
	/* It's a variable.  */

	if (TREE_CODE (type) == OFFSET_TYPE)
	  {
	    /* If you declare a static member so that it
	       can be initialized, the code will reach here.  */
	    tree field = lookup_field (TYPE_OFFSET_BASETYPE (type),
				       declarator, 0);
	    if (field == NULL_TREE || TREE_CODE (field) != VAR_DECL)
	      {
		error ("`%s' is not a static member of class `%s'",
		       IDENTIFIER_POINTER (declarator),
		       TYPE_NAME_STRING (TYPE_OFFSET_BASETYPE (type)));
		type = TREE_TYPE (type);
		decl = build_decl (VAR_DECL, declarator, type);
	      }
	    else decl = field;
	  }
	else decl = build_decl (VAR_DECL, declarator, type);

	if (inlinep)
	  warning_with_decl (decl, "variable `%s' declared `inline'");

	/* An uninitialized decl with `extern' is a reference.  */
	TREE_EXTERNAL (decl)
	  = !initialized && (specbits & (1 << (int) RID_EXTERN));

	/* In class context, static means one per class,
	   public visibility, and static storage.  */
	if (DECL_FIELD_CONTEXT (decl) != 0
	    && IS_AGGR_TYPE (DECL_FIELD_CONTEXT (decl)))
	  {
	    TREE_PUBLIC (decl) = 1;
	    TREE_STATIC (decl) = 1;
	  }
	/* At top level, either `static' or no s.c. makes a definition
	   (perhaps tentative), and absence of `static' makes it public.  */
	else if (current_binding_level == global_binding_level)
	  {
	    TREE_PUBLIC (decl) = !(specbits & (1 << (int) RID_STATIC));
	    TREE_STATIC (decl) = ! TREE_EXTERNAL (decl);
	  }
	/* Not at top level, only `static' makes a static definition.  */
	else
	  {
	    TREE_STATIC (decl) = (specbits & (1 << (int) RID_STATIC)) != 0;
	    TREE_PUBLIC (decl) = TREE_EXTERNAL (decl);
	    /* `extern' with initialization is invalid if not at top level.  */
	    if ((specbits & (1 << (int) RID_EXTERN)) && initialized)
	      error ("`%s' has both `extern' and initializer", name);
	  }
      }

    /* Record `register' declaration for warnings on &
       and in case doing stupid register allocation.  */

    if (specbits & (1 << (int) RID_REGISTER))
      TREE_REGDECL (decl) = 1;

    /* Record constancy and volatility.  */

    if (constp)
      TREE_READONLY (decl) = 1;
    if (volatilep)
      {
	TREE_VOLATILE (decl) = 1;
	TREE_THIS_VOLATILE (decl) = 1;
      }
    if (virtualp)
      DECL_VIRTUAL_P (decl) = 1;
    if (friendp)
      error ("invalid friend declaration");

    if (resume_temporary)
      resume_temporary_allocation ();

    return decl;
  }
}

/* Classes overload their constituent function names automatically.
   When a function name is declared in a record structure,
   its name is changed to it overloaded name.  Since names for
   constructors and destructors can conflict, we place a leading
   '$' for destructors.

   CNAME is the name of the class we are grokking for.

   FUNCTION is a FUNCTION_DECL.  It was created by `grokdeclarator'.

   FLAGS contains bits saying what's special about today's
   arguments.  1 == DESTRUCTOR.  2 == OPERATOR.

   If FUNCTION is a destructor, then we must add the `auto-delete' field
   as a second parameter.  There is some hair associated with the fact
   that we must "declare" this variable in the manner consistent with the
   way the rest of the arguements were declared.

   If FUNCTION is a constructor, and we are doing SOS hacks for dynamic
   classes, then the second hidden argument is the virtual function table
   pointer with which to initialize the object.  */

void
grokclassfn (cname, function, flags, complain)
     tree cname, function;
     enum overload_flags flags;
{
  tree fn_name = DECL_NAME (function);
  tree fntype = TREE_TYPE (function);
  tree arg_types = TYPE_ARG_TYPES (fntype);
  tree type, parm;
  tree ctype = TREE_TYPE (TREE_TYPE (cname));
  char *name;

  if (fn_name == NULL_TREE)
    {
      error ("name missing for member function");
      fn_name = get_identifier ("<anonymous>");
      DECL_NAME (function) = DECL_ORIGINAL_NAME (function) = fn_name;
    }

  if (TREE_CODE (fntype) == METHOD_TYPE)
    {
      /* Must add the class instance variable up front.  */
      /* Right now we just make this a pointer.  But later
	 we may wish to make it special.  */
      type = TYPE_POINTER_TO (ctype);

      parm = build_decl (PARM_DECL, get_identifier (THIS_NAME), type);
      DECL_ARG_TYPE (parm) = type;
      /* We can make this a register, so long as we don't
	 accidently complain if someone tries to take its address.  */
      TREE_REGDECL (parm) = 1;
      TREE_CHAIN (parm) = last_function_parms;
      last_function_parms = parm;
    }

  if (flags == DTOR_FLAG)
    {
      tree const_integer_type = build_type_variant (integer_type_node, 1, 0);
      arg_types = tree_cons (NULL_TREE, const_integer_type, void_list_node);
      name = (char *)alloca (sizeof (DESTRUCTOR_DECL_FORMAT)
			     + IDENTIFIER_LENGTH (cname) + 2);
      sprintf (name, DESTRUCTOR_DECL_FORMAT, IDENTIFIER_POINTER (cname));
      DECL_NAME (function) = get_identifier (name);

      parm = build_decl (PARM_DECL, get_identifier (AUTO_DELETE_NAME), const_integer_type);
      TREE_USED (parm) = 1;
      TREE_READONLY (parm) = 1;
      DECL_ARG_TYPE (parm) = const_integer_type;
      /* This is the same chain as DECL_ARGUMENTS (fndecl).  */
      TREE_CHAIN (last_function_parms) = parm;

      fntype = build_function_type (void_type_node, arg_types);
      TREE_TYPE (function) = build_method_type (ctype, fntype);
    }
  else if (flags == WRAPPER_FLAG || flags == ANTI_WRAPPER_FLAG)
    {
      name = (char *)alloca (sizeof (WRAPPER_DECL_FORMAT)
			     + sizeof (ANTI_WRAPPER_DECL_FORMAT)
			     + IDENTIFIER_LENGTH (cname) + 2);
      sprintf (name,
	       flags == WRAPPER_FLAG ? WRAPPER_DECL_FORMAT : ANTI_WRAPPER_DECL_FORMAT,
	       IDENTIFIER_POINTER (cname));
      DECL_NAME (function) = build_decl_overload (name, arg_types);
      sprintf (name, flags == WRAPPER_FLAG ? WRAPPER_NAME_FORMAT : ANTI_WRAPPER_NAME_FORMAT,
	       IDENTIFIER_POINTER (cname));
      DECL_ORIGINAL_NAME (function) = fn_name = get_identifier (name);
    }
  else if (flags == WRAPPER_PRED_FLAG)
    {
      name = (char *)alloca (sizeof (WRAPPER_PRED_DECL_FORMAT)
			     + sizeof (WRAPPER_PRED_NAME_FORMAT)
			     + IDENTIFIER_LENGTH (cname) + 2);
      sprintf (name, WRAPPER_PRED_DECL_FORMAT, IDENTIFIER_POINTER (cname));
      DECL_NAME (function) = build_decl_overload (name, arg_types);
      sprintf (name, WRAPPER_PRED_NAME_FORMAT, IDENTIFIER_POINTER (cname));
      DECL_ORIGINAL_NAME (function) = fn_name = get_identifier (name);
    }
#ifdef SOS
  else if (TYPE_DYNAMIC (ctype) && DECL_CONSTRUCTOR_P (function))
    {
      arg_types = tree_cons (NULL_TREE, build_pointer_type (ptr_type_node),
			     TREE_CHAIN (arg_types));
      parm = build_decl (PARM_DECL, get_identifier (AUTO_VTABLE_NAME), TYPE_POINTER_TO (ptr_type_node));
      TREE_USED (parm) = 1;
      TREE_READONLY (parm) = 1;
      DECL_ARG_TYPE (parm) = TYPE_POINTER_TO (ptr_type_node);
      TREE_CHAIN (parm) = TREE_CHAIN (last_function_parms);
      TREE_CHAIN (last_function_parms) = parm;
      fntype = build_function_type (TREE_TYPE (TREE_TYPE (fntype)), arg_types);
      /* This will insert the class member type for us.  */
      TREE_TYPE (function) = build_method_type (ctype, fntype);
      arg_types = TYPE_ARG_TYPES (TREE_TYPE (function));
      DECL_NAME (function) = build_decl_overload (IDENTIFIER_POINTER (fn_name),
						  arg_types);
    }
#endif
  else
    {
      tree these_arg_types = arg_types;
      if (TREE_CODE (fntype) == FUNCTION_TYPE)
	these_arg_types = tree_cons (NULL_TREE, TYPE_POINTER_TO (ctype),
				     arg_types);
      DECL_NAME (function) = build_decl_overload (IDENTIFIER_POINTER (fn_name),
						  these_arg_types);
      if (flags == TYPENAME_FLAG)
	TREE_TYPE (DECL_NAME (function)) = TREE_TYPE (fn_name);
    }

  DECL_ARGUMENTS (function) = last_function_parms;

  /* now, the sanity check: report error if this function is not
     really a member of the class it is supposed to belong to.  */
  if (complain)
    {
      tree field, fields;
      int need_quotes = 0;
      char *err_name;

      fields = CLASSTYPE_FN_FIELDS (ctype);
      while (fields)
	{
	  if (fn_name == TREE_PURPOSE (fields))
	    {
	      field = TREE_VALUE (fields);
	      while (field)
		{
		  if (DECL_NAME (function) == DECL_NAME (field))
		    return;
		  field = TREE_CHAIN (field);
		}
	      break;            /* loser */
	    }
	  fields = TREE_CHAIN (fields);
	}

      if (OPERATOR_NAME_P (fn_name))
	{
	  err_name = (char *)alloca (1024);
	  sprintf (err_name, "`operator %s'", operator_name_string (fn_name));
	}
      else if (OPERATOR_TYPENAME_P (fn_name))
	if (complain && TYPE_HAS_CONVERSION (ctype))
	  err_name = "such type conversion operator";
	else
	  err_name = "type conversion operator";
      else if (flags == WRAPPER_FLAG)
	err_name = "wrapper";
      else if (flags == WRAPPER_PRED_FLAG)
	err_name = "wrapper predicate";
      else
	{
	  err_name = IDENTIFIER_POINTER (fn_name);
	  need_quotes = 1;
	}

      if (fields)
	if (need_quotes)
	  error ("argument list for `%s' does not match any in class", err_name);
	else
	  error ("argument list for %s does not match any in class", err_name);
      else if (need_quotes)
	error ("no `%s' member function declared in class", err_name);
      else
	error ("no %s declared in class", err_name);

      /* If we did not find the method in the class, add it to
	 avoid spurious errors.  */
      add_method (ctype, fields, function);
    }
}

/* The precedence rules of this grammar (or any other deterministic LALR
   grammar, for that matter), place the CALL_EXPR somewhere where we
   may not want it.  The solution is to grab the first CALL_EXPR we see,
   pretend that that is the one that belongs to the parameter list of
   the type conversion function, and leave everything else alone.
   We pull it out in place.

   CALL_REQUIRED is non-zero if we should complain if a CALL_EXPR
   does not appear in DECL.  */
tree
grokoptypename (decl, call_required)
     tree decl;
     int call_required;
{
  tree tmp, last;
  tree parms;

  assert (TREE_CODE (decl) == TYPE_EXPR);

  tmp = TREE_OPERAND (decl, 0);
  last = NULL_TREE;

  while (tmp)
    {
      switch (TREE_CODE (tmp))
	{
	case CALL_EXPR:
	  {
	    tree parms = TREE_OPERAND (tmp, 1);

	    if (last)
	      TREE_OPERAND (last, 0) = TREE_OPERAND (tmp, 0);
	    else
	      TREE_OPERAND (decl, 0) = TREE_OPERAND (tmp, 0);
	    if (parms
		&& TREE_CODE (TREE_VALUE (parms)) == TREE_LIST)
	      TREE_VALUE (parms)
		= grokdeclarator (TREE_VALUE (TREE_VALUE (parms)),
				  TREE_PURPOSE (TREE_VALUE (parms)),
				  TYPENAME, 0);
	    if (parms)
	      if (TREE_VALUE (parms) != void_type_node)
		{
		  error ("operator <typename> requires empty parameter list");
		  TREE_OPERAND (tmp, 1) = void_list_node;
		}
	      else
		/* If user specifies `void' explicitly, avoid having
		   two voids, since we put one invisible one on.  */
		TREE_CHAIN (parms) = NULL_TREE;

	    last = grokdeclarator (TREE_OPERAND (decl, 0),
				   TREE_TYPE (decl),
				   TYPENAME, 0);
	    TREE_OPERAND (tmp, 0) = build_typename_overload (last);
	    TREE_TYPE (TREE_OPERAND (tmp, 0)) = last;
	    return tmp;
	  }

	case INDIRECT_REF:
	case ADDR_EXPR:
	case ARRAY_REF:
	  break;

	case SCOPE_REF:
	  /* This is legal when declaring a conversion to
	     something of type pointer-to-member.  */
	  if (TREE_CODE (TREE_OPERAND (tmp, 1)) == INDIRECT_REF)
	    {
	      tmp = TREE_OPERAND (tmp, 1);
	    }
	  else
	    {
#if 0
	      /* We may need to do this if grokdeclarator cannot handle this.  */
	      error ("type `member of class %s' invalid return type",
		     TYPE_NAME_STRING (TREE_OPERAND (tmp, 0)));
	      TREE_OPERAND (tmp, 1) = build_nt (INDIRECT_REF, TREE_OPERAND (tmp, 1));
#endif
	      tmp = TREE_OPERAND (tmp, 1);
	    }
	  break;

	default:
	  assert (0);
	}
      last = tmp;
      tmp = TREE_OPERAND (tmp, 0);
    }

  if (call_required)
    error ("operator <typename> construct requires parameter list");

  last = grokdeclarator (TREE_OPERAND (decl, 0),
			 TREE_TYPE (decl),
			 TYPENAME, 0);
  tmp = build_nt (CALL_EXPR, build_typename_overload (last),
		  void_list_node, 0);
  TREE_TYPE (TREE_OPERAND (tmp, 0)) = last;
  return tmp;
}

/* Given an encoding for an operator name (see parse.y, the rules
   for making an `operator_name' in the variable DECLARATOR,
   return the name of the operator prefix as an IDENTIFIER_NODE.

   TYPE, if nonnull, is the function type for this declarator.
   This information helps to resolve potential ambiguities.

   If REPORT_AMBIGUOUS is non-zero, an error message
   is reported, and a default arity of the operator is
   returned.  Otherwise, return the operator under an OP_EXPR,
   for later evaluation when type information will enable
   proper instantiation.

   IS_DECL is non zero if this is a decl (as opposed to an
   expression).  */
tree
grokopexpr (declarator, type, report_ambiguous, is_decl)
     tree declarator;
     tree type;
     int report_ambiguous;
{
  tree name, parmtypes;
  int seen_classtype_parm = (type != NULL_TREE
			     && TREE_CODE (type) == METHOD_TYPE);

  if (type != NULL_TREE)
    {
      /* Now we know the number of parameters,
	 so build the real operator fnname.  */
      name = build_operator_fnname (declarator, TYPE_ARG_TYPES (type), 0);
    }
  else
    {
      if (TREE_PURPOSE (declarator) == NULL_TREE)
	switch (TREE_CODE (TREE_VALUE (declarator)))
	  {
	  case PLUS_EXPR:
	  case CONVERT_EXPR:
	  case ADDR_EXPR:
	  case BIT_AND_EXPR:
	  case INDIRECT_REF:
	  case MULT_EXPR:
	  case NEGATE_EXPR:
	  case MINUS_EXPR:
	    if (report_ambiguous)
	      {
		error ("operator '%s' ambiguous, (default binary)",
		       opname_tab[(int)TREE_CODE (TREE_VALUE (declarator))]);
		name = build_operator_fnname (declarator, NULL_TREE, 2);
	      }
	    else
	      {
		/* do something intellegent.  */
		TREE_TYPE (declarator) = unknown_type_node;
		return declarator;
	      }
	    break;
	  default:
	    name = build_operator_fnname (declarator, NULL_TREE, -1);
	    break;
	  }
      else if (TREE_CODE (TREE_PURPOSE (declarator)) == MODIFY_EXPR)
	name = build_operator_fnname (declarator, NULL_TREE, -1);
      else abort ();
    }
  /* Now warn if the parameter list does not contain any args
     which are of aggregate type.  */
  if (is_decl && type != NULL_TREE && ! seen_classtype_parm)
    for (parmtypes = TYPE_ARG_TYPES (type);
	 parmtypes;
	 parmtypes = TREE_CHAIN (parmtypes))
      if (IS_AGGR_TYPE (TREE_VALUE (parmtypes))
	  || (TREE_CODE (TREE_VALUE (parmtypes)) == REFERENCE_TYPE
	      && IS_AGGR_TYPE (TREE_TYPE (TREE_VALUE (parmtypes)))))
	{
	  seen_classtype_parm = 1;
	  break;
	}

  if (is_decl && seen_classtype_parm == 0)
    if (TREE_CODE (declarator) == OP_IDENTIFIER
	&& (TREE_CODE (TREE_OPERAND (declarator, 0)) == NEW_EXPR
	    || TREE_CODE (TREE_OPERAND (declarator, 0)) == DELETE_EXPR))
      /* Global operators new and delete are not overloaded.  */
      TREE_OVERLOADED (name) = 0;
    else
      error ("operator has no %suser-defined argument type",
	     type == NULL_TREE ? "(default) " : "");

  return name;
}

tree
grok_enum_decls (type, decl)
     tree type, decl;
{
  tree tag = value_member (type, CLASSTYPE_TAGS (current_class_type));
  tree values;

  if (tag == NULL_TREE)
    abort ();
  if (TREE_ADDRESSABLE (tag))
    return NULL_TREE;
  TREE_ADDRESSABLE (tag) = 1;
  values = TYPE_VALUES (type);
  while (values)
    {
      /* Create a declaration for the enum value name.  */
      tree next = build_lang_decl (CONST_DECL,
				   TREE_PURPOSE (values), type);
      DECL_INITIAL (next) = TREE_VALUE (values);
      TREE_CHAIN (next) = decl;
      decl = next;
      values = TREE_CHAIN (values);
    }
  return decl;
}

/* Tell if a parmlist/exprlist looks like an exprlist or a parmlist.
   An empty exprlist is a parmlist.  An exprlist which
   contains only identifiers at the global level
   is a parmlist.  Otherwise, it is an exprlist. */
static int
parmlist_is_exprlist (exprs)
     tree exprs;
{
  tree decl;

  if (exprs == NULL_TREE || TREE_PARMLIST (exprs))
    return 0;

  if (current_binding_level == global_binding_level)
    {
      /* At the global level, if these are all identifiers,
	 then it is a parmlist.  */
      while (exprs)
	{
	  if (TREE_CODE (TREE_VALUE (exprs)) != IDENTIFIER_NODE)
	    return 1;
	  exprs = TREE_CHAIN (exprs);
	}
      return 0;
    }
  return 1;
}

/* Make sure that the this list of PARMS has a chance of being
   grokked by `grokparms'.

   @@ This is really weak, but the grammar does not allow us
   @@ to easily reject things that this has to catch as syntax errors.  */
static int
parmlist_is_random (parms)
     tree parms;
{
  if (parms == NULL_TREE)
    return 0;

  if (TREE_CODE (parms) != TREE_LIST)
    return 1;

  while (parms)
    {
      if (TREE_VALUE (parms) == void_type_node)
	return 0;

      if (TREE_CODE (TREE_VALUE (parms)) != TREE_LIST)
	return 1;
      parms = TREE_CHAIN (parms);
    }
  return 0;
}

/* Subroutine of `grokparms'.  In a fcn definition, arg types must
   be complete.

   C++: also subroutine of `start_function'.  */
static void
require_complete_types_for_parms (parms)
     tree parms;
{
  while (parms)
    {
      tree type = TREE_TYPE (parms);
      if (TYPE_SIZE (type) == 0)
	{
	  if (DECL_NAME (parms))
	    error ("parameter `%s' has incomplete type",
		   IDENTIFIER_POINTER (DECL_NAME (parms)));
	  else
	    error ("parameter has incomplete type");
	  TREE_TYPE (parms) = error_mark_node;
	}
#if 0
      /* If the arg types are incomplete in a declaration,
	 they must include undefined tags.
	 These tags can never be defined in the scope of the declaration,
	 so the types can never be completed,
	 and no call can be compiled successfully.  */
      /* This is not the right behavior for C++, but not having
	 it is also probably wrong.  */
      else
	{
	  /* Now warn if is a pointer to an incomplete type.  */
	  while (TREE_CODE (type) == POINTER_TYPE
		 || TREE_CODE (type) == REFERENCE_TYPE)
	    type = TREE_TYPE (type);
	  type = TYPE_MAIN_VARIANT (type);
	  if (TYPE_SIZE (type) == 0)
	    {
	      if (DECL_NAME (parm) != 0)
		warning ("parameter `%s' points to incomplete type",
			 IDENTIFIER_POINTER (DECL_NAME (parm)));
	      else
		warning ("parameter points to incomplete type");
	    }
	}
#endif
      parms = TREE_CHAIN (parms);
    }
}

/* Decode the list of parameter types for a function type.
   Given the list of things declared inside the parens,
   return a list of types.

   The list we receive can have three kinds of elements:
   an IDENTIFIER_NODE for names given without types,
   a TREE_LIST node for arguments given as typespecs or names with typespecs,
   or void_type_node, to mark the end of an argument list
   when additional arguments are not permitted (... was not used).

   FUNCDEF_FLAG is nonzero for a function definition, 0 for
   a mere declaration.  A nonempty identifier-list gets an error message
   when FUNCDEF_FLAG is zero.
   If FUNCDEF_FLAG is 1, then parameter types must be complete.
   If FUNCDEF_FLAG is -1, then parameter types may be incomplete.

   If all elements of the input list contain types,
   we return a list of the types.
   If all elements contain no type (except perhaps a void_type_node
   at the end), we return a null list.
   If some have types and some do not, it is an error, and we
   return a null list.

   Also set last_function_parms to either
   a list of names (IDENTIFIER_NODEs) or a chain of PARM_DECLs.
   A list of names is converted to a chain of PARM_DECLs
   by store_parm_decls so that ultimately it is always a chain of decls.

   Note that in C++, paramters can take default values.  These default
   values are in the TREE_PURPOSE field of the TREE_LIST.  It is
   an error to specify default values which are followed by parameters
   that have no defualt values, or an ELLIPSES.  For simplicities sake,
   only parameters which are specified with their types can take on
   default values.  */

static tree
grokparms (first_parm, funcdef_flag)
     tree first_parm;
     int funcdef_flag;
{
  tree result = NULL_TREE;
  tree names = NULL_TREE;
  tree decls = NULL_TREE;

  if (first_parm != 0
      && TREE_CODE (TREE_VALUE (first_parm)) == IDENTIFIER_NODE)
    {
      if (! funcdef_flag)
	warning ("parameter names (without types) in function declaration");
      last_function_parms = first_parm;
      return 0;
    }
  else
    {
      /* Types were specified.  This is a list of declarators
	 each represented as a TREE_LIST node.  */
      register tree parm;
      int any_init = 0, any_error = 0, saw_void = 0;

      for (parm = first_parm;
	   parm != NULL_TREE;
	   parm = TREE_CHAIN (parm))
	{
	  tree type;
	  register tree decl = TREE_VALUE (parm);
	  tree init = TREE_PURPOSE (parm);

	  /* @@ weak defense against parse errors.  */
	  if (decl != void_type_node && TREE_CODE (decl) != TREE_LIST)
	    {
	      /* Give various messages as the need arises.  */
	      if (TREE_CODE (decl) == STRING_CST)
		error ("invalid string constant `%s'",
		       TREE_STRING_POINTER (decl));
	      else if (TREE_CODE (decl) == INTEGER_CST)
		error ("invalid integer constant in parameter list, did you forget to give parameter name?");
	      continue;
	    }

	  if (decl != void_type_node)
	    {
	      decl = grokdeclarator (TREE_VALUE (decl),
				     TREE_PURPOSE (decl),
				     PARM, 0);
	      if (! decl) continue;
	      type = TREE_TYPE (decl);
	      if (type == void_type_node)
		decl = void_type_node;
	      else if (TREE_CODE (type) == METHOD_TYPE)
		{
		  if (DECL_NAME (decl))
		    error_with_decl (decl, "parameter `%s' invalidly declared %s type",
				     "method");
		  else
		    error ("parameter invalidly declared %s type", "method");
		  type = build_pointer_type (type);
		  TREE_TYPE (decl) = type;
		}
	      else if (TREE_CODE (type) == OFFSET_TYPE)
		{
		  if (DECL_NAME (decl))
		    error_with_decl (decl, "parameter `%s' invalidly declared %s type", "offset");
		  else
		    error ("parameter invalidly declared %s type", "offset");
		  type = build_pointer_type (type);
		  TREE_TYPE (decl) = type;
		}
	    }

	  if (decl == void_type_node)
	    {
	      result = chainon (result, void_list_node);
	      saw_void = 1;
	      if (TREE_CHAIN (parm)
		  && (TREE_VALUE (TREE_CHAIN (parm)) != void_type_node
		      || TREE_CHAIN (TREE_CHAIN (parm))))
		error ("`void' in parameter list must be entire list");
	      break;
	    }

	  /* Since there is a prototype, args are passed in their own types.  */
	  DECL_ARG_TYPE (decl) = TREE_TYPE (decl);
#ifdef PROMOTE_PROTOTYPES
	  if (TREE_CODE (type) == INTEGER_TYPE
	      && TYPE_PRECISION (type) < TYPE_PRECISION (integer_type_node))
	    DECL_ARG_TYPE (decl) = integer_type_node;
#endif
	  if (!any_error)
	    {
	      if (init)
		any_init++;
	      else if (any_init)
		{
		  error ("all trailing parameters must have default arguments");
		  any_error = 1;
		}
	    }
	  else
	    init = NULL_TREE;

	  decls = chainon (decls, decl);
	  result = chainon (result,
			    saveable_tree_cons (init, type, NULL_TREE));
	}
      if (any_init && ! saw_void)
	{
	  error ("default arguments cannot precede `...'");
	  any_error = 1;
	}
    }

  last_function_parms = decls;

  /* In a fcn definition, arg types must be complete.  */
  if (funcdef_flag > 0)
    require_complete_types_for_parms (last_function_parms);

  return result;
}

/* These memoizing functions keep track of special properties which
   a class may have.  `grok_ctor_properties' notices whether a class
   has a constructor of the for X(X&), and also complains
   if the class has a constructor of the form X(X).
   `grok_op_properties' takes notice of the various forms of
   operator= which are defined, as well as what sorts of type conversion
   may apply.  Both functions take a FUNCTION_DECL as an argument.  */
static void
grok_ctor_properties (decl)
     tree decl;
{
  tree parmtypes = TREE_CHAIN (TYPE_ARG_TYPES (TREE_TYPE (decl)));
  tree parmtype = parmtypes ? TREE_VALUE (parmtypes) : void_type_node;

  if (TREE_CODE (parmtype) == REFERENCE_TYPE
      && TYPE_MAIN_VARIANT (TREE_TYPE (parmtype)) == current_class_type)
    if (TREE_CHAIN (parmtypes) == NULL_TREE
	|| TREE_VALUE (TREE_CHAIN (parmtypes)) == void_type_node
	|| TREE_PURPOSE (TREE_CHAIN (parmtypes)))
      TYPE_GETS_INIT_REF (current_class_type) = 1;
    else
      TYPE_GETS_INIT_AGGR (current_class_type) = 1;
  else if (TYPE_MAIN_VARIANT (parmtype) == current_class_type)
    {
      if (TREE_CHAIN (parmtypes) != NULL_TREE
	  && TREE_VALUE (TREE_CHAIN (parmtypes)) == void_type_node)
	error ("invalid constructor; you probably meant `%s (%s&)'",
	       IDENTIFIER_POINTER (current_class_name),
	       IDENTIFIER_POINTER (current_class_name));
      IDENTIFIER_ERROR_LOCUS (DECL_NAME (decl)) = current_class_type;
      TYPE_GETS_INIT_AGGR (current_class_type) = 1;
    }
}

static void
grok_op_properties (decl)
     tree decl;
{
  if (! strncmp (IDENTIFIER_POINTER (DECL_NAME (decl)), "op$modify_expr", 11))
    {
      tree parmtypes = TREE_CHAIN (TYPE_ARG_TYPES (TREE_TYPE (decl)));
      tree parmtype = parmtypes ? TREE_VALUE (parmtypes) : void_type_node;

      if (TREE_CODE (parmtype) == REFERENCE_TYPE
	  && TREE_TYPE (parmtype) == current_class_type)
	{
	  TYPE_HAS_ASSIGN_REF (current_class_type) = 1;
	  TYPE_GETS_ASSIGN_REF (current_class_type) = 1;
	}
    }
#if 0
  else if (! strncmp (IDENTIFIER_POINTER (DECL_NAME (decl)), "op$call_expr", 8))
    TYPE_OVERLOADS_CALL_EXPR (current_class_type) = 1;
  else if (! strncmp (IDENTIFIER_POINTER (DECL_NAME (decl)), "op$array_ref", 9))
    TYPE_OVERLOADS_ARRAY_REF (current_class_type) = 1;
#endif
}

/* When a function is declared with an initialializer,
   do the right thing.  Currently, there are two possibilities:

   class B
   {
    public:
     // initialization possibility #1.
     virtual void f () = 0;
     int g ();
   };
   
   class D1 : B
   {
    public:
     int d1;
     // error, no f ();
   };
   
   class D2 : B
   {
    public:
     int d2;
     void f ();
   };
   
   class D3 : B
   {
    public:
     int d3;
     // initialization possibility #2
     void f () = B::f;
   };

   Returns the ASMSPEC that should be used for this function.

*/

static char *
grok_function_init (decl, init)
     tree decl;
     tree init;
{
  /* An initializer for a function tells how this function should
     be inherited.  */
  tree type = TREE_TYPE (decl);
  char *asmspec = 0;

  if (TREE_CODE (type) == FUNCTION_TYPE)
    error_with_decl (decl, "initializer specified for non-member function `%s'");
  else if (! DECL_VIRTUAL_P (decl))
    error_with_decl (decl, "initializer specified for non-virtual method `%s'");
  else if (integer_zerop (init))
    {
      CLASSTYPE_UNINHERITED_VIRTUALS (current_class_type)
	= tree_cons (NULL_TREE, decl, CLASSTYPE_UNINHERITED_VIRTUALS (current_class_type));
      /* Mark this function as being "defined".  */
      DECL_INITIAL (decl) = error_mark_node;
      /* Give this node new rtl.  */
      DECL_RTL (decl) = 0;
      /* @@ Assume names have underscores.  */
      asmspec = "_abort";
    }
  else if (TREE_CODE (init) == MEMBER_REF
	   && TREE_OPERAND (init, 0) == NULL_TREE
	   && TREE_CODE (TREE_TYPE (init)) == METHOD_TYPE)
    {
      tree basetype = TYPE_METHOD_BASETYPE (TREE_TYPE (init));
      tree basefn = TREE_OPERAND (init, 1);
      if (TREE_CODE (basefn) != FUNCTION_DECL)
	error_with_decl (decl, "non-method initializer invalid for method `%s'");
      else if (DECL_OFFSET (TYPE_NAME (basefn)) != 0)
	sorry ("base member function from other than first base class");
      else
	{
	  basetype = get_base_type (basetype, TYPE_METHOD_BASETYPE (type), 1);
	  if (basetype == error_mark_node)
	    ;
	  else if (basetype == 0)
	    error_not_derived_type (TYPE_METHOD_BASETYPE (TREE_TYPE (init)),
				    TYPE_METHOD_BASETYPE (type));
	  else
	    {
	      /* Mark this function as being defined,
		 and give it new rtl.  */
	      DECL_INITIAL (decl) = error_mark_node;
	      DECL_RTL (decl) = DECL_RTL (basefn);
	    }
	}
    }
  else
    error_with_decl (decl, "invalid initializer for virtual method `%s'");
  return asmspec;
}

/* Process the specs, declarator (NULL if omitted) and width (NULL if omitted)
   of a structure component, returning a FIELD_DECL node.
   WIDTH is non-NULL for bit fields only, and is an INTEGER_CST node.

   This is done during the parsing of the struct declaration.
   The FIELD_DECL nodes are chained together and the lot of them
   are ultimately passed to `build_struct' to make the RECORD_TYPE node.

   C++:

   If class A defines that certain functions in class B are friends, then
   the way I have set things up, it is B who is interested in permission
   granted by A.  However, it is in A's context that these declarations
   are parsed.  By returning a void_type_node, class A does not attempt
   to incorporate the declarations of the friends within its structure.

   DO NOT MAKE ANY CHANGES TO THIS CODE WITHOUT MAKING CORRESPONDING
   CHANGES TO CODE IN `start_method'.  */

tree
grokfield (declarator, declspecs, width, init, asmspec)
     tree declarator, declspecs, width, init;
     char *asmspec;
{
  register tree value = grokdeclarator (declarator, declspecs, FIELD, init != 0);
  if (! value) return NULL_TREE; /* friends went bad.  */

  /* Pass friendly classes back.  */
  if (TREE_CODE (value) == VOID_TYPE)
    return void_type_node;
  /* Stash away type declarations.  */
  if (TREE_CODE (value) == TYPE_DECL)
    {
      TREE_NONLOCAL (value) = 1;
      return value;
    }

  if (DECL_IN_AGGR_P (value))
    {
      error_with_decl (value, "`%s' is already defined in aggregate scope");
      return void_type_node;
    }

  if (init != 0)
    {
      if (pedantic
	  && ! TREE_STATIC (value)
	  && ! TREE_READONLY (value))
	{
	  error ("fields cannot have initializers");
	  init = error_mark_node;
	}
      else if (TREE_CODE (value) == FUNCTION_DECL)
	{
	  asmspec = grok_function_init (value, init);
	  init = NULL_TREE;
	}
      else
	{
	  /* We allow initializers to become parameters to base initializers.  */
	  if (TREE_READONLY (init) && TREE_CODE (init) == VAR_DECL)
	    init = decl_constant_value (init);
	  assert (TREE_PERMANENT (init));
	  if (! TREE_LITERAL (init))
	    {
	      /* We can allow references to things that are effectively
		 static, since references are initialized with the address.  */
	      if (TREE_CODE (TREE_TYPE (value)) != REFERENCE_TYPE
		  || (TREE_EXTERNAL (init) == 0
		      && TREE_STATIC (init) == 0))
		{
		  error ("field initializer is not constant");
		  init = error_mark_node;
		}
	    }
	}
    }

  if (TREE_CODE (value) == VAR_DECL)
    {
      if (asmspec == 0)
	{
	  char *buf = (char *)alloca (IDENTIFIER_LENGTH (current_class_name)
				      + IDENTIFIER_LENGTH (DECL_NAME (value))
				      + sizeof (STATIC_NAME_FORMAT));
	  tree name, static_decl;

	  sprintf (buf, STATIC_NAME_FORMAT,
		   IDENTIFIER_POINTER (current_class_name),
		   IDENTIFIER_POINTER (DECL_NAME (value)));
	  name = get_identifier (buf);
	  TREE_PUBLIC (value) = 1;
	  DECL_INITIAL (value) = error_mark_node;
	  /* We cannot call pushdecl here, because that would
	     fill in the value of our TREE_CHAIN.  Instead, we
	     modify finish_decl to do the right thing, namely, to
	     put this decl out straight away.  */
	  if (init)
	    {
	      if (TREE_CODE (init) == INTEGER_CST)
		;
	      else if (TREE_CODE (init) == VAR_DECL
		       && TREE_READONLY (init)
		       && TREE_LITERAL (init)
		       && tree_int_cst_equal (TYPE_SIZE (TREE_TYPE (init)),
					      TYPE_SIZE (ptr_type_node)))
		init = decl_constant_value (init);
	      else if (TREE_CODE (TREE_TYPE (value)) != REFERENCE_TYPE)
		{
		  sorry ("initialization of static class members from other than const int or pointer");
		  init = 0;
		}
	    }
	  asmspec = IDENTIFIER_POINTER (name);
	}
      pending_statics = perm_tree_cons (NULL_TREE, value, pending_statics);
      DECL_RTL_NAME (value) = asmspec;
      DECL_INITIAL (value) = init;
    }
  else if (TREE_CODE (value) == FIELD_DECL)
    {
      DECL_RTL_NAME (value) = asmspec;
      if (DECL_INITIAL (value) == error_mark_node)
	init = error_mark_node;
    }
  else if (TREE_CODE (value) == FUNCTION_DECL
	   && TREE_CHAIN (value) != NULL_TREE)
    {
      /* Need a fresh node here so that we don't get circularity
	 when we link these together.  */
      value = copy_node (value);
      /* When does this happen?  */
      assert (init == NULL_TREE);
    }

  finish_decl (value, init, asmspec);
  if (TREE_CODE (value) != FUNCTION_DECL)
    {
      if (TREE_CODE (value) == TYPE_DECL)
	/* Look like a friend.  It does not matter that we are not.  */
	return void_type_node;

      if (TREE_CODE (value) == VAR_DECL)
	{
	  if (width)
	    {
	      error ("static member `%s' cannot be a bitfield",
		     IDENTIFIER_POINTER (DECL_NAME (value)));
	      return NULL_TREE;
	    }
	}
      /* detect invalid field size.  */
      else if (width)
	{
	  if (TREE_READONLY (width) && TREE_CODE (width) == VAR_DECL)
	    width = decl_constant_value (width);

	  if (TREE_CODE (width) != INTEGER_CST)
	    {
	      error_with_decl (value, "structure field `%s' width not an integer constant");
	      DECL_INITIAL (value) = NULL;
	    }
	  else
	    {
	      DECL_INITIAL (value) = width;
	      TREE_PACKED (value) = 1;
	    }
	}
      else
	{
	  DECL_INITIAL (value) = init;
	}
    }
  DECL_IN_AGGR_P (value) = 1;
  return value;
}

/* Like GROKFIELD, except that the declarator has been
   buried in DECLSPECS.  Find the declarator, and
   return something that looks like it came from
   GROKFIELD.  */
tree
groktypefield (declspecs, parmlist)
     tree declspecs;
     tree parmlist;
{
  tree spec = declspecs;
  tree prev = NULL_TREE;

  tree type_id = NULL_TREE;
  tree quals = NULL_TREE;
  tree lengths = NULL_TREE;
  tree decl = NULL_TREE;

  while (spec)
    {
      register int i;
      register tree id = TREE_VALUE (spec);

      if (TREE_CODE (spec) != TREE_LIST)
	{
	  /* Certain parse errors slip through.  For example,
	     `int class ();' is not caught by the parser. Try
	     weakly to recover here.  */
	  return NULL_TREE;
	}

      if (TREE_CODE (id) == TYPE_DECL
	  || (TREE_CODE (id) == IDENTIFIER_NODE
	      && TREE_TYPE (id)))
	{
	  /* We have a constructor/destructor or
	     conversion operator.  Use it.  */
	  if (prev)
	    TREE_CHAIN (prev) = TREE_CHAIN (spec);
	  else
	    {
	      declspecs = TREE_CHAIN (spec);
	    }
	  type_id = id;
	  goto found;
	}
      prev = spec;
      spec = TREE_CHAIN (spec);
    }

  /* Nope, we have a conversion operator to a scalar type.  */
  spec = declspecs;
  while (spec)
    {
      tree id = TREE_VALUE (spec);

      if (TREE_CODE (id) == IDENTIFIER_NODE)
	{
	  if (id == ridpointers[(int)RID_INT]
	      || id == ridpointers[(int)RID_DOUBLE]
	      || id == ridpointers[(int)RID_FLOAT])
	    {
	      if (type_id)
		error ("extra `%s' ignored",
		       IDENTIFIER_POINTER (id));
	      else
		type_id = id;
	    }
	  else if (id == ridpointers[(int)RID_LONG]
		   || id == ridpointers[(int)RID_SHORT]
		   || id == ridpointers[(int)RID_CHAR])
	    {
	      lengths = tree_cons (NULL_TREE, id, lengths);
	    }
	  else if (id == ridpointers[(int)RID_VOID])
	    {
	      if (type_id)
		error ("spurious `void' type ignored");
	      else
		error ("conversion to `void' type invalid");
	    }
	  else if (id == ridpointers[(int)RID_AUTO]
		   || id == ridpointers[(int)RID_REGISTER]
		   || id == ridpointers[(int)RID_TYPEDEF]
		   || id == ridpointers[(int)RID_CONST]
		   || id == ridpointers[(int)RID_VOLATILE])
	    {
	      error ("type specifier `%s' used invalidly",
		     IDENTIFIER_POINTER (id));
	    }
	  else if (id == ridpointers[(int)RID_FRIEND]
		   || id == ridpointers[(int)RID_VIRTUAL]
		   || id == ridpointers[(int)RID_INLINE]
		   || id == ridpointers[(int)RID_UNSIGNED]
		   || id == ridpointers[(int)RID_SIGNED]
		   || id == ridpointers[(int)RID_STATIC]
		   || id == ridpointers[(int)RID_EXTERN])
	    {
	      quals = tree_cons (NULL_TREE, id, quals);
	    }
	  else
	    assert (0);
	}
      else if (TREE_CODE (id) == RECORD_TYPE)
	{
	  error ("identifier for aggregate type conversion omitted");
	}
      else
	assert (0);
      spec = TREE_CHAIN (spec);
    }

  if (type_id)
    {
      declspecs = chainon (lengths, quals);
    }
  else if (lengths)
    {
      if (TREE_CHAIN (lengths))
	error ("multiple length specifiers");
      type_id = ridpointers[(int)RID_INT];
      declspecs = chainon (lengths, quals);
    }
  else if (quals)
    {
      error ("no type given, defaulting to `operator int ...'");
      type_id = ridpointers[(int)RID_INT];
      declspecs = quals;
    }
  else return NULL_TREE;
 found:
  decl = grokdeclarator (build_nt (CALL_EXPR, type_id, parmlist, NULL_TREE),
			 declspecs, FIELD, 0);
  if (decl == NULL_TREE)
    return NULL_TREE;

  if (TREE_CODE (decl) == FUNCTION_DECL && TREE_CHAIN (decl) != NULL_TREE)
    {
      /* Need a fresh node here so that we don't get circularity
	 when we link these together.  */
      decl = copy_node (decl);
    }

  if (decl == void_type_node
      || (TREE_CODE (decl) == FUNCTION_DECL
	  && TREE_CODE (TREE_TYPE (decl)) != METHOD_TYPE))
    /* bunch of friends.  */
    return decl;

  if (DECL_IN_AGGR_P (decl))
    {
      error_with_decl (decl, "`%s' already defined in aggregate scope");
      return void_type_node;
    }

  finish_decl (decl, NULL_TREE, NULL_TREE);

  /* If this declaration is common to another declaration
     complain about such redundancy, and return NULL_TREE
     so that we don't build a circular list.  */
  if (TREE_CHAIN (decl))
    {
      error_with_decl (decl, "function `%s' declared twice in aggregate");
      return NULL_TREE;
    }
  DECL_IN_AGGR_P (decl) = 1;
  return decl;
}

/* Get the struct, enum or union (CODE says which) with tag NAME.
   Define the tag as a forward-reference if it is not defined.

   C++: If a class derivation is given, process it here, and report
   an error if multiple derivation declarations are not identical.

   If we are compiling for SOS, then
     if CODE_TYPE_NODE is a TREE_LIST, then we have a dynamic class
     declaration.  The name associated with the class is the tree
     purpose, and the real CODE is in the tree value slot.  */
tree
xref_tag (code_type_node, name, basetype_info)
     tree code_type_node;
     tree name, basetype_info;
{
  enum tag_types tag_code;
  enum tree_code code;
  int temp = 0;
  int i, len = list_length (basetype_info) + 1;
  register tree ref;
  struct binding_level *b
    = (class_binding_level ? class_binding_level : current_binding_level);
#ifdef SOS
  tree dynamic_name = error_mark_node;
  if (TREE_CODE (code_type_node) == TREE_LIST)
    {
      dynamic_name = TREE_PURPOSE (code_type_node);
      code_type_node = TREE_VALUE (code_type_node);
    }
#endif

  tag_code = (enum tag_types) TREE_INT_CST_LOW (code_type_node);
  switch (tag_code)
    {
    case record_type:
    case class_type:
      code = RECORD_TYPE;
      break;
    case union_type:
      code = UNION_TYPE;
      break;
    case enum_type:
      code = ENUMERAL_TYPE;
      break;
    default:
      abort ();
    }

  /* If a cross reference is requested, look up the type
     already defined for this tag and return it.  */
  ref = lookup_tag (code, name, b, 0);

  if (! ref)
    {
      /* If no such tag is yet defined, create a forward-reference node
	 and record it as the "definition".
	 When a real declaration of this type is found,
	 the forward-reference will be altered into a real type.  */

      /* In C++, since these migrate into the global scope, we must
	 build them on the permanent obstack.  */
      if (temp == 0)
	temp = allocation_temporary_p ();
      if (temp)
	end_temporary_allocation ();

      if (code == ENUMERAL_TYPE)
	{
	  ref = make_node (ENUMERAL_TYPE);

	  /* Give the type a default layout like unsigned int
	     to avoid crashing if it does not get defined.  */
	  TYPE_MODE (ref) = SImode;
	  TYPE_ALIGN (ref) = TYPE_ALIGN (unsigned_type_node);
	  TREE_UNSIGNED (ref) = 1;
	  TYPE_PRECISION (ref) = TYPE_PRECISION (unsigned_type_node);
	  TYPE_MIN_VALUE (ref) = TYPE_MIN_VALUE (unsigned_type_node);
	  TYPE_MAX_VALUE (ref) = TYPE_MAX_VALUE (unsigned_type_node);

	  /* Enable us to recognize when a type is created in class context.
	     To do nested classes correctly, this should probably be cleared
	     out when we leave this classes scope.  Currently this in only
	     done in `start_enum'.  */

	  pushtag (name, ref);
	}
      else
	{
	  struct binding_level *old_b = class_binding_level;

	  ref = make_lang_type (code);

	  CLASSTYPE_BASECLASSES (ref) = (tree *) malloc (len * sizeof (tree));
	  CLASSTYPE_N_BASECLASSES (ref) = len - 1;
	  CLASSTYPE_THIS_BASECLASS (ref, 0) = NULL_TREE;
	  CLASSTYPE_OFFSET (ref) = integer_zero_node;
	  CLASSTYPE_VTABLE (ref) = NULL_TREE;
	  CLASSTYPE_VIAS (ref) = (unsigned char *) malloc (len);

	  /* Class types don't nest the way enums do.  */
	  class_binding_level = 0;
	  pushtag (name, ref);
	  class_binding_level = old_b;
	}
    }
  else if (basetype_info)
    {
      tree tt1 = basetype_info;
      tree *tt2 = CLASSTYPE_BASECLASSES (ref);

      if (CLASSTYPE_N_BASECLASSES (ref))
	for (i = 1; tt1; i++, tt1 = TREE_CHAIN (tt1))
	  if (TREE_VALUE (tt1) != tt2[i])
	      {
		error ("redeclaration of derivation chain of type `%s'",
		       IDENTIFIER_POINTER (name));
		break;
	      }

      if (tt1 != NULL_TREE)
	{
	  free (CLASSTYPE_BASECLASSES (ref));
	  free (CLASSTYPE_VIAS (ref));
	  CLASSTYPE_BASECLASSES (ref) = (tree *) malloc (len * sizeof (tree));
	  CLASSTYPE_N_BASECLASSES (ref) = len - 1;
	  CLASSTYPE_THIS_BASECLASS (ref, 0) = NULL_TREE;
	  CLASSTYPE_OFFSET (ref) = integer_zero_node;
	  CLASSTYPE_VTABLE (ref) = NULL_TREE;
	  CLASSTYPE_VIAS (ref) = (unsigned char *) malloc (len);
	}
      else
	{
	  /* The user told us something we already knew.  */
	  goto just_return;
	}
    }
#ifdef SOS
  else if (TREE_CODE (ref) != ENUMERAL_TYPE
	   && (dynamic_name != error_mark_node) != TYPE_DYNAMIC (ref))
    error ("type `%s' declared both dynamic and non-dynamic",
	   TYPE_NAME_STRING (ref));
#endif

  if (basetype_info)
    {
      for (i = 1; basetype_info; basetype_info = TREE_CHAIN (basetype_info))
	{
	  int offset = 0;

	  /* The base of a derived struct is public.  */
	  int via_public = (tag_code != class_type
			    || TREE_PURPOSE (basetype_info) == (tree)visibility_public
			    || TREE_PURPOSE (basetype_info) == (tree)visibility_public_virtual);
	  int via_virtual = (TREE_PURPOSE (basetype_info) == (tree)visibility_private_virtual
			     || TREE_PURPOSE (basetype_info) == (tree)visibility_public_virtual
			     || TREE_PURPOSE (basetype_info) == (tree)visibility_default_virtual);
	  tree basetype = TREE_TYPE (TREE_VALUE (basetype_info));

	  if (basetype && TREE_CODE (basetype) == TYPE_DECL)
	    {
	      basetype = TREE_TYPE (basetype);
	    }
	  if (!basetype || TREE_CODE (basetype) != RECORD_TYPE)
	    {
	      error ("base type `%s' fails to be a struct or class type",
		     IDENTIFIER_POINTER (TREE_VALUE (basetype_info)));
	      continue;
	    }
#if 0
	  else if (TYPE_SIZE (basetype) == 0)
	    {
	      error_with_aggr_type (basetype, "base class `%s' has incomplete type");
	      continue;
	    }
#endif
	  else
	    {
#ifdef SOS
	      if (dynamic_name == error_mark_node && TYPE_DYNAMIC (basetype))
		error_with_aggr_type (ref, "non-dynamic type `%s' cannot derive from dynamic type `%s'", TYPE_NAME_STRING (basetype));
#endif
	      if (CLASSTYPE_MARKED (basetype))
		{
		  error_with_aggr_type (basetype, "duplicate base type `%s' invalid");
		  continue;
		}
	      CLASSTYPE_THIS_BASECLASS (ref, i) = basetype;
	      CLASSTYPE_MARKED (basetype) = 1;
	      SET_CLASSTYPE_VIAS (ref, i, via_public, via_virtual);
	      TREE_GETS_ASSIGNMENT (ref) |= TREE_GETS_ASSIGNMENT (basetype);
	      TYPE_OVERLOADS_METHOD_CALL_EXPR (ref) |= TYPE_OVERLOADS_METHOD_CALL_EXPR (basetype);
	      TYPE_HAS_WRAPPER_PRED (ref) |= TYPE_HAS_WRAPPER_PRED (basetype);
	      TREE_GETS_NEW (ref) |= TREE_GETS_NEW (basetype);
	      TREE_GETS_DELETE (ref) |= TREE_GETS_DELETE (basetype);
	      i += 1;
	    }
	}
      /* Set the true number of baseclasses this type really has.  */
      CLASSTYPE_N_BASECLASSES (ref) = i - 1;
      while (--i > 0)
	CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (ref, i)) = 0;
    }

 just_return:

#ifdef SOS
  if (dynamic_name != error_mark_node)
    {
      if (temp == 0)
	temp = allocation_temporary_p ();
      if (temp)
	end_temporary_allocation ();

      if (dynamic_name)
	CLASSTYPE_DYNAMIC_FILENAME (ref) = combine_strings (dynamic_name);
      else
	CLASSTYPE_DYNAMIC_FILENAME (ref) = NULL_TREE;
      TYPE_DYNAMIC (ref) = 1;
      CLASSTYPE_TYPENAME_AS_STRING (ref) = combine_strings (build_string (IDENTIFIER_LENGTH (name), IDENTIFIER_POINTER (name)));

    }
#endif

  /* Until the type is defined, tentatively accept whatever
     structure tag the user hands us.  */
  if (TYPE_SIZE (ref) == NULL_TREE
      && IS_AGGR_TYPE_CODE (code))
    CLASSTYPE_DECLARED_CLASS (ref) = (tag_code == class_type);

  if (temp)
    resume_temporary_allocation ();

  return ref;
}

/* Finish off the processing of a UNION_TYPE structure.
   If there are static members, then all members are
   static, and must be laid out together.  If the
   union is an anonymous union, we arrage for that
   as well.  PUBLICP is nonzero if this union is
   not declared static.  */
void
finish_anon_union (anon_union_decl)
     tree anon_union_decl;
{
  tree type = TREE_TYPE (anon_union_decl);
  tree field, decl;
  tree elems = NULL_TREE;
  int public_p = TREE_PUBLIC (anon_union_decl);
  int static_p = TREE_STATIC (anon_union_decl);
  int external_p = TREE_EXTERNAL (anon_union_decl);

  if ((field = TYPE_FIELDS (type)) == NULL_TREE)
    return;

  if (static_p || external_p)
    {
      error ("optimizer cannot handle static or external anonymous unions");
      return;
    }

  while (field)
    {
      decl = build_decl (VAR_DECL, DECL_NAME (field), TREE_TYPE (field));
      /* tell `pushdecl' that this is not tentative.  */
      DECL_INITIAL (decl) = error_mark_node;
      TREE_PUBLIC (decl) = public_p;
      TREE_STATIC (decl) = static_p;
      TREE_EXTERNAL (decl) = external_p;
      decl = pushdecl (decl);
      DECL_INITIAL (decl) = NULL_TREE;
      elems = tree_cons (DECL_RTL_NAME (field), decl, elems);
      TREE_TYPE (elems) = type;
      field = TREE_CHAIN (field);
    }
  {
    tree anon_decl = build_lang_decl (VAR_DECL, NULL_TREE, type);
    expand_anon_union_decl (decl, NULL_TREE, elems);
  }
}

/* Begin compiling the definition of an enumeration type.
   NAME is its name (or null if anonymous).
   Returns the type object, as yet incomplete.
   Also records info about it so that build_enumerator
   may be used to declare the individual values as they are read.  */

tree
start_enum (name)
     tree name;
{
  register tree enumtype = 0;
  struct binding_level *b
    = (class_binding_level ? class_binding_level : current_binding_level);

  /* If this is the real definition for a previous forward reference,
     fill in the contents in the same object that used to be the
     forward reference.  */

  if (name != 0)
    enumtype = lookup_tag (ENUMERAL_TYPE, name, b, 1);

  if (enumtype == 0 || TREE_CODE (enumtype) != ENUMERAL_TYPE)
    {
      enumtype = make_node (ENUMERAL_TYPE);
      pushtag (name, enumtype);
    }

  if (TYPE_VALUES (enumtype) != 0)
    {
      /* This enum is a named one that has been declared already.  */
      error ("redeclaration of `enum %s'", IDENTIFIER_POINTER (name));

      /* Completely replace its old definition.
	 The old enumerators remain defined, however.  */
      TYPE_VALUES (enumtype) = 0;
    }

  /* Initially, set up this enum as like `int'
     so that we can create the enumerators' declarations and values.
     Later on, the precision of the type may be changed and
     it may be laid out again.  */

  TYPE_PRECISION (enumtype) = TYPE_PRECISION (integer_type_node);
  TYPE_SIZE (enumtype) = 0;
  fixup_unsigned_type (enumtype);

  /* We copy this value because enumerated type constants
     are really of the type of the enumerator, not integer_type_node.  */
  enum_next_value = copy_node (integer_zero_node);

  return enumtype;
}

/* After processing and defining all the values of an enumeration type,
   install their decls in the enumeration type and finish it off.
   ENUMTYPE is the type object and VALUES a list of name-value pairs.
   Returns ENUMTYPE.  */

tree
finish_enum (enumtype, values)
     register tree enumtype, values;
{
  register tree pair;
  register long maxvalue = 0;
  register long minvalue = 0;
  register int i;

  TYPE_VALUES (enumtype) = values;

  /* Calculate the maximum value of any enumerator in this type.  */

  if (values)
    {
      /* Speed up the main loop by performing some precalculations */

      int value = TREE_INT_CST_LOW (TREE_VALUE (values));
      TREE_TYPE (TREE_VALUE (values)) = enumtype;
      minvalue = maxvalue = value;
      
      for (pair = TREE_CHAIN (values); pair; pair = TREE_CHAIN (pair))
	{
	  value = TREE_INT_CST_LOW (TREE_VALUE (pair));
	  if (value > maxvalue)
	    maxvalue = value;
	  else if (value < minvalue)
	    minvalue = value;
	  TREE_TYPE (TREE_VALUE (pair)) = enumtype;
	}
    }
#if 0
  /* Determine the precision this type needs, lay it out, and define it.  */

  for (i = maxvalue; i; i >>= 1)
    TYPE_PRECISION (enumtype)++;

  if (!TYPE_PRECISION (enumtype))
    TYPE_PRECISION (enumtype) = 1;

  /* Cancel the laying out previously done for the enum type,
     so that fixup_unsigned_type will do it over.  */
  TYPE_SIZE (enumtype) = 0;

  fixup_unsigned_type (enumtype);
#endif

  TREE_INT_CST_LOW (TYPE_MAX_VALUE (enumtype)) = maxvalue;

  /* An enum can have some negative values; then it is signed.  */
  if (minvalue < 0)
    {
      TREE_INT_CST_LOW (TYPE_MIN_VALUE (enumtype)) = minvalue;
      TREE_INT_CST_HIGH (TYPE_MIN_VALUE (enumtype)) = -1;
      TREE_UNSIGNED (enumtype) = 0;
    }
  return enumtype;
}

/* Build and install a CONST_DECL for one value of the
   current enumeration type (one that was begun with start_enum).
   Return a tree-list containing the name and its value.
   Assignment of sequential values by default is handled here.  */

tree
build_enumerator (name, value)
     tree name, value;
{
  tree decl, result;

  /* Validate and default VALUE.  */

  if (value != 0 && TREE_CODE (value) != INTEGER_CST)
    {
      error ("enumerator value for `%s' not integer constant",
	     IDENTIFIER_POINTER (name));
      value = 0;
    }

  /* The order of things is reversed here so that we
     can check for possible sharing of enum values,
     to keep that from happening.  */
  /* Default based on previous value.  */
  if (value == 0)
    value = enum_next_value;

  result = saveable_tree_cons (name, value, NULL_TREE);

  /* C++ associates enums with global, function, or class declarations.  */
  if (current_class_type == NULL_TREE || current_function_decl != NULL_TREE)
    {
      /* Create a declaration for the enum value name.  */

      decl = build_decl (CONST_DECL, name, integer_type_node);
      DECL_INITIAL (decl) = value;
      TREE_TYPE (value) = integer_type_node;

      pushdecl (decl);
    }

  /* Set basis for default for next value.  */
  enum_next_value = build_binary_op_nodefault (PLUS_EXPR, value,
					       integer_one_node, PLUS_EXPR);
  if (TREE_UID (enum_next_value) < TREE_UID (result))
    enum_next_value = copy_node (enum_next_value);

  return result;
}

/* Create the FUNCTION_DECL for a function definition.
   DECLSPECS and DECLARATOR are the parts of the declaration;
   they describe the function's name and the type it returns,
   but twisted together in a fashion that parallels the syntax of C.

   This function creates a binding context for the function body
   as well as setting up the FUNCTION_DECL in current_function_decl.

   Returns 1 on success.  If the DECLARATOR is not suitable for a function
   (it defines a datum instead), we return 0, which tells
   yyparse to report a parse error.

   For C++, we must first check whether that datum makes any sense.
   For example, "class A local_a(1,2);" means that variable local
   a is an aggregate of type A, which should have a constructor
   applied to it with the argument list [1, 2].

   @@ There is currently no way to retrieve the storage
   @@ allocated to FUNCTION (or all of its parms) if we return
   @@ something we had previously.  */

int
start_function (declspecs, declarator, pre_parsed_p)
     tree declarator, declspecs;
     int pre_parsed_p;
{
  tree decl1, olddecl;
  tree ctype = NULL_TREE;
  tree fntype;
  tree restype;

  /* Sanity check.  */
  assert (TREE_VALUE (void_list_node) == void_type_node);

  /* Assume, until we see it does. */
  current_function_returns_value = 0;
  current_function_returns_null = 0;
  warn_about_return_type = 0;
  current_function_assigns_this = 0;
  current_function_just_assigned_this = 0;
  current_function_parms_stored = 0;

  temp_name_counter = 0;

  if (pre_parsed_p)
    {
      tree t;

      /* We must waste one node's worth of space to satisfy `pushdecl'.  */
      decl1 = copy_node (declarator);
      last_function_parms = DECL_ARGUMENTS (decl1);
      last_function_parm_tags = 0;
      fntype = TREE_TYPE (decl1);
      if (TREE_CODE (fntype) == METHOD_TYPE)
	ctype = TYPE_METHOD_BASETYPE (fntype);

      /* In a fcn definition, arg types must be complete.  */
      require_complete_types_for_parms (last_function_parms);
    }
  else
    {
      decl1 = grokdeclarator (declarator, declspecs, FUNCDEF, 1);

      /* If the declarator is not suitable for a function definition,
	 cause a syntax error.  */
      if (decl1 == 0 || TREE_CODE (decl1) != FUNCTION_DECL) return 0;

      fntype = TREE_TYPE (decl1);
      if (TREE_CODE (fntype) == METHOD_TYPE)
	ctype = TYPE_METHOD_BASETYPE (fntype);
    }

  /* Warn if function was previously implicitly declared
     (but not if we warned then).  */
  if (! warn_implicit && IDENTIFIER_IMPLICIT_DECL (DECL_NAME (decl1)) != 0)
    warning_with_decl (IDENTIFIER_IMPLICIT_DECL (DECL_NAME (decl1)),
		       "`%s' implicitly declared before its definition");

  current_function_decl = decl1;

  announce_function (0, current_function_decl);

  if (TYPE_SIZE (TREE_TYPE (fntype)) == 0)
    {
      if (IS_AGGR_TYPE (TREE_TYPE (fntype)))
	error_with_aggr_type (TREE_TYPE (fntype),
			      "return-type `%s' is an incomplete type");
      else
	error ("return-type is an incomplete type");

      /* Make it return void instead.  */
      if (ctype)
	TREE_TYPE (decl1)
	  = build_method_type (ctype,
			       build_function_type (void_type_node,
						    TREE_CHAIN (TYPE_ARG_TYPES (TREE_TYPE (decl1)))));
      else
	TREE_TYPE (decl1)
	  = build_function_type (void_type_node,
				 TYPE_ARG_TYPES (TREE_TYPE (decl1)));
    }

  if (warn_about_return_type)
    warning ("return-type defaults to `int'");

  /* Make the init_value nonzero so pushdecl knows this is not tentative.
     error_mark_node is replaced below (in poplevel) with the LET_STMT.  */
  DECL_INITIAL (current_function_decl) = error_mark_node;

  /* If this definition isn't a prototype and we had a prototype declaration
     before, copy the arg type info from that prototype.  */
  olddecl = lookup_name_current_level (DECL_NAME (current_function_decl));
  if (olddecl != 0
      && TREE_TYPE (TREE_TYPE (current_function_decl)) == TREE_TYPE (TREE_TYPE (olddecl))
      && TYPE_ARG_TYPES (TREE_TYPE (current_function_decl)) == 0)
    {
      fntype = TREE_TYPE (olddecl);
      TREE_TYPE (current_function_decl) = fntype;
    }

  /* This is a definition, not a reference.  */
  TREE_EXTERNAL (current_function_decl) = 0;
  /* This function exists in static storage.
     (This does not mean `static' in the C sense!)  */
  TREE_STATIC (current_function_decl) = 1;

  /* Record the decl so that the function name is defined.
     If we already have a decl for this name, and it is a FUNCTION_DECL,
     use the old decl.  */

  current_function_decl = pushdecl (current_function_decl);
  DECL_MAIN_VARIANT (current_function_decl) = current_function_decl;

  if (TREE_OVERLOADED (current_function_decl))
    push_overloaded_decl (current_function_decl);
  /* Is it really a static member function after all?  If so,
     take it all back.  */
  if (ctype != 0 && DECL_STATIC_FUNCTION_P (current_function_decl))
    {
      TREE_TYPE (current_function_decl)
	= build_function_type (TREE_TYPE (fntype),
			       TREE_CHAIN (TYPE_ARG_TYPES (fntype)));
      last_function_parms = TREE_CHAIN (last_function_parms);
      DECL_ARGUMENTS (current_function_decl) = last_function_parms;
      ctype = 0;
    }

  pushlevel (0);
  current_binding_level->parm_flag = 1;

  /* Save the parm names or decls from this function's declarator
     where store_parm_decls will find them.  */
  current_function_parms = last_function_parms;
  current_function_parm_tags = last_function_parm_tags;

  make_function_rtl (current_function_decl);

  if (ctype)
    {
      pushclass (ctype, 1);
      /* We know that this was set up by `grokclassfn'.
	 We do not wait until `store_parm_decls', since evil
	 parse errors may never get us to that point.  Here
	 we keep the consistency between `current_class_type'
	 and `current_class_decl'.  */
      current_class_decl = last_function_parms;
      assert (TREE_CODE (current_class_decl) == PARM_DECL);
      C_C_D = build (INDIRECT_REF, current_class_type, current_class_decl);
    }
  else
    {
      if (DECL_STATIC_FUNCTION_P (current_function_decl))
	pushclass (DECL_CONTEXT (current_function_decl), 2);
      else
	push_memoized_context (0, 1);
    }

  restype = TREE_TYPE (fntype);

  /* Promote the value to int before returning it.  */
  if (TREE_CODE (restype) == INTEGER_TYPE
      && TYPE_PRECISION (restype) < TYPE_PRECISION (integer_type_node))
    restype = integer_type_node;
  DECL_RESULT_TYPE (current_function_decl) = restype;
  DECL_RESULT (current_function_decl)
    = build_decl (RESULT_DECL, value_identifier, restype);

  /* Allocate further tree nodes temporarily during compilation
     of this function only.  */
  temporary_allocation ();

  /* If this fcn was already referenced via a block-scope `extern' decl
     (or an implicit decl), propagate certain information about the usage.  */
  if (TREE_ADDRESSABLE (DECL_NAME (current_function_decl)))
    TREE_ADDRESSABLE (current_function_decl) = 1;

  return 1;
}

/* Store the parameter declarations into the current function declaration.
   This is called after parsing the parameter declarations, before
   digesting the body of the function.

   Also install to binding contour return value identifier, if any.  */

void
store_parm_decls ()
{
  register tree fndecl = current_function_decl;
  register tree parm;
  int parms_have_cleanups = 0;

  /* This is either a chain of PARM_DECLs (if a prototype was used)
     or a list of IDENTIFIER_NODEs (for an old-fashioned C definition).  */
  tree specparms = current_function_parms;

  /* This is a list of types declared among parms in a prototype.  */
  tree parmtags = current_function_parm_tags;

  /* This is a chain of PARM_DECLs from old-style parm declarations.  */
  register tree parmdecls = getdecls ();

  /* This is a chain of any other decls that came in among the parm
     declarations.  If a parm is declared with  enum {foo, bar} x;
     then CONST_DECLs for foo and bar are put here.  */
  tree nonparms = 0;

  if (current_binding_level == global_binding_level)
    {
      fatal ("parse errors have confused me too much");
    }

  /* Initialize RTL machinery.  */
  init_function_start (fndecl);

  /* Create a binding level for the parms.  */
  expand_start_bindings (0);

  if (specparms != 0 && TREE_CODE (specparms) != TREE_LIST)
    {
      /* This case is when the function was defined with an ANSI prototype.
	 The parms already have decls, so we need not do anything here
	 except record them as in effect
	 and complain if any redundant old-style parm decls were written.  */

      register tree next;
      tree others = 0;

      if (parmdecls != 0)
	error_with_decl (fndecl,
			 "parm types given both in parmlist and separately");

      for (parm = nreverse (specparms); parm; parm = next)
	{
	  tree cleanup;
	  next = TREE_CHAIN (parm);
	  if (DECL_NAME (parm) == 0)
	    {
#if 0
	      error_with_decl (parm, "parameter name omitted");
#else
	      /* for C++, this is not an error.  */
	      pushdecl (parm);
#endif
	    }
	  else if (TREE_TYPE (parm) == void_type_node)
	    error_with_decl (parm, "parameter `%s' declared void");
	  else
	    pushdecl (parm);
	  cleanup = maybe_build_cleanup (parm);
	  if (cleanup)
	    {
	      expand_decl (parm, cleanup);
	      expand_decl_init (parm);
	      parms_have_cleanups = 1;
	    }
	}

      /* Get the decls in their original chain order
	 and record in the function.  */
      DECL_ARGUMENTS (fndecl) = getdecls ();

      storetags (chainon (parmtags, gettags ()));
    }
  else
    {
      /* SPECPARMS is an identifier list--a chain of TREE_LIST nodes
	 each with a parm name as the TREE_VALUE.

	 PARMDECLS is a list of declarations for parameters.
	 Warning! It can also contain CONST_DECLs which are not parameters
	 but are names of enumerators of any enum types
	 declared among the parameters.

	 First match each formal parameter name with its declaration.
	 Associate decls with the names and store the decls
	 into the TREE_PURPOSE slots.  */

      for (parm = specparms; parm; parm = TREE_CHAIN (parm))
	{
	  register tree tail, found = NULL;

	  if (TREE_VALUE (parm) == 0)
	    {
	      error_with_decl (fndecl, "parameter name missing from parameter list");
	      TREE_PURPOSE (parm) = 0;
	      continue;
	    }

	  /* See if any of the parmdecls specifies this parm by name.
	     Ignore any enumerator decls.  */
	  for (tail = parmdecls; tail; tail = TREE_CHAIN (tail))
	    if (DECL_NAME (tail) == TREE_VALUE (parm)
		&& TREE_CODE (tail) == PARM_DECL)
	      {
		found = tail;
		break;
	      }

	  /* If declaration already marked, we have a duplicate name.
	     Complain, and don't use this decl twice.   */
	  if (found && DECL_CONTEXT (found) != 0)
	    {
	      error_with_decl (found, "multiple parameters named `%s'");
	      found = 0;
	    }

	  /* If the declaration says "void", complain and ignore it.  */
	  if (found && TREE_TYPE (found) == void_type_node)
	    {
	      error_with_decl (found, "parameter `%s' declared void");
	      TREE_TYPE (found) = integer_type_node;
	      DECL_ARG_TYPE (found) = integer_type_node;
	      layout_decl (found, 0);
	    }

	  /* If no declaration found, default to int.  */
	  if (!found)
	    {
	      found = build_decl (PARM_DECL, TREE_VALUE (parm),
				  integer_type_node);
	      DECL_ARG_TYPE (found) = TREE_TYPE (found);
	      DECL_SOURCE_LINE (found) = DECL_SOURCE_LINE (fndecl);
	      DECL_SOURCE_FILE (found) = DECL_SOURCE_FILE (fndecl);
	      if (warn_implicit)
		warning_with_decl (found, "type of `%s' defaults to `int'");
	      pushdecl (found);
	    }

	  TREE_PURPOSE (parm) = found;

	  /* Mark this decl as "already found" -- see test, above.
	     It is safe to clobber DECL_CONTEXT temporarily
	     because the final values are not stored until
	     the `poplevel' in `finish_function'.  */
	  DECL_CONTEXT (found) = error_mark_node;
	}

      /* Complain about declarations not matched with any names.
	 Put any enumerator constants onto the list NONPARMS.  */

      for (parm = parmdecls; parm; )
	{
	  tree next = TREE_CHAIN (parm);
	  TREE_CHAIN (parm) = 0;

	  /* Complain about args with incomplete types.  */
	  if (TYPE_SIZE (TREE_TYPE (parm)) == 0)
	    {
	      error_with_decl (parm, "parameter `%s' has incomplete type");
	      TREE_TYPE (parm) = error_mark_node;
	    }

	  if (TREE_CODE (parm) != PARM_DECL)
	    nonparms = chainon (nonparms, parm);

	  else if (DECL_CONTEXT (parm) == 0)
	    {
	      error_with_decl (parm,
			       "declaration for parameter `%s' but no such parameter");
	      /* Pretend the parameter was not missing.
		 This gets us to a standard state and minimizes
		 further error messages.  */
	      specparms
		= chainon (specparms,
			   tree_cons (parm, NULL_TREE, NULL_TREE));
	    }

	  parm = next;
	}

      /* Chain the declarations together in the order of the list of names.  */
      /* Store that chain in the function decl, replacing the list of names.  */
      parm = specparms;
      DECL_ARGUMENTS (fndecl) = 0;
      {
	register tree last;
	for (last = 0; parm; parm = TREE_CHAIN (parm))
	  if (TREE_PURPOSE (parm))
	    {
	      if (last == 0)
		DECL_ARGUMENTS (fndecl) = TREE_PURPOSE (parm);
	      else
		TREE_CHAIN (last) = TREE_PURPOSE (parm);
	      last = TREE_PURPOSE (parm);
	      TREE_CHAIN (last) = 0;
	      DECL_CONTEXT (last) = 0;
	    }
      }

      /* If there was a previous prototype,
	 set the DECL_ARG_TYPE of each argument according to
	 the type previously specified, and report any mismatches.  */

      if (TYPE_ARG_TYPES (TREE_TYPE (fndecl)))
	{
	  register tree type;
	  for (parm = DECL_ARGUMENTS (fndecl),
	       type = TYPE_ARG_TYPES (TREE_TYPE (fndecl));
	       parm || (type && TREE_VALUE (type) != void_type_node);
	       parm = TREE_CHAIN (parm), type = TREE_CHAIN (type))
	    {
	      if (parm == 0 || type == 0
		  || TREE_VALUE (type) == void_type_node)
		{
		  error_with_decl (fndecl, "argument decls of `%s' don't match prototype");
		  break;
		}
	      /* Type for passing arg must be consistent
		 with that declared for the arg.  */
	      if (! comptypes (DECL_ARG_TYPE (parm), TREE_VALUE (type), 1))
		error_with_decl (fndecl, "argument `%s' doesn't match function prototype");
	      if (DECL_ARG_TYPE (parm) == integer_type_node
		  && TREE_VALUE (type) == TREE_TYPE (parm))
		{
		  error ("a formal parameter type that promotes to `int'");
		  error ("can match only `int' in the prototype");
		}
	      if (DECL_ARG_TYPE (parm) == double_type_node
		  && TREE_VALUE (type) == TREE_TYPE (parm))
		{
		  error ("a formal parameter type that promotes to `double'");
		  error ("can match only `double' in the prototype");
		}
	    }
	}
    }

  /* Now store the final chain of decls for the arguments
     as the decl-chain of the current lexical scope.
     Put the enumerators in as well, at the front so that
     DECL_ARGUMENTS is not modified.  */

  storedecls (chainon (nonparms, DECL_ARGUMENTS (fndecl)));

  /* Now fill in DECL_REFERENCE_SLOT for any of the parm decls.
     A parameter is assumed not to have any side effects.
     If this should change for any reason, then this
     will have to wrap the bashed reference type in a save_expr.

     Also, if the parameter type is declared to be an X
     and there is an X(X&) constructor, we cannot lay it
     into the stack (any more), so we make this parameter
     look like it is really of reference type.  Functions
     which pass parameters to this function will know to
     create a temporary in their frame, and pass a reference
     to that.  */
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = TREE_CHAIN (parm))
    {
#if 0
      if (TYPE_NEEDS_CONSTRUCTING (TREE_TYPE (parm)))
	{
	  warning_with_decl (parm, "parameter `%s' is really a reference to a temp in caller's frame");
	  TREE_TYPE (parm) = build_reference_type (TREE_TYPE (parm));
	  DECL_ARG_TYPE (parm) = TREE_TYPE (parm);
	}
#endif
      if (DECL_NAME (parm)
	  && TREE_CODE (TREE_TYPE (parm)) == REFERENCE_TYPE
	  && TYPE_SIZE (TREE_TYPE (TREE_TYPE (parm))))
	SET_DECL_REFERENCE_SLOT (parm, build (INDIRECT_REF, TREE_TYPE (TREE_TYPE (parm)), parm));
    }

  /* Initialize the RTL code for the function.  */
  DECL_SAVED_INSNS (fndecl) = 0;
  expand_function_start (fndecl, parms_have_cleanups);

  /* Create a binding contour which can be used to catch
     cleanup-generated temporaries.  Also, if the return value needs or
     has initialization, deal with that now.  */
  pushlevel (0);
  expand_start_bindings (0);
  current_function_parms_stored = 1;
}

void
store_return_init (init)
     tree init;
{
  tree decl = DECL_RESULT (current_function_decl);

  /* Can't let this happen for constructors.  */
  if (DECL_CONSTRUCTOR_P (current_function_decl))
    {
      error ("can't redefine default return value for constructors");
      return;
    }

  /* If we have a named return value, put that in our scope as well.  */
  if (DECL_NAME (decl) != value_identifier)
    {
      /* Let `finish_decl' know that this initializer is ok.  */
      DECL_INITIAL (decl) = init;
      pushdecl (decl);
      finish_decl (decl, init, 0);
    }
}

void
setup_vtbl_ptr ()
{
  if (! flag_this_is_variable
      && current_class_type
      && TYPE_VIRTUAL_P (current_class_type))
    {
      tree vfield = build_vfield_ref (C_C_D, current_class_type);
      current_vtable_decl = DECL_VTBL_PTR (TYPE_NAME (current_class_type));
      DECL_RTL (current_vtable_decl) = 0;
      DECL_INITIAL (current_vtable_decl) = error_mark_node;
      finish_decl (current_vtable_decl, vfield, 0);
      current_vtable_decl = build_indirect_ref (current_vtable_decl, "vtbl decl (compiler error)");
    }
  else
    current_vtable_decl = NULL_TREE;
}

/* Finish up a function declaration and compile that function
   all the way to assembler language output.  The free the storage
   for the function definition.

   This is called after parsing the body of the function definition.
   LINENO is the current line number.

   C++: CALL_POPLEVEL is non-zero if an extra call to poplevel
   (and expand_end_bindings) must be made to take care of the binding
   contour for the base initialazers.  This is only relevant for
   constructors.  */

void
finish_function (lineno, call_poplevel)
     int lineno;
     int call_poplevel;
{
  register tree fndecl = current_function_decl;
  tree fntype = TREE_TYPE (fndecl);
  tree rval;
  struct rtx_def *head, *last_parm_insn, *mark;
  extern struct rtx_def *get_last_insn ();
  extern struct rtx_def *cleanup_label;

/*  TREE_READONLY (fndecl) = 1;
    This caused &foo to be of type ptr-to-const-function
    which then got a warning when stored in a ptr-to-function variable.  */

  /* This happens on strange parse errors.  */
  if (! current_function_parms_stored)
    {
      call_poplevel = 0;
      store_parm_decls ();
    }

  /* Clean house because we will need to reorder insns here.  */
  do_pending_stack_adjust ();

  if (DESTRUCTOR_NAME_P (DECL_NAME (fndecl)))
    {
      tree cond = integer_one_node;
      tree member, expr, exprstmt;
      tree auto_delete_node = lookup_name (get_identifier (AUTO_DELETE_NAME));
      int i, n_baseclasses = CLASSTYPE_N_BASECLASSES (current_class_type);
      tree basetype = n_baseclasses > 0
	? CLASSTYPE_BASECLASS (current_class_type, 1) : NULL_TREE;
      int ok_to_optimize_dtor = 0;

      if (current_function_assigns_this)
	{
	  cond = build_binary_op (NE_EXPR, current_class_decl, integer_zero_node);
	}
      else
	{
	  /* If this destructor is empty, then we don't need to check
	     whether `this' is NULL in some cases.  */
	  mark = get_last_insn ();
	  last_parm_insn = (struct rtx_def *)get_first_nonparm_insn ();
	  ok_to_optimize_dtor
	    = (mark == last_parm_insn
	       && (n_baseclasses == 0
		   || TYPE_HAS_DESTRUCTOR (basetype)));
	}

      /* These initializations might go inline.  Protect
	 the binding level of the parms.  */
      pushlevel (0);

      /* If there were no base classes needing destructors, then
	 BASETYPE will still be 0 here.  */
      if (current_function_assigns_this)
	{
	  TREE_ANY_ASSIGNS_THIS (current_class_type) = 1;
	  current_function_assigns_this = 0;
	  current_function_just_assigned_this = 0;
	}

      /* Generate the code to call destructor on base class.
	 If this destructor belongs to a class with virtual
	 functions, then set the virtual function table
	 pointer to represent the type of our base class.  */

      /* This side-effect makes call to `build_delete' generate the
	 code we have to have at the end of this destructor.  */
      TYPE_HAS_DESTRUCTOR (current_class_type) = 0;

      exprstmt = build_delete (current_class_type, C_C_D, auto_delete_node, 0);
      if (basetype && TYPE_VIRTUAL_P (basetype))
	exprstmt = build (COMPOUND_EXPR, void_type_node,
			  build_virtual_init (basetype, basetype, current_class_decl),
			  exprstmt);

      TYPE_HAS_DESTRUCTOR (current_class_type) = 1;

      /* If we did not assign to this, then `this' is non-zero at
	 the end of a destructor.  As a special optimization, don't
	 emit test if this is an empty destructor.  If it does nothing,
	 it does nothing.  If it calls a base constructor, the base
	 constructor will perform the test.  */

      if (exprstmt != error_mark_node)
	{
	  if (cond != integer_one_node)
	    expand_start_cond (cond, 0);
	  expand_expr_stmt (exprstmt);
	  do_pending_stack_adjust ();
	  if (cond != integer_one_node)
	    expand_end_cond ();
	}

      poplevel (2, 0, 0);

      /* Dont execute destructor code if `this' is NULL.  */
      mark = get_last_insn ();
      last_parm_insn = (struct rtx_def *)get_first_nonparm_insn ();
      if (last_parm_insn == 0) last_parm_insn = mark;
      else last_parm_insn = (struct rtx_def *) previous_insn (last_parm_insn);

      if (! ok_to_optimize_dtor)
	{
	  cond = build_binary_op (NE_EXPR, current_class_decl, integer_zero_node);
	  expand_start_cond (cond, 0);
	  reorder_insns (next_insn (mark), get_last_insn (), last_parm_insn);
	  expand_end_cond ();
	}
    }
  else if (current_function_assigns_this)
    {
      /* Does not need to call finish_base_init, because
	 that is done (if needed) just after assignment to this
	 is seen.  */

      TREE_ANY_ASSIGNS_THIS (current_class_type) = 1;

      if (DECL_CONSTRUCTOR_P (current_function_decl))
	{
	  if (call_poplevel)
	    {
	      expand_end_bindings (getdecls (), 1, 0);
	      poplevel (1, 1, 0);
	    }
	  c_expand_return (current_class_decl);
	}

      current_function_assigns_this = 0;
      current_function_just_assigned_this = 0;
    }
  else if (DECL_CONSTRUCTOR_P (fndecl))
    {
      tree cond, thenclause;
      /* Allow constructor for a type to get a new instance of the object
	 using `build_new'.  */
      tree uninherited_virtuals = CLASSTYPE_UNINHERITED_VIRTUALS (current_class_type);
      CLASSTYPE_UNINHERITED_VIRTUALS (current_class_type) = NULL_TREE;

      DECL_RETURNS_FIRST_ARG (fndecl) = 1;

      cond = build_binary_op (EQ_EXPR, current_class_decl, integer_zero_node);
      thenclause = build_modify_expr (current_class_decl, NOP_EXPR,
				      build_x_new (current_class_name, void_type_node));
      CLASSTYPE_UNINHERITED_VIRTUALS (current_class_type) = uninherited_virtuals;

      /* must keep the first insn safe.  */
      head = (struct rtx_def *)get_insns ();

      /* this note will come up to the top with us.  */
      mark = get_last_insn ();

      /* These initializations might go inline.  Protect
	 the binding level of the parms.  */
      pushlevel (0);

      expand_start_cond (cond, 0);
      expand_expr_stmt (thenclause);
      /* Yow!  */
      if (TYPE_USES_VIRTUAL_BASECLASSES (current_class_type))
	expand_aggr_vbase_init (current_class_type, C_C_D,
				current_class_decl);
      expand_end_cond ();

      /* Make `finish_base_init' set up virtual function table pointer(s).  */
      finish_base_init (0);

      poplevel (2, 0, 0);

      /* This is where the real function begins.  If there were no
	 insns in this function body, then the last_parm_insn
	 is also the last insn.

	 If optimization is enabled, last_parm_insn may move, so
	 we don't hold on to it (across finish_base_init).  */
      last_parm_insn = (struct rtx_def *)get_first_nonparm_insn ();
      if (last_parm_insn == 0) last_parm_insn = mark;
      else last_parm_insn = (struct rtx_def *) previous_insn (last_parm_insn);

      reorder_insns (next_insn (mark), get_last_insn (), last_parm_insn);

      if (call_poplevel)
	{
	  expand_end_bindings (getdecls (), 1, 0);
	  poplevel (1, 1, 0);
	}

      c_expand_return (current_class_decl);

      current_function_assigns_this = 0;
      current_function_just_assigned_this = 0;
    }

  /* remove the binding contour which is used
     to catch cleanup-generated temporaries.  */
  expand_end_bindings (0, 0, 0);
  poplevel (0, 0, 0);

  /* Must mark the RESULT_DECL as being in this function.  */

  DECL_CONTEXT (DECL_RESULT (fndecl)) = DECL_INITIAL (fndecl);

  /* Obey `register' declarations if `setjmp' is called in this fn.  */
  if (flag_traditional && current_function_calls_setjmp)
    setjmp_protect (DECL_INITIAL (fndecl));

  /* Generate rtl for function exit.  */
  head = get_last_insn ();
  expand_function_end (input_filename, lineno);

  if (cleanup_label)
    {
      mark = get_last_insn ();
      /* Emit label at beginning of cleanup code for parmeters.  */
      emit_label (cleanup_label);
    }

  expand_end_bindings (0, 0, 0);

  /* reset scope for C++: if we were in the scope of a class,
     then when we finish this function, we are not longer so.
     This cannot be done until we know for sure that no more
     class members will ever be referenced in this function
     (i.e., calls to destructors).  */
  if (current_class_name)
    popclass (1);
  else
    pop_memoized_context (1);

  /* Forget about all overloaded functions defined in
     this scope which go away.  */
  while (overloads_to_forget)
    {
      IDENTIFIER_GLOBAL_VALUE (TREE_PURPOSE (overloads_to_forget))
	= TREE_VALUE (overloads_to_forget);
      overloads_to_forget = TREE_CHAIN (overloads_to_forget);
    }

  poplevel (1, 0, 1);

  if (cleanup_label)
    reorder_insns (next_insn (mark), get_last_insn (), head);

  /* So we can tell if jump_optimize sets it to 1.  */
  current_function_returns_null = 0;

  /* Run the optimizers and output the assembler code for this function.  */
  rest_of_compilation (fndecl);

  if (TREE_THIS_VOLATILE (fndecl) && current_function_returns_null)
    warning ("`volatile' function does return");
  else if (warn_return_type && current_function_returns_null
	   && TREE_TYPE (fntype) != void_type_node)
    /* If this function returns non-void and control can drop through,
       complain.  */
    warning ("control reaches end of non-void function");
  /* With just -W, complain only if function returns both with
     and without a value.  */
  else if (extra_warnings
	   && current_function_returns_value && current_function_returns_null)
    warning ("this function may return with or without a value");

  /* Free all the tree nodes making up this function.  */
  /* Switch back to allocating nodes permanently
     until we start another function.  */
  permanent_allocation ();

  if (DECL_SAVED_INSNS (fndecl) == 0)
    {
      /* Stop pointing to the local nodes about to be freed.  */
      /* But DECL_INITIAL must remain nonzero so we know this
	 was an actual function definition.  */
      DECL_INITIAL (fndecl) = error_mark_node;
      DECL_ARGUMENTS (fndecl) = 0;
    }

  /* Let the error reporting routines know that we're outside a function.  */
  current_function_decl = NULL_TREE;
  clear_anon_parm_name ();
}

/* Create the FUNCTION_DECL for a function definition.
   LINE1 is the line number that the definition absolutely begins on.
   LINE2 is the line number that the name of the function appears on.
   DECLSPECS and DECLARATOR are the parts of the declaration;
   they describe the function's name and the type it returns,
   but twisted together in a fashion that parallels the syntax of C.

   This function creates a binding context for the function body
   as well as setting up the FUNCTION_DECL in current_function_decl.

   Returns a FUNCTION_DECL on success.

   If the DECLARATOR is not suitable for a function (it defines a datum
   instead), we return 0, which tells yyparse to report a parse error.

   May return void_type_node indicating that this method is actually
   a friend.  See grokfield for more details.

   Came here with a `.pushlevel' .

   DO NOT MAKE ANY CHANGES TO THIS CODE WITHOUT MAKING CORRESPONDING
   CHANGES TO CODE IN `grokfield'.  */
tree
start_method (declspecs, declarator)
     tree declarator, declspecs;
{
  tree fndecl = grokdeclarator (declarator, declspecs, MEMFUNCDEF, 0);

  /* Something too ugly to handle.  */
  if (fndecl == 0)
    return 0;

  /* Pass friends back.  */
  if (fndecl == void_type_node)
    return void_type_node;

  if (TREE_CODE (fndecl) != FUNCTION_DECL)
    {
      /* Not a function, tell parser to report parse error.  */
      return 0;
    }

  if (DECL_IN_AGGR_P (fndecl))
    {
      if (IDENTIFIER_ERROR_LOCUS (DECL_NAME (fndecl)) != current_class_type)
	error_with_decl (fndecl, "`%s' is already defined in aggregate scope");
      return void_type_node;
    }
  if (TREE_CODE (fndecl) == FUNCTION_DECL
	   && TREE_CHAIN (fndecl) != NULL_TREE)
    {
      /* Need a fresh node here so that we don't get circularity
	 when we link these together.  */
      fndecl = copy_node (fndecl);
    }

  finish_decl (fndecl, NULL, NULL);

  {
    tree type = TREE_TYPE (DECL_ORIGINAL_NAME (fndecl));

    if (type && TREE_CODE (type) == TYPE_DECL
	&& TREE_TYPE (type) == current_class_type)
      grok_ctor_properties (fndecl);
    else if (OPERATOR_NAME_P (DECL_NAME (fndecl)))
      {
	TREE_OPERATOR (fndecl) = 1;
	grok_op_properties (fndecl);
      }
  }

  /* Make a place for the parms */
  pushlevel (0);
  current_binding_level->parm_flag = 1;
  
  DECL_IN_AGGR_P (fndecl) = 1;
  return fndecl;
}

/* Go through the motions of finishing a function definition.
   We don't compile this method until after the whole class has
   been processed.

   FINISH_METHOD must return something that looks as though it
   came from GROKFIELD (since we are defining a method, after all).

   This is called after parsing the body of the function definition.
   STMTS is the chain of statements that makes up the function body.

   DECL is the ..._DECL that `start_method' provided.  */

tree
finish_method (decl)
     tree decl;
{
  register tree fndecl = decl;
  int old_uid;
  tree old_initial;

  register tree link;

  if (decl == void_type_node)
    return decl;

  old_initial = DECL_INITIAL (fndecl);

  /* Undo the level for the parms (from start_method).
     This is like poplevel, but it causes nothing to be
     saved.  Saving information here confuses symbol-table
     output routines.  Besides, this information will
     be correctly output when this method is actually
     compiled.  */
  {
    tree tags = current_binding_level->tags;
    tree subblocks = current_binding_level->blocks;
    tree block = 0;

    /* Clear out the meanings of the local variables of this level;
       also record in each decl which block it belongs to.  */

    for (link = current_binding_level->names; link; link = TREE_CHAIN (link))
      {
	if (DECL_NAME (link) != 0)
	  {
	    IDENTIFIER_LOCAL_VALUE (DECL_NAME (link)) = 0;
	  }
	DECL_CONTEXT (link) = 0;
      }

    /* Restore all name-meanings of the outer levels
       that were shadowed by this level.  */

    for (link = current_binding_level->shadowed; link; link = TREE_CHAIN (link))
      IDENTIFIER_LOCAL_VALUE (TREE_PURPOSE (link)) = TREE_VALUE (link);

    /* Pop the current level, and free the structure for reuse.  */

    {
      register struct binding_level *level = current_binding_level;
      current_binding_level = current_binding_level->level_chain;

      level->level_chain = free_binding_level;
      free_binding_level = level;
      if (current_binding_level->parm_flag < 0)
	{
	  class_binding_level = current_binding_level;
	  do
	    {
	      current_binding_level = current_binding_level->level_chain;
	    }
	  while (current_binding_level->parm_flag < 0);
	}
    }
  }

  DECL_INITIAL (fndecl) = old_initial;
  return decl;
}

/* Hand off a unique name which can be used for variable we don't really
   want to know about anyway, for example, the anonymous variables which
   are needed to make references work.  Declare this thing so we can use it.
   The variable created will be of type TYPE.  */

tree
get_temp_name (type)
     tree type;
{
  char buf[sizeof (AUTO_TEMP_FORMAT) + 12];
  tree decl;
  int temp = 0;

  if (current_binding_level == global_binding_level)
    {
      temp = allocation_temporary_p ();
      if (temp)
	end_temporary_allocation ();
      sprintf (buf, AUTO_TEMP_FORMAT, global_temp_name_counter++);
    }
  else
    {
      sprintf (buf, AUTO_TEMP_FORMAT, temp_name_counter++);
    }

  decl = pushdecl (build_decl (VAR_DECL, get_identifier (buf), type));
  TREE_USED (decl) = 1;
  /* If this is a local variable, then lay out its rtl now.
     Otherwise, callers of this function are responsible for dealing
     with this variable's rtl.  */
  if (current_binding_level != global_binding_level)
    {
      expand_decl (decl, 0);
      expand_decl_init (decl);
    }
  else if (temp)
    {
      resume_temporary_allocation ();
    }
  return decl;
}

/* Get a variable which we can use for multiple assignments.
   It is not entered into current_binding_level, because
   that breaks things when it comes time to do final cleanups
   (which take place "outside" the binding contour of the function).

   Because it is not entered into the binding contour, `expand_end_bindings'
   does not see this variable automatically.  Users of this function
   must either pass this variable to expand_end_bindings or do
   themselves what expand_end_bindings was meant to do (like keeping
   the variable live if -noreg was specified).  */
tree
get_temp_regvar (type, init)
     tree type, init;
{
  static char buf[sizeof (AUTO_TEMP_FORMAT) + 8] = { '_' };
  tree decl;

  sprintf (buf+1, AUTO_TEMP_FORMAT, temp_name_counter++);
  decl = pushdecl (build_decl (VAR_DECL, get_identifier (buf), type));
  TREE_USED (decl) = 1;
  TREE_REGDECL (decl) = 1;

  if (init)
    store_init_value (decl, init);

  /* We can expand these without fear, since they cannot need
     constructors or destructors.  */
  expand_decl (decl, NULL_TREE);
  expand_decl_init (decl);
  return decl;
}

/* Make the macro TEMP_NAME_P available to units which do not
   include c-tree.h.  */
int
temp_name_p (decl)
     tree decl;
{
  return TEMP_NAME_P (decl);
}

/* Called when a new struct TYPE is defined.
   If this structure or union completes the type of any previous
   variable declaration, lay it out and output its rtl.  */

void
hack_incomplete_structures (type)
     tree type;
{
  if (current_binding_level->n_incomplete != 0)
    {
      tree decl;
      for (decl = current_binding_level->names; decl; decl = TREE_CHAIN (decl))
	if (TREE_TYPE (decl) == type
	    && TREE_CODE (decl) != TYPE_DECL)
	  {
	    int toplevel = global_binding_level == current_binding_level;
	    layout_decl (decl, 0);
	    rest_of_decl_compilation (decl, 0, toplevel, 0);
	    if (! toplevel)
	      {
		expand_decl (decl, maybe_build_cleanup (decl));
		expand_decl_init (decl);
		--current_binding_level->n_incomplete;
	      }
	  }
    }
}

void
mark_inline_for_output (decl)
     tree decl;
{
  pending_addressable_inlines = perm_tree_cons (NULL_TREE, decl,
						pending_addressable_inlines);
}

/* Finish and output a table which is generated by the compiler.
   NAME is the name to give the table.
   TYPE is the type of the table entry.
   INIT is all the elements in the table.
   PUBLICP is non-zero if this table should be given external visibility.  */
tree
finish_table (name, type, init, publicp)
     tree name, type, init;
{
  tree itype, atype, decl;

  itype = build_index_type (build_int_2 (list_length (init), 0));
  atype = build_array_type (type, itype);
  layout_type (atype);
  decl = build_decl (VAR_DECL, name, atype);
  decl = pushdecl (decl);
  TREE_STATIC (decl) = 1;
  TREE_PUBLIC (decl) = publicp;
  init = build (CONSTRUCTOR, atype, NULL_TREE, init);
  TREE_LITERAL (init) = 1;
  TREE_STATIC (init) = 1;
  DECL_INITIAL (decl) = init;
  finish_decl (decl, init,
	       build_string (IDENTIFIER_LENGTH (DECL_NAME (decl)),
			     IDENTIFIER_POINTER (DECL_NAME (decl))));
  return decl;
}

/* This is temporary until later.  */
/* Construct, lay out and return the type of objects which are of type TYPE
   as members of type BASETYPE.  If that type exists already, reuse it.  */
tree
build_member_type (basetype, type)
     tree basetype, type;
{
  register tree t;
  int hashcode;

  /* Make a node of the sort we want.  */
  if (TREE_CODE (type) == FUNCTION_TYPE)
    {
      abort ();
      t = build_method_type (basetype, type);
    }
  else
    {
      int hashcode;

      t = make_node (OFFSET_TYPE);
      TYPE_OFFSET_BASETYPE (t) = basetype;
      TREE_TYPE (t) = type;
      hashcode = TREE_UID (basetype) + TREE_UID (type);
      t = type_hash_canon (hashcode, t);
    }
  return t;
}

char *
lang_printable_name (decl)
     tree decl;
{
  if (DECL_PRINTABLE_NAME (decl) == 0)
    {
      if (TREE_CODE (decl) == FUNCTION_DECL)
	{
	  int temp = allocation_temporary_p ();
	  char *name = (char *)fndecl_as_string (0, 0, decl, 1);
	  end_temporary_allocation ();
	  DECL_PRINTABLE_NAME (decl) = oballoc (strlen (name) + 1);
	  strcpy (DECL_PRINTABLE_NAME (decl), name);
	  if (temp)
	    resume_temporary_allocation ();
	}
      else if (DECL_NAME (decl) == 0)
	DECL_PRINTABLE_NAME (decl) = "((anonymous))";
      else
	DECL_PRINTABLE_NAME (decl) = IDENTIFIER_POINTER (DECL_NAME (decl));
    }
  return DECL_PRINTABLE_NAME (decl);
}

char *
lang_rtl_name (decl)
     tree decl;
{
  if (DECL_RTL_NAME (decl) == 0)
    {
      tree name;

      if (TREE_CODE (decl) == FUNCTION_DECL)
	{
	  if (DECL_LANGUAGE (decl) == lang_cplusplus)
	    name = DECL_NAME (decl);
	  else if (DECL_LANGUAGE (decl) == lang_c)
	    name = DECL_ORIGINAL_NAME (decl);
	  else abort ();
	}
      else name = DECL_NAME (decl);
      DECL_RTL_NAME (decl) = IDENTIFIER_POINTER (name);
    }

  return DECL_RTL_NAME (decl);
}
