/* Breadth-first and depth-first routines for
   searching multiple-inheritance lattice for GNU C++.
   Copyright (C) 1987 Free Software Foundation, Inc.
   Contributed by Michael Tiemann (tiemann@mcc.com)

This file is part of GNU CC.
   
GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


/* High-level class interface. */

#include "config.h"
#include "tree.h"
#include "cplus-tree.h"
#include "obstack.h"
#include "flags.h"
#include "assert.h"
#include <stdio.h>

#define NULL 0

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

extern int xmalloc ();
extern void free ();

void init_search ();

/* Stack of places to restore the search obstack back to.  */
   
struct search_level
{
  /* Pointer back to previous such level.  */
  struct search_level *prev;
  /* First place we start putting data.  */
  tree *first;
  /* Number of entries we can have from `first'.
     Right now we are dumb: if we overflow, abort.  */
  int limit;
};

/* Obstack used for remembering decision points of breadth-first.  */
static struct obstack search_obstack;
static struct search_level *search_stack;

static tree lookup_fields_1 (), lookup_fnfields_1 ();

/* Allocate a level of searching.  */
static struct search_level *
push_search_level (stack, obstack)
     struct search_level *stack;
     struct obstack *obstack;
{
  struct search_level tem;

  tem.prev = stack;
  stack = (struct search_level *) obstack_next_free (obstack);
  obstack_grow (obstack, &tem, sizeof (tem));
  obstack_finish (obstack);
  stack->first = (tree *) obstack_base (obstack);
  stack->limit = obstack_room (obstack) / sizeof (tree *);
  return stack;
}

/* Discard a level of search allocation.  */

static struct search_level *
pop_search_level (stack, obstack)
     struct search_level *stack;
     struct obstack *obstack;
{
  struct search_level *tem = stack;
  stack = tem->prev;
  obstack_free (obstack, tem);
  return stack;
}

/* Search memoization.  */
struct type_level
{
  /* Pointer back to previous such level.  */
  struct type_level *prev;
  /* First object allocated in obstack of entries.  */
  char *entries;
  /* Number of types memoized in this context.  */
  int len;
  /* Type being memoized; save this if we are saving
     memoized contexts.  */
  tree type;

  /* First place we start putting data.  */
  tree *first;
  /* Number of entries we can have from `first'.
     Right now we are dumb: if we overflow, abort.  */
  int limit;
};

/* Obstack used for memoizing member and member function lookup.  */

static struct obstack type_obstack, type_obstack_entries;
static struct type_level *type_stack;

/* Make things that look like tree nodes, but allocate them
   on type_obstack_entries.  */
static int my_tree_node_counter;
static tree my_tree_cons (), my_build_string ();

extern int flag_memoize_lookups, flag_save_memoized_contexts;

/* Variables for gathering statistics.  */
static int my_memoized_entry_counter;
static int memoized_fast_finds[2], memoized_adds[2], memoized_fast_rejects[2];
static int memoized_fields_searched[2];
static int n_fields_searched;
static int n_calls_lookup_field, n_calls_lookup_field_1;
static int n_calls_lookup_fnfields, n_calls_lookup_fnfields_1;
static int n_calls_get_base_type;
static int n_outer_fields_searched;
static int n_contexts_saved;

/* Local variables to help save memoization contexts.  */
static tree prev_type_memoized;
static struct type_level *prev_type_stack;

/* Allocate a level of type memoziation context.  */
static struct type_level *
push_type_level (stack, obstack)
     struct type_level *stack;
     struct obstack *obstack;
{
  struct type_level tem;

  tem.prev = stack;
  obstack_finish (&type_obstack_entries);
  tem.entries = (char *) obstack_base (&type_obstack_entries);
  tem.len = 0;
  tem.type = NULL_TREE;
  stack = (struct type_level *) obstack_next_free (obstack);
  obstack_grow (obstack, &tem, sizeof (tem));
  obstack_finish (obstack);
  stack->first = (tree *) obstack_base (obstack);
  stack->limit = obstack_room (obstack) / sizeof (tree *);
  return stack;
}

/* Discard a level of type memoziation context.  */

static struct type_level *
pop_type_level (stack, obstack)
     struct type_level *stack;
     struct obstack *obstack;
{
  struct type_level *tem = stack;
  stack = tem->prev;
  obstack_free (&type_obstack_entries, tem->entries);
  obstack_free (obstack, tem);
  return stack;
}

/* Make something that looks like a TREE_LIST, but
   do it on the type_obstack_entries obstack.  */
static tree
my_tree_cons (purpose, value, chain)
     tree purpose, value, chain;
{
  tree p = (tree)obstack_alloc (&type_obstack_entries, sizeof (struct tree_list));
  bzero (&p->common, sizeof (p->common));
  TREE_UID (p) = ++my_tree_node_counter;
  TREE_CODE (p) = TREE_LIST;
  TREE_PURPOSE (p) = purpose;
  TREE_VALUE (p) = value;
  TREE_CHAIN (p) = chain;
  return p;
}

static tree
my_build_string (str)
     char *str;
{
  tree p = (tree)obstack_alloc (&type_obstack_entries, sizeof (struct tree_string));
  bzero (&p->common, sizeof (p->common));
  TREE_UID (p) = ++my_tree_node_counter;
  TREE_CODE (p) = STRING_CST;
  TREE_STRING_POINTER (p) = str;
  TREE_STRING_LENGTH (p) = strlen (str);
  return p;
}

static tree
my_copy_node (node)
     tree node;
{
  extern struct obstack *current_obstack;
  struct obstack *ambient_obstack = current_obstack;
  tree t;

  current_obstack = &type_obstack_entries;

  t = copy_node (node);
  TREE_UID (t) = ++my_tree_node_counter;

  current_obstack = ambient_obstack;
  return t;
}

/* Memoizing machinery to make searches for multiple inheritance
   reasonably efficient.  */
#define MEMOIZE_HASHSIZE 8
typedef struct memoized_entry
{
  struct memoized_entry *chain;
  int uid;
  tree data_members[MEMOIZE_HASHSIZE];
  tree function_members[MEMOIZE_HASHSIZE];
} *ME;

#define MEMOIZED_CHAIN(ENTRY) (((ME)ENTRY)->chain)
#define MEMOIZED_UID(ENTRY) (((ME)ENTRY)->uid)
#define MEMOIZED_FIELDS(ENTRY,INDEX) (((ME)ENTRY)->data_members[INDEX])
#define MEMOIZED_FNFIELDS(ENTRY,INDEX) (((ME)ENTRY)->function_members[INDEX])
#define MEMOIZED_HASH_FN(NODE) (TREE_UID (NODE)&(MEMOIZE_HASHSIZE - 1))

static struct memoized_entry *
my_new_memoized_entry (chain)
     struct memoized_entry *chain;
{
  struct memoized_entry *p =
    (struct memoized_entry *)obstack_alloc (&type_obstack_entries,
					    sizeof (struct memoized_entry));
  bzero (p, sizeof (struct memoized_entry));
  MEMOIZED_CHAIN (p) = chain;
  MEMOIZED_UID (p) = ++my_memoized_entry_counter;
  return p;
}

/* When a new function or class context is entered, we build
   a table of types which have been searched for members.
   The table is an array (obstack) of types.  When a type is
   entered into the obstack, is CLASSTYPE_MTABLE_ENTRY
   field is set to point to a new record, of type struct memoized_entry.

   A non-NULL TREE_TYPE of the entry contains a visibility error message.

   The slots for the data members are arrays of tree nodes.
   These tree nodes are lists, with the TREE_PURPOSE
   of this list the known member name, and the TREE_VALUE
   as the FIELD_DECL for the member.

   For member functions, the TREE_PURPOSE is again the
   name of the member functions for that class,
   and the TREE_VALUE of the list is a pairs
   whose TREE_PURPOSE is a member functions of this name,
   and whose TREE_VALUE is a list of known argument lists this
   member function has been called with.  The TREE_TYPE of the pair,
   if non-NULL, is an error message to print.  */

/* Tell search machinery that we are entering a new context, and
   to update tables appropriately.

   TYPE is the type of the context we are entering, which can
   be NULL_TREE if we are not in a class's scope.

   USE_OLD, if nonzero tries to use previous context.  */
void
push_memoized_context (type, use_old)
     tree type;
     int use_old;
{
  int len;
  tree *tem;

  if (prev_type_stack)
    {
      if (use_old && prev_type_memoized == type)
	{
	  n_contexts_saved++;
	  type_stack = prev_type_stack;
	  prev_type_stack = 0;

	  tem = &type_stack->first[0];
	  len = type_stack->len;
	  while (len--)
	    CLASSTYPE_MTABLE_ENTRY (tem[len*2]) = tem[len*2+1];
	  return;
	}
      /* Otherwise, need to pop old stack here.  */
      type_stack = pop_type_level (prev_type_stack, &type_obstack);
      prev_type_memoized = 0;
      prev_type_stack = 0;
    }

  if (flag_memoize_lookups)
    {
      type_stack = push_type_level (type_stack, &type_obstack);
      type_stack->type = type;
    }
}

/* Tell search machinery that we have left a context.
   We do not currently save these contexts for later use.
   If we wanted to, we could not use pop_search_level, since
   poping that level allows the data we have collected to
   be clobbered; a stack of obstacks would be needed.  */
pop_memoized_context (use_old)
     int use_old;
{
  int i, len;
  tree *tem = &type_stack->first[0];

  if (! flag_save_memoized_contexts)
    use_old = 0;
  else if (use_old)
    {
      len = type_stack->len;
      while (len--)
	tem[len*2+1] = (tree)CLASSTYPE_MTABLE_ENTRY (tem[len*2]);

      prev_type_stack = type_stack;
      prev_type_memoized = type_stack->type;
    }

  if (flag_memoize_lookups)
    {
      len = type_stack->len;
      while (len--)
	CLASSTYPE_MTABLE_ENTRY (tem[len*2])
	  = MEMOIZED_CHAIN (CLASSTYPE_MTABLE_ENTRY (tem[len*2]));

      if (! use_old)
	type_stack = pop_type_level (type_stack, &type_obstack);
      else
	type_stack = type_stack->prev;
    }
}

/* Some simple list processing predicates.  */

/* Check whether TYPE is immediately derived from PARENT.
   Return actual base information if so.  Otherwise, return 0.  */
tree
get_base_type_1 (parent, type)
     register tree parent, type;
{
  register int i;

  parent = CLASSTYPE_MAIN_VARIANT (parent);
  type = CLASSTYPE_MAIN_VARIANT (type);

  for (i = 1; i < CLASSTYPE_N_BASECLASSES (type); i++)
    if (CLASSTYPE_MAIN_VARIANT (CLASSTYPE_THIS_BASECLASS (type, i)) == parent)
      return CLASSTYPE_THIS_BASECLASS (type, i);

  return 0;
}

/* Check whether TYPE is derived from PARENT.
   Return the actual base information if so.  Otherwise return 0.
   If PROTECT is nonzero, then emit an error message if
   TYPE is derived from PARENT via private visibility rules.  */
tree
get_base_type (parent, type, protect)
     register tree parent, type;
{
  tree xtype = type;
  tree otype;
  int head = 0, tail = 0;
  int is_private = 0;
  tree rval = 0;
  int rval_private = 0;

  n_calls_get_base_type++;

  search_stack = push_search_level (search_stack, &search_obstack);

  parent = CLASSTYPE_MAIN_VARIANT (parent);
  type = CLASSTYPE_MAIN_VARIANT (type);

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	if (CLASSTYPE_MARKED5 (CLASSTYPE_THIS_BASECLASS (type, i)) == 0)
	  {
	    int via_private = is_private || !CLASSTYPE_VIA_PUBLIC (type, i);
	    /* The immediate base class of the class we are in
	       does let its public members through.  */
	    if (type == current_class_type)
	      via_private = 0;

	    CLASSTYPE_MARKED5 (CLASSTYPE_THIS_BASECLASS (type, i)) = 1;
	    otype = type;
	    obstack_grow (&search_obstack, &CLASSTYPE_THIS_BASECLASS (type, i), sizeof (tree *));
	    obstack_grow (&search_obstack, &via_private, sizeof (int));
	    tail += 2;
	    if (tail >= search_stack->limit)
	      abort ();
	  }
	else if (protect && ! CLASSTYPE_VIA_VIRTUAL (type, i))
	  {
	    error_with_aggr_type (parent, "type `%s' is ambiguous base class for type `%s'",
				  TYPE_NAME_STRING (xtype));
	    error ("(base class for types `%s' and `%s')",
		   TYPE_NAME_STRING (type),
		   TYPE_NAME_STRING (otype));
	    rval = error_mark_node;
	    break;
	  }

    dont_queue:
      /* Process head of queue, if one exists.  */
      if (head >= tail)
	break;

      type = search_stack->first[head++];
      is_private = (int)search_stack->first[head++];
      if (CLASSTYPE_MAIN_VARIANT (type) == parent)
	{
	  if (rval == 0)
	    {
	      rval = type;
	      rval_private = is_private;
	    }
	  goto dont_queue;
	}
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

    while (tp < search_tail)
      {
	CLASSTYPE_MARKED5 (*tp++) = 0;
	tp += 1;
      }
  }
  search_stack = pop_search_level (search_stack, &search_obstack);

  if (rval == error_mark_node)
    return error_mark_node;

  if (rval && protect && rval_private)
    {
      error ("type `%s' is derived from private `%s'",
	     TYPE_NAME_STRING (xtype),
	     TYPE_NAME_STRING (parent));
      return error_mark_node;
    }

  return rval;
}

/* Return the number of levels between type PARENT and type TYPE,
   Return -1 if TYPE is not derived from PARENT.
   Return -2 if PARENT is an ambiguous base class of TYPE.
   Return -3 if PARENT is private to TYPE, and protect is non-zero.

   If PATH_PTR is non-NULL, then also build a list of type
   from TYPE to PARENT, with TREE_VIA_VIRUAL and TREE_VIA_PUBLIC
   set.  */
get_base_distance (parent, type, protect, path_ptr)
     register tree parent, type;
     int protect;
     tree *path_ptr;
{
  int head = 0, tail = 0;
  int is_private = 0;
  int rval;
  int depth = 0;
  int rval_private = 0;
  tree basetypes;

  search_stack = push_search_level (search_stack, &search_obstack);

  parent = CLASSTYPE_MAIN_VARIANT (parent);
  type = CLASSTYPE_MAIN_VARIANT (type);

  if (path_ptr)
    basetypes = CLASSTYPE_AS_LIST (type);

  if (parent == type)
    {
      /* If the distance is 0, then we don't really need
	 a path pointer, but we shouldn't let garbage go back.  */
      if (path_ptr)
	*path_ptr = basetypes;
      return 0;
    }

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	if (CLASSTYPE_MARKED5 (CLASSTYPE_THIS_BASECLASS (type, i)) == 0)
	  {
	    tree btypes;

	    int via_private = is_private || !CLASSTYPE_VIA_PUBLIC (type, i);
	    /* The immediate base class of the class we are in
	       does let its public members through.  */
	    if (type == current_class_type)
	      via_private = 0;

	    CLASSTYPE_MARKED5 (CLASSTYPE_THIS_BASECLASS (type, i)) = 1;
	    obstack_grow (&search_obstack, &CLASSTYPE_THIS_BASECLASS (type, i), sizeof (tree *));

	    obstack_grow (&search_obstack, (tree*)&depth, sizeof (tree *));
	    obstack_grow (&search_obstack, &via_private, sizeof (int));
	    if (path_ptr)
	      {
		btypes = tree_cons (NULL_TREE, CLASSTYPE_BASECLASS (type, i),
				    basetypes);
		TREE_VIA_PUBLIC (btypes) = CLASSTYPE_VIA_PUBLIC (type, i);
		TREE_VIA_VIRTUAL (btypes) = CLASSTYPE_VIA_VIRTUAL (type, i);
		obstack_grow (&search_obstack, &btypes, sizeof (tree *));
		tail += 1;
	      }
	    tail += 3;
	    if (tail >= search_stack->limit)
	      abort ();
	  }
	else if (! CLASSTYPE_VIA_VIRTUAL (type, i))
	  {
	    rval = -2;
	    goto done;
	  }

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	{
	  rval = -1;
	  break;
	}

      type = search_stack->first[head++];
      depth = (int)search_stack->first[head++] + 1;
      is_private = (int)search_stack->first[head++];
      if (path_ptr)
	basetypes = search_stack->first[head++];
      if (CLASSTYPE_MAIN_VARIANT (type) == parent)
	{
	  rval = depth;
	  rval_private = is_private;
	  break;
	}
    }
 done:
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

    while (tp < search_tail)
      {
	CLASSTYPE_MARKED5 (*tp++) = 0;
	if (path_ptr)
	  tp += 3;
	else
	  tp += 2;
      }
  }
  search_stack = pop_search_level (search_stack, &search_obstack);

  if (rval && protect && rval_private)
    return -3;

  if (path_ptr)
    *path_ptr = basetypes;
  return rval;
}

/* Search for a member with name NAME in a multiple inheritance lattice
   specified by TYPE.  If it does not exist, return NULL_TREE.
   If the member is ambiguously referenced, return `error_mark_node'.
   Otherwise, return the FIELD_DECL.  */

/* Do a 1-level search for NAME as a member of TYPE.  The caller
   must figure out whether it has a visible path to this field.
   (Since it is only one level, this is reasonable.)  */
static tree
lookup_field_1 (type, name)
     tree type, name;
{
  register tree field = TYPE_FIELDS (type);

  n_calls_lookup_field_1++;
  while (field)
    {
      n_fields_searched++;
      if (DECL_ANON_UNION_ELEM (field))
	{
	  tree temp = lookup_field_1 (TREE_TYPE (field), name);
	  if (temp)
	    return temp;
	}
      if (DECL_NAME (field) == name)
	break;
      field = TREE_CHAIN (field);
    }
  return field;
}

/* Compute the visibility of FIELD.  This is done by computing
   the visibility available to each type in BASETYPES (which comes
   as a list of [via_public/basetype] in reverse order, namely base
   class before derived class).  The first one which defines a
   visibility defines the visibility for the field.  Otherwise, the
   visibility of the field is that which occurs normally.

   Uses global variables CURRENT_CLASS_TYPE and
   CURRENT_FUNCTION_DECL to use friend relationships
   if necessary.

   This will be static when lookup_fnfield comes into this file.  */

#define PUBLIC_RETURN do { TREE_FIELD_PUBLIC (field) = 1; return visibility_public; } while (0)
#define PROTECTED_RETURN do { TREE_FIELD_PROTECTED (field) = 1; return visibility_protected; } while (0)
#define PRIVATE_RETURN do { TREE_FIELD_PRIVATE (field) = 1; return visibility_private; } while (0)

enum visibility_type
compute_visibility (basetypes, field)
     tree basetypes, field;
{
  enum visibility_type visibility = visibility_public;
  tree types, type;
  tree context = DECL_FIELD_CONTEXT (field);

  /* Virtual function tables are never private.
     But we should know that we are looking for this,
     and not even try to hide it.  */
  if (VFIELD_NAME_P (DECL_NAME (field)) == 1)
    return visibility_public;

  /* Make these special cases fast.  */
  if (TREE_VALUE (basetypes) == current_class_type)
    {
      if (TREE_FIELD_PUBLIC (field))
	return visibility_public;
      if (TREE_FIELD_PROTECTED (field))
	return visibility_protected;
      if (TREE_FIELD_PRIVATE (field))
	return visibility_private;
    }

  /* Member function manipulating its own members.  */
  if (current_class_type == context)
    PUBLIC_RETURN;

  /* Member found immediately within object.  */
  if (TREE_CHAIN (basetypes) == NULL_TREE)
    {
      /* At object's top level, public members are public.  */
      if (TREE_PROTECTED (field) == 0 && TREE_PRIVATE (field) == 0)
	PUBLIC_RETURN;

      /* Friend function manipulating members it gets (for being a friend).  */
      if (is_friend (context, current_function_decl))
	PUBLIC_RETURN;

      /* Inner than that, without special visibility,

	   protected members are ok if type of object is current_class_type
	   is derived therefrom.  This means that if the type of the object
	   is a base type for our current class type, we cannot access
	   protected members.

	   private members are not ok.  */
      if (current_class_type && DECL_VISIBILITY (field) == NULL_TREE)
	{
	  if (TREE_PRIVATE (field))
	    PRIVATE_RETURN;

	  if (TREE_PROTECTED (field))
	    {
	      if (context == current_class_type
		  || (type = get_base_type (current_class_type, context, 0)))
		PUBLIC_RETURN;
	      else
		PROTECTED_RETURN;
	    }
	  else abort ();
	}
    }
  /* Friend function manipulating members it gets (for being a friend).  */
  if (is_friend (context, current_function_decl))
    PUBLIC_RETURN;

  /* must reverse more than one element */
  basetypes = nreverse (basetypes);

  types = basetypes;

  while (types)
    {
      tree member;
      type = CLASSTYPE_MAIN_VARIANT (TREE_VALUE (types));

      member = purpose_member (type, DECL_VISIBILITY (field));
      if (member)
	{
	  visibility = (enum visibility_type)TREE_VALUE (member);
	  if (visibility == visibility_public
	      || is_friend (type, current_function_decl)
	      || (visibility == visibility_protected
		  && current_class_type
		  && get_base_type (context, current_class_type, 0)))
	    visibility = visibility_public;
	  goto ret;
	}

      /* Friends inherit the visibility of the class they inherit from.  */
      if (is_friend (type, current_function_decl))
	{
	  if (TREE_PROTECTED (field))
	    {
	      visibility = visibility_public;
	      goto ret;
	    }
	  if (TREE_PRIVATE (field))
	    if (type == context)
	      {
		visibility = visibility_public;
		goto ret;
	      }
	    else
	      /* may be a friend of a deeper base class */ ;
	  else if (visibility == visibility_public)
	    {
	      visibility = visibility_public;
	      goto ret;
	    }
	  else
	    /* may be a friend of a deeper base class */ ;
	}

      types = TREE_CHAIN (types);
      /* If the next type was not VIA_PUBLIC, then fields of all
	 remaining class past that one are private.  */
      if (types && ! TREE_VIA_PUBLIC (types))
	visibility = visibility_private;
    }

  /* No special visibilities apply.  Use normal rules.
     No assignment needed for BASETYPEs here from the nreverse.
     This is because we use it only for information about the
     path to the base.  The code earlier dealt with what
     happens when we are at the base level.  */

  if (visibility == visibility_public)
    {
      basetypes = nreverse (basetypes);
      if (TREE_PRIVATE (field))
	PRIVATE_RETURN;
      if (TREE_PROTECTED (field))
	{
	  /* Used to check if the current class type was derived from
	     the type that contains the field.  This is wrong for
	     multiple inheritance because is gives one class reference
	     to protected members via another classes protected path.
	     I.e., if A; B1 : A; B2 : A;  Then B1 and B2 can access
	     their own members which are protected in A, but not
	     those same members in one another.  */
	  if (
#if 1
	      current_class_type
	      && get_base_type (context, current_class_type, 0)
#else
	      current_class_type
	      && value_member (current_class_type, basetypes)
#endif
	      )
	    PUBLIC_RETURN;
	  PROTECTED_RETURN;
	}
      PUBLIC_RETURN;
    }

  if (visibility == visibility_private
      && current_class_type != NULL_TREE)
    {
      if (TREE_PRIVATE (field))
	{
	  nreverse (basetypes);
	  PRIVATE_RETURN;
	}

      /* See if the field isn't protected.  */
      if (TREE_PROTECTED (field))
	{
	  tree test;
#if 0
	  test = get_base_type (type, current_class_type, 0);
#else
	  test = value_member (current_class_type, basetypes);
#endif
	  nreverse (basetypes);
	  if (test)
	    PUBLIC_RETURN;
	  PROTECTED_RETURN;
	}

      /* See if the field isn't a public member of
	 a private base class.  */

      visibility = visibility_public;
      types = TREE_CHAIN (basetypes);
      while (types)
	{
	  if (! TREE_VIA_PUBLIC (types))
	    {
	      if (visibility == visibility_private)
		{
		  visibility = visibility_private;
		  goto ret;
		}
	      visibility = visibility_private;
	    }
	  if (TREE_VALUE (types) == context)
	    {
	      visibility = visibility_public;
	      goto ret;
	    }
	  types = TREE_CHAIN (types);
	}
      abort ();
    }

 ret:
  nreverse (basetypes);

  if (visibility == visibility_public)
    TREE_FIELD_PUBLIC (field) = 1;
  else if (visibility == visibility_protected)
    TREE_FIELD_PROTECTED (field) = 1;
  else if (visibility == visibility_private)
    TREE_FIELD_PRIVATE (field) = 1;
  else abort ();
  return visibility;
}

/* Make an entry in the memoized table for type TYPE
   that the entry for NAME is FIELD.  */

tree
make_memoized_table_entry (type, name, function_p)
     tree type, name;
     int function_p;     
{
  int index = MEMOIZED_HASH_FN (name);
  tree entry, *prev_entry;

  memoized_adds[function_p] += 1;
  if (CLASSTYPE_MTABLE_ENTRY (type) == NULL_TREE)
    {
      obstack_grow (&type_obstack, &type, sizeof (tree *));
      obstack_blank (&type_obstack, sizeof (struct memoized_entry *));
      CLASSTYPE_MTABLE_ENTRY (type) = my_new_memoized_entry (0);
      type_stack->len++;
      if (type_stack->len * 2 >= type_stack->limit)
	abort ();
    }
  if (function_p)
    prev_entry = &MEMOIZED_FNFIELDS (CLASSTYPE_MTABLE_ENTRY (type), index);
  else
    prev_entry = &MEMOIZED_FIELDS (CLASSTYPE_MTABLE_ENTRY (type), index);

  entry = my_tree_cons (name, 0, *prev_entry);
  *prev_entry = entry;

  /* Don't know the error message to give yet.  */
  TREE_TYPE (entry) = error_mark_node;

  return entry;
}

tree
lookup_field (xbasetype, name, protect)
     register tree xbasetype, name;
     int protect;
{
  int head = 0, tail = 0;
  tree type, rval;
  tree basetype, basetypes;
  enum visibility_type this_v = visibility_default;
  tree entry;
  enum visibility_type own_visibility = visibility_default;
  int vbase_name_p = VBASE_NAME_P (name);

  /* Things for memoization.  */
  char *errstr = 0;

  /* Set this to nonzero if we don't know how to compute
     accurate error messages for visibility.  */
  int memoization_too_hard = 0;
  int index = MEMOIZED_HASH_FN (name);

  if (TREE_CODE (xbasetype) == TREE_LIST)
    basetypes = xbasetype, basetype = TREE_VALUE (xbasetype);
  else
    basetypes = CLASSTYPE_AS_LIST (xbasetype), basetype = xbasetype;

  if (CLASSTYPE_MTABLE_ENTRY (basetype))
    {
      tree tem = MEMOIZED_FIELDS (CLASSTYPE_MTABLE_ENTRY (basetype), index);

      while (tem && TREE_PURPOSE (tem) != name)
	{
	  memoized_fields_searched[0]++;
	  tem = TREE_CHAIN (tem);
	}
      if (tem)
	{
	  if (protect && TREE_TYPE (tem))
	    {
	      error (TREE_STRING_POINTER (TREE_TYPE (tem)),
		     IDENTIFIER_POINTER (name),
		     TYPE_NAME_STRING (DECL_FIELD_CONTEXT (TREE_VALUE (tem))));
	      return error_mark_node;
	    }
	  if (TREE_VALUE (tem) == NULL_TREE)
	    memoized_fast_rejects[0] += 1;
	  else
	    memoized_fast_finds[0] += 1;
	  return TREE_VALUE (tem);
	}
    }

  n_calls_lookup_field++;
  if (flag_memoize_lookups && ! global_bindings_p ())
    entry = make_memoized_table_entry (basetype, name, 0);
  else
    entry = 0;

  rval = lookup_field_1 (basetype, name);

  if (rval)
    {
      if (flag_memoize_lookups || protect)
	{
	  if (TREE_PRIVATE (rval) | TREE_PROTECTED (rval))
	    this_v = compute_visibility (basetypes, rval);
	  if (TREE_CODE (rval) == CONST_DECL)
	    {
	      if (this_v == visibility_private)
		errstr = "enum `%s' is a private value of class `%s'";
	      else if (this_v == visibility_protected)
		errstr = "enum `%s' is a protected value of class `%s'";
	    }
	  else
	    {
	      if (this_v == visibility_private)
		errstr = "member `%s' is a private member of class `%s'";
	      else if (this_v == visibility_protected)
		errstr = "member `%s' is a protected member of class `%s'";
	    }
	}

      if (entry)
	{
	  if (errstr)
	    {
	      /* This depends on behavior of lookup_field_1!  */
	      tree error_string = my_build_string (errstr);
	      TREE_TYPE (entry) = error_string;
	    }
	  else
	    {
	      /* Let entry know there is no problem with this access.  */
	      TREE_TYPE (entry) = NULL_TREE;

	      /* And since everything is ok, bear the
		 cost of generating correct code.  */
	      if (DECL_OFFSET (TYPE_NAME (basetype)) != 0)
		{
		  rval = my_copy_node (rval);
		  DECL_FIELD_CONTEXT (rval) = basetype;
		}
	    }
	  TREE_VALUE (entry) = rval;
	}
      else if (DECL_OFFSET (TYPE_NAME (basetype)) != 0
	       && ! (errstr && protect))
	{
	  rval = my_copy_node (rval);
	  DECL_FIELD_CONTEXT (rval) = build_classtype_variant (basetype, integer_zero_node, 1);
	}

      if (errstr && protect)
	{
	  error (errstr, IDENTIFIER_POINTER (name), TYPE_NAME_STRING (basetype));
	  return error_mark_node;
	}
      return rval;
    }

  type = CLASSTYPE_MAIN_VARIANT (basetype);

  search_stack = push_search_level (search_stack, &search_obstack);
  TREE_VIA_PUBLIC (basetypes) = 1;

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	{
	  if (CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) == 0)
	    {
	      tree btypes;

	      CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) = 1;
	      btypes = tree_cons (NULL_TREE, CLASSTYPE_BASECLASS (type, i),
				  basetypes);
	      TREE_VIA_PUBLIC (btypes) = CLASSTYPE_VIA_PUBLIC (type, i);
	      TREE_VIA_VIRTUAL (btypes) = CLASSTYPE_VIA_VIRTUAL (type, i);
	      obstack_grow (&search_obstack, &btypes, sizeof (tree *));
	      tail += 1;
	      if (tail >= search_stack->limit)
		abort ();
	    }
	}

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	break;

      basetypes = search_stack->first[head++];
      type = TREE_VALUE (basetypes);

      /* See if we can find NAME in TYPE.  If RVAL is nonzero,
	 and we do find NAME in TYPE, verify that such a second
	 sighting is in fact legal.  */

      if (rval)
	{
	  /* Just another way of finding the same member.  */
	  if (DECL_FIELD_CONTEXT (rval) == type)
	    {
	      enum visibility_type new_v
		= compute_visibility (basetypes, rval);
	      if (this_v != new_v)
		errstr = "conflicting visibilities to member `%s'";
	    }
	  /* Same baseclass, different places in the lattice.  */
	  else if (CLASSTYPE_MAIN_VARIANT (DECL_FIELD_CONTEXT (rval))
		   == CLASSTYPE_MAIN_VARIANT (type))
	    errstr = "member `%s' belongs to distinct base classes `%s'";
	  else
	    {
	      tree nval = lookup_field_1 (type, name);

	      if (nval && get_base_type (type, DECL_FIELD_CONTEXT (rval), 0) == 0)
		{
		  /* We found it in other than a baseclass of RVAL's.  */
		  errstr = "request for member `%s' is ambiguous";
		}
	    }
	  if (errstr && entry)
	    {
	      tree error_string = my_build_string (errstr);
	      TREE_TYPE (entry) = error_string;
	    }
	  if (errstr && protect)
	    break;
	}
      else
	{
	  rval = lookup_field_1 (type, name);
	  if (rval)
	    {
	      if (DECL_OFFSET (TYPE_NAME (type)) != 0
		  || TREE_VIA_VIRTUAL (basetypes))
		{
		  if (TREE_VIA_VIRTUAL (basetypes))
		    type = build_classtype_variant (type, integer_zero_node, 1);
		  rval = my_copy_node (rval);
		  DECL_FIELD_CONTEXT (rval) = build_classtype_variant (type, integer_zero_node, 1);
		}

	      if (entry || protect)
		this_v = compute_visibility (basetypes, rval);
	      if (entry)
		TREE_VALUE (entry) = rval;

	      /* These may look ambiguous, but they really are not.  */
	      if (vbase_name_p)
		break;
	    }
	}
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

    /* If this FIELD_DECL defines its own visibility, deal with that.  */
    if (rval && errstr == 0
	&& DECL_VISIBILITY (rval)
	&& (protect || entry))
      {
	while (tp < search_tail)
	  {
	    /* If is possible for one of the derived types on the
	       path to have defined special visibility for this
	       field.  Look for such declarations and report an
	       error if a conflict is found.  */
	    enum visibility_type new_v;

	    if (this_v != visibility_default)
	      new_v = compute_visibility (*tp, rval);
	    if (this_v != visibility_default && new_v != this_v)
	      {
		errstr = "conflicting visibilities to member `%s'";
		this_v = visibility_default;
	      }
	    own_visibility = new_v;
	    CLASSTYPE_MARKED2 (TREE_VALUE (*tp++)) = 0;
	  }
      }
    else
      {
	while (tp < search_tail)
	  {
	    CLASSTYPE_MARKED2 (TREE_VALUE (*tp++)) = 0;
	  }
      }
  }
  search_stack = pop_search_level (search_stack, &search_obstack);

  if (errstr == 0)
    {
      if (own_visibility == visibility_private)
	errstr = "member `%s' declared private";
      else if (own_visibility == visibility_protected)
	errstr = "member `%s' declared protected";
      else if (this_v == visibility_private)
	errstr = TREE_PRIVATE (rval) ? "member `%s' is private" : "member `%s' is from private base class";
      else if (this_v == visibility_protected)
	errstr = "member `%s' is protected";
    }

  if (entry)
    {
      if (errstr)
	{
	  tree error_string = my_build_string (errstr);
	  /* Save error message with entry.  */
	  TREE_TYPE (entry) = error_string;
	}
      else
	{
	  /* Mark entry as having no error string.  */
	  TREE_TYPE (entry) = NULL_TREE;
	}
    }

  if (errstr && protect)
    {
      error (errstr, IDENTIFIER_POINTER (name), TYPE_NAME_STRING (type));
      rval = error_mark_node;
    }
  return rval;
}

/* TYPE is a class type. Return the list of fields with name
   NAME, or NULL_TREE is no such field exists.  */
static tree
lookup_fnfields_1 (type, name)
     tree type, name;
{
  register tree field = CLASSTYPE_FN_FIELDS (type);

  n_calls_lookup_fnfields_1++;
  while (field)
    {
      n_outer_fields_searched++;	
      if (TREE_PURPOSE (field) == name)
	break;
      field = TREE_CHAIN (field);
    }
  return field;
}

/* Given a list of member functions FIELDS (which are implicitly
   named TREE_PURPOSE (FIELDS), and come from base type
   DECL_FIELD_CONTEXT (TREE_VALUE (FIELDS))), attempt to find the
   actual method which can accept (using conversions) PARMS.
   The types of PARMS are already computed in PARMTYPES.  */
tree
lookup_fnfield (fields, parms, parmtypes)
     tree fields, parms, parmtypes;
{
  abort ();
}

/* Starting from BASETYPE, return a TREE_BASELINK-like object
   which gives the following information (in a list):

   TREE_TYPE: list of basetypes needed to get to...
   TREE_VALUE: list of all functions in of given type
   which have name NAME.

   No visibility information is computed by this function,
   other then to adorn the list of basetypes with
   TREE_VIA_PUBLIC.

   If FIND_AMBIGUOUS is non-zero, then if we find two ways to get
   to the same member function, both those ways are found,
   and the caller must know what to do about this.  */
tree
lookup_fnfields (basetypes, name, find_ambiguous)
     tree basetypes, name;
     int find_ambiguous;
{
  int head = 0, tail = 0;
  tree type, rval, rvals = NULL_TREE;
  tree basetype;
  tree entry;

  /* For now, don't try this.  */
  int protect = find_ambiguous;

  /* Things for memoization.  */
  char *errstr = 0;

  /* Set this to nonzero if we don't know how to compute
     accurate error messages for visibility.  */
  int memoization_too_hard = 0;
  int index = MEMOIZED_HASH_FN (name);

  basetype = TREE_VALUE (basetypes);

  if (CLASSTYPE_MTABLE_ENTRY (basetype))
    {
      tree tem = MEMOIZED_FNFIELDS (CLASSTYPE_MTABLE_ENTRY (basetype), index);

      while (tem && TREE_PURPOSE (tem) != name)
	{
	  memoized_fields_searched[1]++;
	  tem = TREE_CHAIN (tem);
	}
      if (tem)
	{
	  if (protect && TREE_TYPE (tem))
	    {
	      error_with_aggr_type (DECL_FIELD_CONTEXT (TREE_VALUE (tem)),
				    TREE_STRING_POINTER (TREE_TYPE (tem)),
				    IDENTIFIER_POINTER (name));
	      return error_mark_node;
	    }
	  if (TREE_VALUE (tem) == NULL_TREE)
	    {
	      memoized_fast_rejects[1] += 1;
	      return NULL_TREE;
	    }
	  else
	    {
	      /* Want to return this, but we must make sure
		 that visibility information is consistent.  */
	      tree baselink = TREE_VALUE (tem);
	      tree memoized_basetypes = TREE_PURPOSE (baselink);
	      tree these_basetypes = basetypes;
	      while (memoized_basetypes && these_basetypes)
		{
		  memoized_fields_searched[1]++;
		  if (TREE_VALUE (memoized_basetypes) != TREE_VALUE (these_basetypes))
		    break;
		  memoized_basetypes = TREE_CHAIN (memoized_basetypes);
		  these_basetypes = TREE_CHAIN (these_basetypes);
		}
	      if (memoized_basetypes == these_basetypes)
		{
		  memoized_fast_finds[1] += 1;
		  return TREE_VALUE (tem);
		}
	      /* else, we must re-find this field by hand.  */
	      baselink = tree_cons (basetypes, TREE_VALUE (baselink), TREE_CHAIN (baselink));
	      return baselink;
	    }
	}
    }

  n_calls_lookup_fnfields++;
  if (flag_memoize_lookups && ! global_bindings_p ())
    entry = make_memoized_table_entry (basetype, name, 1);
  else
    entry = 0;

  rval = lookup_fnfields_1 (basetype, name);

  if (rval)
    {
      rvals = my_tree_cons (basetypes, rval, NULL_TREE);
      TREE_BASELINK (rvals) = TREE_BASELINK (rval);

      if (entry)
	{
	  TREE_VALUE (entry) = rvals;
	  TREE_TYPE (entry) = NULL_TREE;
	}

      if (errstr && protect)
	{
	  error_with_aggr_type (basetype, errstr, IDENTIFIER_POINTER (name));
	  return error_mark_node;
	}
      return rvals;
    }

  type = CLASSTYPE_MAIN_VARIANT (basetype);

  search_stack = push_search_level (search_stack, &search_obstack);
  TREE_VIA_PUBLIC (basetypes) = 1;

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	{
	  if (CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) == 0)
	    {
	      tree btypes;

	      CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) = 1;
	      btypes = my_tree_cons (NULL_TREE, CLASSTYPE_BASECLASS (type, i),
				     basetypes);
	      TREE_VIA_PUBLIC (btypes) = CLASSTYPE_VIA_PUBLIC (type, i);
	      TREE_VIA_VIRTUAL (btypes) = CLASSTYPE_VIA_VIRTUAL (type, i);
	      obstack_grow (&search_obstack, &btypes, sizeof (tree *));
	      tail += 1;
	      if (tail >= search_stack->limit)
		abort ();
	    }
	}

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	break;

      basetypes = search_stack->first[head++];
      type = TREE_VALUE (basetypes);

      /* See if we can find NAME in TYPE.  If RVAL is nonzero,
	 and we do find NAME in TYPE, verify that such a second
	 sighting is in fact legal.  */

      if (rval)
	{
	  tree context = DECL_FIELD_CONTEXT (TREE_VALUE (rval));
	  /* Just another way of finding the same member.  */
	  if (context == type)
	    ;
	  /* Same baseclass, maybe different places in the lattice.  */
	  else if (CLASSTYPE_MAIN_VARIANT (context) == CLASSTYPE_MAIN_VARIANT (type))
	    {
	      if (TREE_VIA_VIRTUAL (TREE_PURPOSE (rvals)))
		if (TREE_VIA_VIRTUAL (basetypes))
		  ;
		else
		  errstr = "member `%s' belongs to virtual and non-virtual baseclasses `%s'";
	      else if (TREE_VIA_VIRTUAL (basetypes))
		errstr = "member `%s' belongs to virtual and non-virtual baseclasses `%s'";
	      else
		errstr = "member `%s' belongs to MI-distinct base classes `%s'";
	    }
	  else
	    {
	      tree nval = lookup_fnfields_1 (type, name);

	      if (nval && get_base_type (type, context, 0) == 0)
		{
		  /* We found it in other than a baseclass of RVAL's.  */
		  rvals = my_tree_cons (basetypes, nval, rvals);
		  TREE_BASELINK (rvals) = TREE_BASELINK (nval);
		}
	    }
	  if (errstr && entry)
	    {
	      tree error_string = my_build_string (errstr);
	      TREE_TYPE (entry) = error_string;
	    }
	  if (errstr && find_ambiguous)
	    {
	      rvals = error_mark_node;
	      break;
	    }
	}
      else
	{
	  rval = lookup_fnfields_1 (type, name);
	  if (rval)
	    {
	      rvals = my_tree_cons (basetypes, rval, NULL_TREE);
	      TREE_BASELINK (rvals) = TREE_BASELINK (rval);
	      if (entry)
		TREE_VALUE (entry) = rvals;
	    }
	}
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

    while (tp < search_tail)
      CLASSTYPE_MARKED2 (TREE_VALUE (*tp++)) = 0;
  }
  search_stack = pop_search_level (search_stack, &search_obstack);

  if (entry)
    {
      if (errstr)
	{
	  tree error_string = my_build_string (errstr);
	  /* Save error message with entry.  */
	  TREE_TYPE (entry) = error_string;
	}
      else
	{
	  /* Mark entry as having no error string.  */
	  TREE_TYPE (entry) = NULL_TREE;
	}
    }

  if (errstr && protect)
    {
      error_with_aggr_type (type, errstr, IDENTIFIER_POINTER (name));
      rvals = error_mark_node;
    }

  return rvals;
}

/* BREADTH-FIRST SEARCH ROUTINES.  */
/* Search a multiple inheritance hierarchy by breadth-first search.

   TYPE is an aggregate type, possibly in a multiple-inheritance hierarchy.
   TESTFN is a function, which, if true, means that our condition has been met,
   and its return value should be returned.
   QFN, if non-NULL, is a predicate dictating whether the type should
   even be queued.  */

int
breadth_first_search (type, testfn, qfn)
     tree type;
     int (*testfn)();
     int (*qfn)();
{
  int head = 0, tail = 0;
  int rval = 0;

  search_stack = push_search_level (search_stack, &search_obstack);

  while (1)
    {
      int n_baselinks = CLASSTYPE_N_BASECLASSES (type);
      int i;

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	if (CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (type, i)) == 0
	    && (qfn == 0 || (*qfn) (type, i)))
	  {
	    CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (type, i)) = 1;
	    obstack_grow (&search_obstack, &type, sizeof (tree*));
	    obstack_grow (&search_obstack, &i, sizeof (int*));
	    tail += 2;
	    if (tail >= search_stack->limit)
	      abort ();
	  }

      /* Process head of queue, if one exists.  */
      if (head >= tail)
	{
	  rval = 0;
	  break;
	}

      type = search_stack->first[head++];
      i = (int)search_stack->first[head++];
      if (rval = (*testfn) (type, i))
	break;
      type = CLASSTYPE_THIS_BASECLASS (type, i);
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;
    while (tp < search_tail)
      {
	tree type = *tp++;
	int i = (int)(*tp++);
	CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (type, i)) = 0;
      }
  }

  search_stack = pop_search_level (search_stack, &search_obstack);
  return rval;
}

/* Functions to use in breadth first searches.  */
typedef tree (*pft)();
typedef int (*pfi)();

int tree_needs_constructor_p (type, i)
     tree type;
{
  tree basetype = i == 0 ? type : CLASSTYPE_THIS_BASECLASS (type, i);
  return TYPE_NEEDS_CONSTRUCTOR (basetype);
}

static tree declarator;

static tree get_virtuals_named_this (type, i)
     tree type;
     int i;
{
  tree basetype = i == 0? type : CLASSTYPE_THIS_BASECLASS (type, i);
  tree fields = lookup_fnfields (CLASSTYPE_AS_LIST (basetype), declarator, 0);
  if (fields == 0 || fields == error_mark_node)
    return 0;
  /* Get to the function decls, and return the first virtual function
     with this name, if there is one.  */
  fields = TREE_VALUE (TREE_VALUE (fields));
  while (fields && ! DECL_VIRTUAL_P (fields))
    fields = TREE_CHAIN (fields);
  return fields;
}

pft get_virtual_named (decl)
     tree decl;
{
  declarator = decl;
  return get_virtuals_named_this;
}

static tree get_virtual_destructor (type, i)
     tree type;
     int i;
{
  type = i == 0 ? type : CLASSTYPE_THIS_BASECLASS (type, i);
  if (TYPE_HAS_DESTRUCTOR (type)
      && DECL_VIRTUAL_P (TREE_VALUE (CLASSTYPE_FN_FIELDS (type))))
    return TREE_VALUE (CLASSTYPE_FN_FIELDS (type));
  return 0;
}

int tree_has_any_destructor_p (type, i)
     tree type;
     int i;
{
  if (i == 0)
    return TYPE_NEEDS_DESTRUCTOR (type);
  return TYPE_NEEDS_DESTRUCTOR (CLASSTYPE_THIS_BASECLASS (type, i));
}

/* Given a class type TYPE, and a function decl FNDECL,
   look for the first function the TYPE's heirarchy which
   FNDECL could match as a virtual function.

   DTORP is nonzero if we are looking for a destructor.  Destructors
   need special treatment because they do not match by name.  */
tree
get_first_matching_virtual (type, fndecl, dtorp)
     tree type, fndecl;
     int dtorp;
{
  tree tmp = NULL_TREE;

  /* Breadth first search routines start searching basetypes
     of TYPE, so we must perform first ply of search here.  */
  if (dtorp)
    {
      if (tree_has_any_destructor_p (type, 0))
	tmp = get_virtual_destructor (type, 0);

      if (tmp)
	return tmp;

      tmp = (tree) breadth_first_search (type,
					 (pfi) get_virtual_destructor,
					 tree_has_any_destructor_p);
      return tmp;
    }
  else
    {
      tree drettype, dtypes, btypes;
      int matched;

      declarator = DECL_ORIGINAL_NAME (fndecl);
      tmp = get_virtuals_named_this (type, 0);

      if (tmp == 0)
	tmp = (tree) breadth_first_search (type, (pfi) get_virtuals_named_this, 0);

      if (tmp == 0)
	return 0;

      drettype = TREE_TYPE (TREE_TYPE (fndecl));
      dtypes = TYPE_ARG_TYPES (TREE_TYPE (fndecl));
      while (tmp)
	{
	  btypes = TYPE_ARG_TYPES (TREE_TYPE (tmp));
	  if (compparms (TREE_CHAIN (btypes), TREE_CHAIN (dtypes), 1))
	    {
	      if (! comptypes (TREE_TYPE (TREE_TYPE (tmp)), drettype, 1))
		error_with_decl (fndecl, "conflicting return type specified for virtual function `%s'");
	      break;
	    }
	  tmp = TREE_CHAIN (tmp);
	}
      if (tmp == NULL_TREE && warn_overloaded_virtual)
	error_with_decl (fndecl, "conficting specification deriving virtual function `%s'");

      return tmp;
    }
}

/* For the type TYPE, return a list of member functions available from
   base classes with name NAME.  The TREE_VALUE of the list is a chain of
   member functions with name NAME.  The TREE_PURPOSE of the list a
   basetype, or a list of base types (in reverse order) which were
   traversed to reach the chain of member functions.  If we reach a base
   type which provides a member function of name NAME, and which has at
   most one base type itself, then we can terminate the search.  */

tree
get_baselinks (type, name)
     tree type, name;
{
  tree xtype = type;
  int head = 0, tail = 0;
  int is_private = 0;
  tree rval = 0, nval = 0;
  tree basetypes = CLASSTYPE_AS_LIST (type);

  search_stack = push_search_level (search_stack, &search_obstack);

  while (1)
    {
      int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

      /* Process and/or queue base types.  */
      for (i = 1; i <= n_baselinks; i++)
	if (CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (type, i)) == 0)
	  {
	    tree btypes;

	    btypes = tree_cons (NULL_TREE, CLASSTYPE_THIS_BASECLASS (type, i),
				basetypes);
	    TREE_VIA_PUBLIC (btypes) = CLASSTYPE_VIA_PUBLIC (type, i);
	    TREE_VIA_VIRTUAL (btypes) = CLASSTYPE_VIA_VIRTUAL (type, i);
	    CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (type, i)) = 1;
	    obstack_grow (&search_obstack, &btypes, sizeof (tree *));

	    tail += 1;
	    if (tail >= search_stack->limit)
	      abort ();
	  }

    dont_queue:
      /* Process head of queue, if one exists.  */
      if (head >= tail)
	break;

      basetypes = search_stack->first[head++];
      type = TREE_VALUE (basetypes);
      nval = lookup_fnfields_1 (type, name);
      if (nval != NULL_TREE)
	{
	  rval = tree_cons (basetypes, nval, rval);
	  if (CLASSTYPE_N_BASECLASSES (type) <= 1)
	    goto dont_queue;
	}
    }
  {
    tree *tp = search_stack->first;
    tree *search_tail = tp + tail;

    while (tp < search_tail)
      {
	CLASSTYPE_MARKED (TREE_VALUE (*tp++)) = 0;
      }
  }
  search_stack = pop_search_level (search_stack, &search_obstack);
  return rval;
}

/* DEPTH-FIRST SEARCH ROUTINES.  */

/* Assign unique numbers to _CLASSTYPE members of the lattice
   specified by TYPE.  The root nodes are marked first; the nodes
   are marked depth-fisrt, left-right.  */

static int cid;

/* Matrix implementing a relation from CLASSTYPE X CLASSTYPE => INT.
   Relation yields 1 if C1 <= C2, 0 otherwise.  */
typedef char mi_boolean;
static mi_boolean *mi_matrix;

/* Type for which this matrix is defined.  */
static tree mi_type;

/* Size of the matrix for indexing purposes.  */
static int mi_size;

/* Return nonzero if class C2 derives from class C1.  */
#define DERIVES_FROM(C1, C2)	\
  ((mi_matrix+mi_size*(CLASSTYPE_CID (C1)-1))[CLASSTYPE_CID (C2)-1])
#define DERIVES_FROM_STAR(C)	\
  (mi_matrix+(CLASSTYPE_CID (C)-1))

/* The main function which implements depth first search.  */
static void
dfs_walk (type, fn, qfn)
     tree type;
     void (*fn)();
     int (*qfn)();
{
  int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

  for (i = 1; i <= n_baselinks; i++)
    if ((*qfn)(CLASSTYPE_BASECLASS (type, i)))
      {
	dfs_walk (CLASSTYPE_BASECLASS (type, i), fn, qfn);
      }

  fn (type);
}

/* Predicate functions which serve for dfs_walk.  */
static int numberedp (type) tree type;
{ return CLASSTYPE_CID (type); }
static int unnumberedp (type) tree type;
{ return CLASSTYPE_CID (type) == 0; }

static int markedp (type) tree type;
{ return CLASSTYPE_MARKED (type); }
static int bfs_markedp (type, i) tree type; int i;
{ return CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (type, i)); }
static int unmarkedp (type) tree type;
{ return CLASSTYPE_MARKED (type) == 0; }
static int bfs_unmarkedp (type, i) tree type; int i;
{ return CLASSTYPE_MARKED (CLASSTYPE_THIS_BASECLASS (type, i)) == 0; }
static int marked2p (type) tree type;
{ return CLASSTYPE_MARKED2 (type); }
static int bfs_marked2p (type, i) tree type; int i;
{ return CLASSTYPE_MARKED2 (CLASSTYPE_THIS_BASECLASS (type, i)); }
static int unmarked2p (type) tree type;
{ return CLASSTYPE_MARKED2 (type) == 0; }
static int bfs_unmarked2p (type, i) tree type; int i;
{ return CLASSTYPE_MARKED2 (CLASSTYPE_THIS_BASECLASS (type, i)) == 0; }
static int marked3p (type) tree type;
{ return CLASSTYPE_MARKED3 (type); }
static int bfs_marked3p (type, i) tree type; int i;
{ return CLASSTYPE_MARKED3 (CLASSTYPE_THIS_BASECLASS (type, i)); }
static int unmarked3p (type) tree type;
{ return CLASSTYPE_MARKED3 (type) == 0; }
static int bfs_unmarked3p (type, i) tree type; int i;
{ return CLASSTYPE_MARKED3 (CLASSTYPE_THIS_BASECLASS (type, i)) == 0; }
static int marked4p (type) tree type;
{ return CLASSTYPE_MARKED4 (type); }
static int bfs_marked4p (type, i) tree type; int i;
{ return CLASSTYPE_MARKED4 (CLASSTYPE_THIS_BASECLASS (type, i)); }
static int unmarked4p (type) tree type;
{ return CLASSTYPE_MARKED4 (type) == 0; }
static int bfs_unmarked4p (type, i) tree type; int i;
{ return CLASSTYPE_MARKED4 (CLASSTYPE_THIS_BASECLASS (type, i)) == 0; }

static int dfs_search_slot_nonempty_p (type) tree type;
{ return CLASSTYPE_SEARCH_SLOT (type) != 0; }


/* The worker functions for `dfs_walk'.  These do not need to
   test anything (vis a vis marking) if they are paired with
   a predicate function (above).  */

/* Assign each type within the lattice a number which is unique
   in the lattice.  The first number assigned is 1.  */

static void
dfs_number (type)
     tree type;
{
  CLASSTYPE_CID (type) = ++cid;
}

static void
dfs_unnumber (type)
     tree type;
{
  CLASSTYPE_CID (type) = 0;
}

static void
dfs_mark (type) tree type;
{ CLASSTYPE_MARKED (type) = 1; }

static void
dfs_unmark (type) tree type;
{ CLASSTYPE_MARKED (type) = 0; }

static void
dfs_mark2 (type) tree type;
{ CLASSTYPE_MARKED2 (type) = 1; }

static void
dfs_unmark2 (type) tree type;
{ CLASSTYPE_MARKED2 (type) = 0; }

static void
dfs_mark3 (type) tree type;
{ CLASSTYPE_MARKED3 (type) = 1; }

static void
dfs_unmark3 (type) tree type;
{ CLASSTYPE_MARKED3 (type) = 0; }

static void
dfs_mark4 (type) tree type;
{ CLASSTYPE_MARKED4 (type) = 1; }

static void
dfs_unmark4 (type) tree type;
{ CLASSTYPE_MARKED4 (type) = 0; }

static void
dfs_unmark12 (type) tree type;
{ CLASSTYPE_MARKED (type) = 0;
  CLASSTYPE_MARKED2 (type) = 0; }

static void
dfs_unmark34 (type) tree type;
{ CLASSTYPE_MARKED3 (type) = 0;
  CLASSTYPE_MARKED4 (type) = 0; }

static tree
dfs_clear_search_slot (type) tree type;
{ CLASSTYPE_SEARCH_SLOT (type) = 0; }

static tree vbase_types;
static tree vbase_decl, vbase_decl_ptr, vbase_char_star;
static tree vbase_init_result;

static void
dfs_find_vbases (type)
     tree type;
{
  int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);

  for (i = n_baselinks; i > 0; i--)
    if (CLASSTYPE_VIA_VIRTUAL (type, i)
	&& CLASSTYPE_SEARCH_SLOT (CLASSTYPE_BASECLASS (type, i)) == 0)
      {
	tree vbase = CLASSTYPE_BASECLASS (type, i);
	tree offset_info = value_member (vbase, vbase_types);

	CLASSTYPE_SEARCH_SLOT (vbase)
	  = build (PLUS_EXPR, TYPE_POINTER_TO (vbase),
		   vbase_char_star, TREE_PURPOSE (offset_info));
      }
  CLASSTYPE_MARKED3 (type) = 1;
  CLASSTYPE_MARKED4 (type) = 1;
}

static void
dfs_init_vbase_pointers (type)
     tree type;
{
  tree fields = TYPE_FIELDS (type);
  tree this_vbase_ptr = convert_to_nonzero_pointer (TYPE_POINTER_TO (type), vbase_decl_ptr);
  while (fields && DECL_NAME (fields)
	 && VBASE_NAME_P (DECL_NAME (fields)))
    {
      tree ref = build (COMPONENT_REF, TREE_TYPE (fields),
			build_indirect_ref (this_vbase_ptr, 0), fields);
      tree init = CLASSTYPE_SEARCH_SLOT (TREE_TYPE (TREE_TYPE (fields)));
      vbase_init_result = tree_cons (NULL_TREE, build_modify_expr (ref, NOP_EXPR, init),
				     vbase_init_result);
      fields = TREE_CHAIN (fields);
    }
  CLASSTYPE_MARKED3 (type) = 0;
}

/* Sometimes this needs to clear both 3 and 4.  Other times,
   just 4, but optimizer should make both equivalent
   (though it does not currently).  */
static void
dfs_clear_vbase_slots (type)
     tree type;
{
  CLASSTYPE_SEARCH_SLOT (type) = 0;
  CLASSTYPE_MARKED3 (type) = 0;
  CLASSTYPE_MARKED4 (type) = 0;
}

tree

init_vbase_pointers (type, decl_ptr)
     tree type;
     tree decl_ptr;
{
  if (TYPE_USES_VIRTUAL_BASECLASSES (type))
    {
      vbase_types = CLASSTYPE_VBASECLASSES (type);
      vbase_decl_ptr = decl_ptr;
      vbase_decl = build_indirect_ref (decl_ptr, 0);
      vbase_char_star = build (NOP_EXPR, ptr_type_node, decl_ptr);
      vbase_init_result = NULL_TREE;
#ifdef sparc
      expand_asm_operands (build_string (32, "! start of vbase initialization"), 0, 0, 0, 0, 0, 0);
#endif
      dfs_walk (type, dfs_find_vbases, unmarked3p);
      dfs_walk (type, dfs_init_vbase_pointers, marked3p);
      dfs_walk (type, dfs_clear_vbase_slots, marked4p);

      return vbase_init_result;
    }
  return 0;
}

tree
build_vbase_vtables_init (for_type, type, true_exp, decl_ptr)
     tree for_type, type;
     tree true_exp, decl_ptr;
{
  if (TYPE_USES_VIRTUAL_BASECLASSES (type))
    {
      tree vtable_init_result = NULL_TREE;
      tree vbases = CLASSTYPE_VBASECLASSES (type);

      vbase_types = CLASSTYPE_VBASECLASSES (for_type);
      vbase_decl_ptr = true_exp ? build_unary_op (ADDR_EXPR, true_exp, 0) : decl_ptr;
      vbase_decl = true_exp ? true_exp : build_indirect_ref (decl_ptr, 0);
      vbase_char_star = build (NOP_EXPR, ptr_type_node, vbase_decl_ptr);
#ifdef sparc
      expand_asm_operands (build_string (32, "! start of vtable initialization"), 0, 0, 0, 0, 0, 0);
#endif
      /* This is an object of type IN_TYPE,  */
      dfs_walk (CLASSTYPE_MAIN_VARIANT (for_type), dfs_find_vbases, unmarked4p);
      /* Initialized with vtables of type TYPE.  */

      while (vbases)
	{
	  tree basetype = get_base_type (TREE_VALUE (vbases), type, 0);
	  /* This time through, not every class's vtable
	     is going to be initialized.  That is, we only initialize
	     the "last" vtable pointer.  */
	  if (basetype
	      && TYPE_VIRTUAL_P (basetype)
	      && CLASSTYPE_THIS_VARIANT (basetype) == TREE_VALUE (vbases)
	      && (basetype = TREE_VALUE (vbases), 1)
	      && CLASSTYPE_SEARCH_SLOT (basetype))
	    {
	      tree ref = build_vfield_ref (build_indirect_ref (CLASSTYPE_SEARCH_SLOT (basetype)),
					   basetype);
	      tree vtbl = CLASSTYPE_VTABLE (get_base_type (basetype, for_type, 0));
	      tree init = build_unary_op (ADDR_EXPR, vtbl, 0);
	      TREE_USED (vtbl) = 1;
	      vtable_init_result = tree_cons (NULL_TREE, build_modify_expr (ref, NOP_EXPR, init),
					      vtable_init_result);
	    }
	  vbases = TREE_CHAIN (vbases);
	}

      dfs_walk (type, dfs_clear_vbase_slots, marked4p);
      if (vtable_init_result)
	return build_compound_expr (vtable_init_result);
    }
  return error_mark_node;
}

tree
clear_search_slots (type)
     tree type;
{
  dfs_walk (type, dfs_clear_search_slot, dfs_search_slot_nonempty_p);
}

static void
dfs_get_vbase_types (type)
     tree type;
{
  int i;
  tree these_vbase_types = CLASSTYPE_VBASECLASSES (type);

  if (these_vbase_types)
    {
      while (these_vbase_types)
	{
	  if (! CLASSTYPE_MARKED2 (TREE_VALUE (these_vbase_types)))
	    {
	      vbase_types = tree_cons (NULL_TREE,
				       TREE_VALUE (these_vbase_types),
				       vbase_types);
	      CLASSTYPE_MARKED2 (TREE_VALUE (these_vbase_types)) = 1;
	    }
	  these_vbase_types = TREE_CHAIN (these_vbase_types);
	}
    }
  else for (i = CLASSTYPE_N_BASECLASSES (type); i > 0; i--)
    if (CLASSTYPE_VIA_VIRTUAL (type, i)
	&& ! CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)))
      {
	vbase_types = tree_cons (NULL_TREE, CLASSTYPE_BASECLASS (type, i),
				 vbase_types);
	CLASSTYPE_MARKED2 (CLASSTYPE_BASECLASS (type, i)) = 1;
      }
  CLASSTYPE_MARKED (type) = 1;
}

tree
get_vbase_types (type)
     tree type;
{
  vbase_types = NULL_TREE;

  dfs_walk (type, dfs_get_vbase_types, unmarkedp);
  dfs_walk (type, dfs_unmark12, markedp);

  return vbase_types;
}

static void
dfs_record_inheritance (type)
     tree type;
{
  int i, n_baselinks = CLASSTYPE_N_BASECLASSES (type);
  mi_boolean *derived_row = DERIVES_FROM_STAR (type);

  for (i = n_baselinks; i > 0; i--)
    {
      int j;
      tree baseclass = CLASSTYPE_BASECLASS (type, i);
      mi_boolean *base_row = DERIVES_FROM_STAR (baseclass);

      /* Don't search if there's nothing there!  */
      if (CLASSTYPE_N_BASECLASSES (baseclass))
	for (j = mi_size*(CLASSTYPE_CID (baseclass)-1); j >= 0; j -= mi_size)
	  derived_row[j] |= base_row[j];
      DERIVES_FROM (baseclass, type) = 1;
    }

  CLASSTYPE_MARKED (type) = 1;
}

/* Given a _CLASSTYPE node in a multiple inheritance lattice,
   convert the lattice into a simple relation such that,
   given to CIDs, C1 and C2, one can determine if C1 <= C2
   or C2 <= C1 or C1 <> C2.

   Once constructed, we walk the lattice depth fisrt,
   applying various functions to elements as they are encountered.

   We use malloc here, in case we want to randomly free these tables.  */

#define SAVE_MI_MATRIX

void
build_mi_matrix (type)
     tree type;
{
  cid = 0;

#ifdef SAVE_MI_MATRIX
  if (CLASSTYPE_MI_MATRIX (type))
    {
      mi_matrix = CLASSTYPE_MI_MATRIX (type);
      mi_type = type;
      dfs_walk (type, dfs_number, unnumberedp);
      return;
    }
#endif

  mi_size = CLASSTYPE_N_SUPERCLASSES (type);
  mi_matrix = (char *)malloc ((mi_size+1) * (mi_size+1));
  mi_type = type;
  bzero (mi_matrix, mi_size * mi_size);
  dfs_walk (type, dfs_number, unnumberedp);
  dfs_walk (type, dfs_record_inheritance, unmarkedp);
  dfs_walk (type, dfs_unmark, markedp);
}

void
free_mi_matrix ()
{
  dfs_walk (mi_type, dfs_unnumber, numberedp);

#ifdef SAVE_MI_MATRIX
  CLASSTYPE_MI_MATRIX (mi_type) = mi_matrix;
#else
  free (mi_matrix);
  mi_size = 0;
  cid = 0;
#endif
}

/* Local variables for detecting ambiguities of virtual functions
   when two or more classes are joined at a multiple inheritance
   seam.  */
typedef tree mi_ventry[3];
static mi_ventry *mi_vmatrix;
static int *mi_vmax;
static int mi_vrows, mi_vcols;
#define MI_VMATRIX(ROW,COL) ((mi_vmatrix + (ROW)*mi_vcols)[COL])

/* Build a table of virtual functions for a multiple-inheritance
   structure.  Here, there are N base classes, and at most
   M entries per class.

   This function does nothing if N is 0 or 1.  */
void
build_mi_virtuals (rows, cols)
     int rows, cols;
{
  if (rows < 2)
    return;
  mi_vrows = rows;
  mi_vcols = cols;
  mi_vmatrix = (mi_ventry *)malloc ((rows+1) * (cols+1) * sizeof (mi_ventry));
  mi_vmax = (int *)malloc ((rows+1) * sizeof (int));

  bzero (mi_vmax, rows * sizeof (int));

  /* Row indicies start at 1, so adjust this.  */
  mi_vmatrix -= cols + 1;
  mi_vmax -= 1;
}

/* Comparison function for virtual function table entries.  */
static int
rank_mi_virtuals (v1, v2)
     mi_ventry *v1, *v2;
{
  tree p1, p2;

  if (TREE_UID (DECL_ORIGINAL_NAME ((*v1)[0]))
      - TREE_UID (DECL_ORIGINAL_NAME ((*v2)[0])))
    return (TREE_UID (DECL_ORIGINAL_NAME ((*v1)[0]))
	    - TREE_UID (DECL_ORIGINAL_NAME ((*v2)[0])));
  p1 = (*v1)[1];
  p2 = (*v2)[1];

  if (p1 == p2)
    return 0;

  while (p1 && p2)
    {
      if (TREE_UID (TREE_TYPE (TREE_VALUE (p1)))
	  - TREE_UID (TREE_TYPE (TREE_VALUE (p2))))
	return (TREE_UID (TREE_TYPE (TREE_VALUE (p1)))
		- TREE_UID (TREE_TYPE (TREE_VALUE (p2))));
      if (TREE_CHAIN (p1))
	{
	  if (! TREE_CHAIN (p2))
	    return 1;
	  p1 = TREE_CHAIN (p1);
	  p2 = TREE_CHAIN (p2);
	}
      else if (TREE_CHAIN (p2))
	return -1;
      else
	{
	  /* When matches of argument lists occur, pick lowest
	     TREE_UID to keep searching time to a minimum on
	     later passes--like hashing, only different.
	     *MUST BE STABLE*.  */
	  if (TREE_UID ((*v2)[1]) < TREE_UID ((*v1)[1]))
	    (*v1)[1] = (*v2)[1];
	  else
	    (*v2)[1] = (*v1)[1];
	  return 0;
	}
    }
  return 0;
}

/* Install the virtuals functions got from the initializer VIRTUALS to
   the table at index ROW.  */
void
add_mi_virtuals (row, virtuals)
     int row;
     tree virtuals;
{
  int col = 0;

  if (mi_vmatrix == 0)
    return;
  while (virtuals)
    {
      tree decl = TREE_OPERAND (FNADDR_FROM_VTABLE_ENTRY (TREE_VALUE (virtuals)), 0);
      MI_VMATRIX (row, col)[0] = decl;
      MI_VMATRIX (row, col)[1] = TREE_CHAIN (TYPE_ARG_TYPES (TREE_TYPE (decl)));
      MI_VMATRIX (row, col)[2] = TREE_VALUE (virtuals);
      virtuals = TREE_CHAIN (virtuals);
      col += 1;
    }
  mi_vmax[row] = col;

  qsort (mi_vmatrix + row * mi_vcols,
	 col,
	 sizeof (mi_ventry),
	 rank_mi_virtuals);
}

/* If joining two types results in an ambiguity in the virtual
   function table, report such here.  */
void
report_ambiguous_mi_virtuals (rows, type)
     int rows;
     tree type;
{
  int *mi_vmin;
  int row1, col1, row, col;

  if (mi_vmatrix == 0)
    return;

  /* Now virtuals are all sorted, so we merge to find ambiguous cases.  */
  mi_vmin = (int *)alloca ((rows+1) * sizeof (int));
  bzero (mi_vmin, rows * sizeof (int));

  /* adjust.  */
  mi_vmin -= 1;

  for (row1 = 1; row1 < rows; row1++)
    for (col1 = mi_vmax[row1]-1; col1 >= mi_vmin[row1]; col1--)
      {
	tree this_decl = MI_VMATRIX (row1, col1)[0];
	tree these_args = MI_VMATRIX (row1, col1)[1];
	tree this_context = DECL_CONTEXT (this_decl);

	if (this_context != type)
	  this_context = CLASSTYPE_THIS_VARIANT (get_base_type (this_context, type, 0));

	for (row = row1+1; row <= rows; row++)
	  for (col = mi_vmax[row]-1; col >= mi_vmin[row]; col--)
	    {
	      mi_ventry this_entry;

	      if (MI_VMATRIX (row, col)[0] == 0)
		continue;

	      this_entry[0] = this_decl;
	      this_entry[1] = these_args;
	      this_entry[2] = MI_VMATRIX (row1, col1)[2];
	      if (rank_mi_virtuals (&this_entry,
				    &MI_VMATRIX (row, col)) == 0)
		{
		  /* They are equal.  There are four possibilities:
		     
		     (1) Derived class is defining this virtual function.
		     (2) Two paths to the same virtual function in the
		         same base class.
		     (3) A path to a virtual function declared in one base
		         class, and another path to a virtual function in a
			 base class of the base class.
		     (4) Two paths to the same virtual function in different
		         base classes.
		     
		     The first three cases are ok (non-ambiguous).  */

		  tree that_context, tmp;
		  int this_before_that;

		  if (type == this_context)
		    ;
		  else if ((that_context=get_base_type (DECL_CONTEXT (MI_VMATRIX (row, col)[0]), type, 0)) != NULL_TREE
			   ? (that_context=CLASSTYPE_THIS_VARIANT (that_context)) == this_context
			   : 0)
		    ;
		  else if (that_context != NULL_TREE
			   && (! (this_before_that = ! (that_context == (tmp=get_base_type (that_context, this_context, 0), tmp != NULL_TREE ? CLASSTYPE_THIS_VARIANT (tmp) : 0)))
			       || (this_before_that = this_context == (tmp=get_base_type (this_context, that_context, 0), tmp != NULL_TREE ? CLASSTYPE_THIS_VARIANT (tmp) : 0))))
		    {
		      /* In this case, we must rebuild this entry in the virtual
			 function table to point our main vtable entry
			 (from this_context) to the new vtable entry
			 (from that_context).

			 As far as I can tell, this should only happen for
			 virtual base classes.
			 @@ but it happens to all base classes now.

			 I'm not sure this function does anything
			 desirable anymore.  */
#if 0
		      merge_virtual_function (type, this_decl, this_context,
					      that_context, this_before_that,
					      MI_VMATRIX (row, col)[2]);
#endif
		    }
		  else
		    {
		      error_with_decl (MI_VMATRIX (row, col)[0], "ambiguous virtual function `%s'");
		      error_with_decl (this_decl, "ambiguating function `%s' (joined by type `%s')", IDENTIFIER_POINTER (current_class_name));
		    }
		  MI_VMATRIX (row, col)[0] = 0;

		  /* Let zeros propagate.  */
		  if (col == mi_vmax[row]-1)
		    {
		      int i = col;
		      while (i >= mi_vmin[row]
			     && MI_VMATRIX (row, i)[0] == 0)
			i--;
		      mi_vmax[row] = i;
		    }
		  else if (col == mi_vmin[row])
		    {
		      int i = col;
		      while (i < mi_vmax[row]
			     && MI_VMATRIX (row, i)[0] == 0)
			i++;
		      mi_vmin[row] = i;
		    }
		}
	    }
      }
  free (mi_vmatrix + mi_vcols + 1);
  mi_vmatrix = 0;
  free (mi_vmax + 1);
  mi_vmax = 0;
}

/* Subroutines of push_class_decls ().  */

/* Add the instance variables which this class contributed to the
   current class binding contour.  When a redefinition occurs,
   if the redefinition is strictly within a single inheritance path,
   we just overwrite (in the case of a data field) or
   cons (in the case of a member function) the old declaration with
   the new.  If the fields are not within a single inheritance path,
   we must cons them in either case.  */

static void
dfs_pushdecls (type)
     tree type;
{
  tree fields;

  for (fields = TYPE_FIELDS (type); fields; fields = TREE_CHAIN (fields))
    {
      /* Unmark so that if we are in a constructor, and then find that
	 this field was initialized by a base initializer,
	 we can emit an error message.  */
      TREE_USED (fields) = 0;
      if (DECL_ANON_UNION_ELEM (fields))
	{
	  dfs_pushdecls (TREE_TYPE (fields));
	  continue;
	}
      TREE_FIELD_PUBLIC (fields) = 0;
      TREE_FIELD_PROTECTED (fields) = 0;
      TREE_FIELD_PRIVATE (fields) = 0;

      if (DECL_NAME (fields))
	{
	  tree value = IDENTIFIER_CLASS_VALUE (DECL_NAME (fields));
	  if (value)
	    {
	      /* Possible ambiguity.  If its defining type(s)
		 is (are all) derived from us, no problem.  */

	      if (TREE_CODE (value) == FIELD_DECL
		  || TREE_CODE (value) == VAR_DECL
		  || TREE_CODE (value) == CONST_DECL)
		value = DERIVES_FROM (DECL_FIELD_CONTEXT (value), type)
		  ? fields : tree_cons (NULL_TREE, fields,
					build_tree_list (NULL_TREE, value));
	      else
		{
		  /* All children may derive from us, in which case
		     there is no problem.  Otherwise, we have to
		     keep lists around of what the ambiguities might be.  */
		  tree values;
		  int problem = 0;

		  for (values = value; values; values = TREE_CHAIN (values))
		    {
		      tree sub_values = TREE_VALUE (values);
		      if (TREE_CODE (sub_values) == TREE_LIST)
			{
			  for (; sub_values; sub_values = TREE_CHAIN (sub_values))
			    if (! DERIVES_FROM (DECL_FIELD_CONTEXT (TREE_VALUE (sub_values)), type))
			      {
				value = tree_cons (NULL_TREE, TREE_VALUE (values), value);
				problem = 1;
				break;
			      }
			}
		      else
			{
			  if (! DERIVES_FROM (DECL_FIELD_CONTEXT (sub_values), type))
			    {
			      value = tree_cons (NULL_TREE, values, value);
			      problem = 1;
			      break;
			    }
			}
		    }
		  if (! problem) value = fields;
		}

	      /* Mark this as a potentially ambiguous member.  */
	      if (TREE_CODE (value) == TREE_LIST)
		{
		  /* Leaving TREE_TYPE blank is intentional.
		     We cannot use `error_mark_node' (lookup_name)
		     or `unknown_type_node' (all member functions use this).  */
		  TREE_NONLOCAL (value) = 1;
		}

	      IDENTIFIER_CLASS_VALUE (DECL_NAME (fields)) = value;
	    }
	  else IDENTIFIER_CLASS_VALUE (DECL_NAME (fields)) = fields;
	}
    }

  fields = CLASSTYPE_FN_FIELDS (type);

  /* Farm out constructors and destructors.  */

  if (TYPE_HAS_DESTRUCTOR (type) | TYPE_HAS_CONSTRUCTOR (type))
    fields = TREE_CHAIN (fields);

  /* This does not work for multiple inheritance yet.  */
  while (fields)
    {
      /* This will cause lookup_name to return a pointer
	 to the tree_list of possible methods of this name.
	 If the order is a problem, we can nreverse them.  */
      tree tmp;
      tree old = IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields));

      if (old && TREE_CODE (old) == TREE_LIST)
	tmp = tree_cons (TREE_PURPOSE (fields), fields, old);
      else
	{
	  /* Only complain if we shadow something we can access.  */
	  if (old && (DECL_CONTEXT (old) == current_class_type
		      || ! TREE_PRIVATE (old)))
	    /* Should figure out visibility more accurately.  */
	    warning ("shadowing member `%s' with member function",
		     IDENTIFIER_POINTER (TREE_PURPOSE (fields)));
	  tmp = build_tree_list (TREE_PURPOSE (fields), fields);
	}

      TREE_TYPE (tmp) = unknown_type_node;
      TREE_OVERLOADED (tmp) = TREE_OVERLOADED (fields);
      TREE_NONLOCAL (tmp) = 1;
      IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields)) = tmp;

      tmp = TREE_VALUE (fields);
      while (tmp != 0)
	{
	  TREE_FIELD_PUBLIC (tmp) = 0;
	  TREE_FIELD_PROTECTED (tmp) = 0;
	  TREE_FIELD_PRIVATE (tmp) = 0;
	  tmp = TREE_CHAIN (tmp);
	}

      fields = TREE_CHAIN (fields);
    }

  CLASSTYPE_MARKED (type) = 1;
}

/* Consolidate unique (by name) member functions.  */
static void
dfs_compress_decls (type)
     tree type;
{
  tree fields = CLASSTYPE_FN_FIELDS (type);

  /* Farm out constructors and destructors.  */
  if (TYPE_HAS_DESTRUCTOR (type) | TYPE_HAS_CONSTRUCTOR (type))
    fields = TREE_CHAIN (fields);

  for (; fields; fields = TREE_CHAIN (fields))
    {
      tree tmp = IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields));

      /* This was replaced in scope by somebody else.  Just leave it
	 alone.  */
      if (TREE_CODE (tmp) != TREE_LIST)
	continue;

      if (TREE_CHAIN (tmp) == NULL_TREE
	  && TREE_VALUE (TREE_VALUE (tmp))
	  && TREE_CHAIN (TREE_VALUE (TREE_VALUE (tmp))) == NULL_TREE)
	{
	  IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields))
	    = TREE_VALUE (TREE_VALUE (tmp));
	}
    }

  CLASSTYPE_MARKED (type) = 0;
}

/* When entering the scope of a class, we cache all of the
   fields that that class provides within its inheritance
   lattice.  Where ambiguities result, we mark them
   with `error_mark_node' so that if they are encountered
   without explicit qualification, we can emit an error
   message.  */
void
push_class_decls (type)
     tree type;
{
#if 0
  tree tags = TYPE_TAGS (type);

  while (tags)
    {
      tree code_type_node;
      tree tag;

      switch (TREE_CODE (TREE_VALUE (tags)))
	{
	case ENUMERAL_TYPE:
	  code_type_node = enum_type_node;
	  break;
	case RECORD_TYPE:
	  code_type_node = record_type_node;
	  break;
	case CLASS_TYPE:
	  code_type_node = class_type_node;
	  break;
	case UNION_TYPE:
	  code_type_node = union_type_node;
	  break;
	default:
	  assert (0);
	}
      tag = xref_tag (code_type_node, TREE_PURPOSE (tags),
		      CLASSTYPE_BASECLASS (TREE_VALUE (tags), 1));
      pushdecl (build_decl (TYPE_DECL, TREE_PURPOSE (tags), TREE_VALUE (tags)));
    }
#endif

  /* Push class fields into CLASS_VALUE scope, and mark.  */
  dfs_walk (type, dfs_pushdecls, unmarkedp);

  /* Compress fields which have only a single entry
     by a given name, and unmark.  */
  dfs_walk (type, dfs_compress_decls, markedp);
}

static void
dfs_popdecls (type)
     tree type;
{
  tree fields = TYPE_FIELDS (type);

  while (fields)
    {
      if (DECL_ANON_UNION_ELEM (fields))
	{
	  dfs_popdecls (TREE_TYPE (fields));
	}
      else if (DECL_NAME (fields))
	IDENTIFIER_CLASS_VALUE (DECL_NAME (fields)) = NULL_TREE;
      fields = TREE_CHAIN (fields);
    }
  for (fields = CLASSTYPE_FN_FIELDS (type); fields; fields = TREE_CHAIN (fields))
    {
      IDENTIFIER_CLASS_VALUE (TREE_PURPOSE (fields)) = NULL_TREE;
    }

  CLASSTYPE_MARKED (type) = 1;
}

void
pop_class_decls (type)
     tree type;
{
  /* Clear out the IDENTIFIER_CLASS_VALUE which this
     class may have occupied, and mark.  */
  dfs_walk (type, dfs_popdecls, unmarkedp);

  /* Unmark.  */
  dfs_walk (type, dfs_unmark, markedp);
}

/* Given a base type PARENT, and a derived type TYPE, build
   a name which distinguishes exactly the PARENT member of TYPE's type.

   FORMAT is a string which controls how sprintf formats the name
   we have generated.

   For example, given

	class A; class B; class C : A, B;

   it is possible to distinguish "A" from "C's A".  And given

	class L;
	class A : L; class B : L; class C : A, B;

   it is possible to distinguish "L" from "A's L", and also from
   "C's L from A".  */
tree
build_type_pathname (format, parent, type)
     char *format;
     tree parent, type;
{
  extern struct obstack temporary_obstack;
  char *base = obstack_base (&temporary_obstack);
  char *name;
  int i;
  tree id;

  parent = CLASSTYPE_MAIN_VARIANT (parent);
  obstack_grow (&temporary_obstack,
		TYPE_NAME_STRING (type), TYPE_NAME_LENGTH (type));
  obstack_1grow (&temporary_obstack, '$');
  obstack_grow0 (&temporary_obstack,
		 TYPE_NAME_STRING (parent), TYPE_NAME_LENGTH (parent));
  i = obstack_object_size (&temporary_obstack);
  name = obstack_base (&temporary_obstack) + i;
  obstack_blank (&temporary_obstack, strlen (format) + i + 1);
  sprintf (name, format, base);
  id = get_identifier (name);
  obstack_free (&temporary_obstack, base);
  return id;
}

static int
bfs_unmark_finished_struct (type, i)
     tree type;
     int i;
{
  type = i == 0 ? type : CLASSTYPE_THIS_BASECLASS (type, i);
  if (CLASSTYPE_MARKED4 (type))
    {
      tree decl = CLASSTYPE_VTABLE (type);
      if (TYPE_USES_VIRTUAL_BASECLASSES (type))
	fixup_vbase_offsets (type);
      finish_decl (decl, DECL_INITIAL (decl), 0);
    }
  CLASSTYPE_MARKED3 (type) = 0;
  CLASSTYPE_MARKED4 (type) = 0;
  return 0;
}

void
unmark_finished_struct (type)
     tree type;
{
  bfs_unmark_finished_struct (type, 0);
  breadth_first_search (type, bfs_unmark_finished_struct, bfs_marked3p);
}

void
print_search_statistics ()
{
  if (flag_memoize_lookups)
    {
      fprintf (stderr, "%d memoized contexts saved\n",
	       n_contexts_saved);
      fprintf (stderr, "%d local tree nodes made\n", my_tree_node_counter);
      fprintf (stderr, "%d local hash nodes made\n", my_memoized_entry_counter);
      fprintf (stderr, "fields statistics:\n");
      fprintf (stderr, "  memoized finds = %d; rejects = %d; (searches = %d)\n",
	       memoized_fast_finds[0], memoized_fast_rejects[0],
	       memoized_fields_searched[0]);
      fprintf (stderr, "  memoized_adds = %d\n", memoized_adds[0]);
      fprintf (stderr, "fnfields statistics:\n");
      fprintf (stderr, "  memoized finds = %d; rejects = %d; (searches = %d)\n",
	       memoized_fast_finds[1], memoized_fast_rejects[1],
	       memoized_fields_searched[1]);
      fprintf (stderr, "  memoized_adds = %d\n", memoized_adds[1]);
    }
  fprintf (stderr, "%d fields searched in %d[%d] calls to lookup_field[_1]\n",
	   n_fields_searched, n_calls_lookup_field, n_calls_lookup_field_1);
  fprintf (stderr, "%d fnfields searched in %d calls to lookup_fnfields\n",
	   n_outer_fields_searched, n_calls_lookup_fnfields);
  fprintf (stderr, "%d calls to get_base_type\n", n_calls_get_base_type);
}

void init_search_processing ()
{
  obstack_init (&search_obstack);
  obstack_init (&type_obstack);
  obstack_init (&type_obstack_entries);

  /* This gives us room to build our chains of basetypes,
     whether or not we decide to memoize them.  */
  type_stack = push_type_level (0, &type_obstack);
}

tree
get_wrapper (type)
     tree type;
{
  abort ();
}
