/* Routines for building, ordering, and printing the keyword list.
   Copyright (C) 1989 Free Software Foundation, Inc.
   written by Douglas C. Schmidt (schmidt@ics.uci.edu)

This file is part of GNU GPERF.

GNU GPERF is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU GPERF is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GPERF; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <assert.h>
#include <stdio.h>
#include <builtin.h>
#include "options.h"
#include "readline.h"
#include "keylist.h"
#include "hashtable.h"

/* Current release version. */
extern char *version_string;

/* See comments in perfect.cc. */
extern int occurrences[ALPHABET_SIZE]; 

/* Ditto. */
extern int asso_values[ALPHABET_SIZE];

/* Used in function reorder, below. */
static bool determined[ALPHABET_SIZE]; 

/* Default type for generated code. */
static char *Key_List::default_array_type  = "char *"; 

/* Generated function ``in_word_set'' default return type. */
static char *Key_List::default_return_type = "char *";

/* How wide the printed field width must be to contain the maximum hash value. */
static int field_width = 2;

/* Destructor dumps diagnostics during debugging. */

Key_List::~Key_List (void) 
{ 
  if (option[DEBUG])
    {
      report_error ("\nDumping key list information:\ntotal unique keywords = %d"
                             "\ntotal keywords = %d\nmaximum key length = %d.\n", 
                             list_len, total_keys, max_key_len);
      dump ();
      report_error ("End dumping list.\n\n");
    }
}

/* Gathers the input stream into a buffer until one of two things occur:

   1. We read a '%' followed by a '%'
   2. We read a '%' followed by a '}'

   The first symbolizes the beginning of the keyword list proper,
   The second symbolizes the end of the C source code to be generated
   verbatim in the output file.

   I assume that the keys are separated from the optional preceding struct
   declaration by a consecutive % followed by either % or } starting in 
   the first column. The code below uses an expandible buffer to scan off 
   and return a pointer to all the code (if any) appearing before the delimiter. */

char *
Key_List::get_special_input (char delimiter)
{ 
  int size  = 80;
  char *buf = new char[size];
  int c, i;

  for (i = 0; (c = getchar ()) != EOF; i++)
    {
      if (c == '%')
        {
          if ((c = getchar ()) == delimiter)
            {
        
              while ((c = getchar ()) != '\n')
                ; /* discard newline */
              
              if (i == 0)
                return "";
              else
                {
                  buf[delimiter == '%' && buf[i - 2] == ';' ? i - 2 : i - 1] = '\0';
                  return buf;
                }
            }
          else
            ungetc (c, stdin);
        }
      else if (i >= size) /* Yikes, time to grow the buffer! */
        { 
          char *temp = new char[size *= 2];
          int j;
          
          for (j = 0; j < i; j++)
            temp[j] = buf[j];
          
          delete (buf);
          buf = temp;
        }
      buf[i] = c;
    }
  
  return NULL;        /* Problem here. */
}

/* Stores any C text that must be included verbatim into the 
   generated code output. */

char *
Key_List::save_include_src (void)
{
  int c;
  
  if ((c = getchar ()) != '%')
    {
      ungetc (c, stdin);
      return "";
    }
  else if ((c = getchar ()) != '{')
    report_error ("internal error, %c != '{' on line %d in file %s%a", c, __LINE__, __FILE__);
  else 
    return get_special_input ('}');
}

/* Determines from the input file whether the user wants to build a table
   from a user-defined struct, or whether the user is content to simply
   use the default array of keys. */

char *
Key_List::get_array_type (void)
{
  return get_special_input ('%');
}  
  
/* strcspn - find length of initial segment of S consisting entirely
   of characters not from REJECT (borrowed from Henry Spencer's
   ANSI string package). */

static inline int
Key_List::strcspn (char *s, char *reject)
{
  char *scan;
  char *rej_scan;
  int   count = 0;

  for (scan = s; *scan; scan++) 
    {

      for (rej_scan = reject; *rej_scan; rej_scan++)
        if (*scan == *rej_scan)
          return count;

      count++;
    }

  return count;
}

/* Sets up the Return_Type, the Struct_Tag type and the Array_Type
   based upon various user Options. */

void 
Key_List::set_output_types (void)
{
  if (option[TYPE] && !(array_type = get_array_type ()))
    return;                     /* Something's wrong, bug we'll catch it later on.... */
  else if (option[TYPE])        /* Yow, we've got a user-defined type... */
    {    
      int struct_tag_length = strcspn (array_type, "{\n\0");
      
      if (option[POINTER])      /* And it must return a pointer... */
        {    
          return_type = new char[struct_tag_length + 2];
          strncpy (return_type, array_type, struct_tag_length);
          return_type[struct_tag_length] = '\0';
          strcat (return_type, "*");
        }
      
      struct_tag = new char[struct_tag_length + 1];
      strncpy (struct_tag, array_type,  struct_tag_length);
      struct_tag[struct_tag_length] = '\0';
    }  
  else if (option[POINTER])     /* Return a char *. */
    return_type = default_array_type;
}

/* Reads in all keys from standard input and creates a linked list pointed
   to by Head.  This list is then quickly checked for ``links,'' i.e.,
   unhashable elements possessing identical key sets and lengths. */

void 
Key_List::read_keys (void)
{
  Read_Line get_line;
  char     *ptr;
  
  include_src = save_include_src ();
  set_output_types ();

  /* Oops, problem with the input file. */  
  if (! (ptr = get_line ())) 
    report_error ("No words in input file, did you forget to prepend %s"
                           " or use -t accidentally?\n%a", "%%");

  /* Read in all the keywords from the input file. */
  else 
    {                      
      List_Node *temp, *trail;

      head = new List_Node (ptr, strcspn (ptr, ",\n"));
      
      for (temp = head;
           (ptr = get_line ()) && strcmp (ptr, "%%");
           total_keys++, temp = temp->next)
        temp->next = new List_Node (ptr, strcspn (ptr, ",\n"));
      
      /* See if any additional C code is included at end of this file. */
      if (ptr)
        additional_code = TRUE;

      /* If this becomes TRUE we've got a link. */
      bool       link = FALSE;  
      
      /* Hash table this number of times larger than keyword number. */
      const int table_multiple = 5; 

      /* Make large hash table for efficiency. */
      Hash_Table found_link ((list_len = total_keys) * table_multiple); 

      /* Test whether there are any links and also set the maximum length of
        an identifier in the keyword list. */
      
      for (temp = head, trail = NULL; temp; temp = temp->next)
        {
          List_Node *ptr = found_link (temp, option[NOLENGTH]);
          
          /* Check for links.  We deal with these by building an equivalence class
             of all duplicate values (i.e., links) so that only 1 keyword is
             representative of the entire collection.  This *greatly* simplifies
             processing during later stages of the program. */

          if (ptr)              
            {                   
              list_len--;
              trail->next = temp->next;
              temp->link  = ptr->link;
              ptr->link   = temp;
              link        = TRUE;

              /* Complain if user hasn't enabled the duplicate option. */
              if (!option[DUP])
                report_error ("Key link: \"%s\" = \"%s\", with key set \"%s\".\n", temp->key, ptr->key, temp->key_set);
              else if (option[DEBUG])
                report_error ("Key link: \"%s\" = \"%s\", with key set \"%s\".\n", temp->key, ptr->key, temp->key_set);                
            }
          else
            trail = temp;
            
          /* Update minimum and maximum keyword length, if needed. */
          if (temp->length > max_key_len) 
            max_key_len = temp->length;
          else if (temp->length < min_key_len) 
            min_key_len = temp->length;
        }

      /* Exit program if links exists and option[DUP] not set, since we can't continue */
      if (link) 
				{
					if (option[DUP])
						{
							if (!option[SWITCH])
								{
									report_error ("turning on -S option.\n");
									option = SWITCH;
								}
							report_error ("Some input keys have identical hash values, examine output carefully...\n");
						}
					else
						report_error ("Some input keys have identical hash values,\ntry different key positions or use option -D.\n%a");
				}					
			else if (option[DUP])
				{
					/* If no links, clear the DUP option so we can use the length
						 table, if output. */
					option != DUP;
				}

    }
}

/* Recursively merges two sorted lists together to form one sorted list. The
   ordering criteria is by frequency of occurrence of elements in the key set
   or by the hash value.  This is a kludge, but permits nice sharing of
   almost identical code without incurring the overhead of a function
   call comparison. */
  
inline List_Node *
Key_List::merge (List_Node *list1, List_Node *list2)
{
  if (!list1)
    return list2;
  else if (!list2)
    return list1;
  else if (occurrence_sort && list1->occurrence < list2->occurrence
           || hash_sort && list1->hash_value > list2->hash_value)
    {
      list2->next = merge (list2->next, list1);
      return list2;
    }
  else
    {
      list1->next = merge (list1->next, list2);
      return list1;
    }
}

/* Applies the merge sort algorithm to recursively sort the key list by
   frequency of occurrence of elements in the key set. */
  
inline List_Node *
Key_List::merge_sort (List_Node *head)
{ 
  if (!head || !head->next)
    return head;
  else
    {
      List_Node *middle = head;
      List_Node *temp   = head->next->next;
    
      while (temp)
        {
          temp   = temp->next;
          middle = middle->next;
          if (temp)
            temp = temp->next;
        } 
    
      temp         = middle->next;
      middle->next = NULL;
      return merge (merge_sort (head), merge_sort (temp));
    }   
}

/* Returns the frequency of occurrence of elements in the key set. */

static inline int 
Key_List::get_occurrence (List_Node *ptr)
{
  int   value = 0;
  char *temp;

  for (temp = ptr->key_set; *temp; temp++)
    value += occurrences[*temp];
  
  return value;
}

/* Enables the index location of all key set elements that are now 
   determined. */
  
static inline void 
Key_List::set_determined (List_Node *ptr)
{
  char *temp;
  
  for (temp = ptr->key_set; *temp; temp++)
    determined[*temp] = TRUE;
  
}

/* Returns TRUE if PTR's key set is already completely determined. */

static inline bool 
Key_List::already_determined (List_Node *ptr)
{
  bool  is_determined = TRUE;
  char *temp;

  for (temp = ptr->key_set; is_determined && *temp; temp++)
    is_determined = determined[*temp];
  
  return is_determined;
}

/* Reorders the table by first sorting the list so that frequently occuring 
   keys appear first, and then the list is reorded so that keys whose values 
   are already determined will be placed towards the front of the list.  This
   helps prune the search time by handling inevitable collisions early in the
   search process.  See Cichelli's paper from Jan 1980 JACM for details.... */

void 
Key_List::reorder (void)
{
  List_Node *ptr;

  for (ptr = head; ptr; ptr = ptr->next)
    ptr->occurrence = get_occurrence (ptr);
  
  hash_sort       = FALSE;
  occurrence_sort = TRUE;
  
  for (ptr = head = merge_sort (head); ptr->next; ptr = ptr->next)
    {
      set_determined (ptr);
    
      if (already_determined (ptr->next))
        continue;
      else
        {
          List_Node *trail_ptr = ptr->next;
          List_Node *run_ptr   = trail_ptr->next;
      
          for (; run_ptr; run_ptr = trail_ptr->next)
            {
        
              if (already_determined (run_ptr))
                {
                  trail_ptr->next = run_ptr->next;
                  run_ptr->next   = ptr->next;
                  ptr = ptr->next = run_ptr;
                }
              else
                trail_ptr = run_ptr;
            }
        }
    }     
}

/* Determines the maximum and minimum hash values.  One notable feature is 
   Ira Pohl's optimal algorithm to calculate both the maximum and minimum
   items in a list in O(3n/2) time (faster than the O (2n) method). 
   Returns the maximum hash value encountered. */
  
int 
Key_List::print_min_max (void)
{
  int          min_hash_value;
  int          max_hash_value;
  List_Node   *temp;
  
  if (odd (list_len)) /* Pre-process first item, list now has an even length. */
    {              
      min_hash_value  = max_hash_value  = head->hash_value;
      temp            = head->next;
    }
  else /* List is already even length, no problem extra work necessary. */
    {                      
      min_hash_value = MAX_INT;
      max_hash_value = NEG_MAX_INT;
      temp           = head;
    }
  
  for ( ; temp; temp = temp->next) /* Find max and min in optimal o(3n/2) time. */
    { 
      int key_2, key_1 = temp->hash_value;
      temp  = temp->next;
      key_2 = temp->hash_value;
      
      if (key_1 < key_2)
        {
          if (key_1 < min_hash_value)
            min_hash_value = key_1;
          if (key_2 > max_hash_value)
            max_hash_value = key_2;
        }
      else
        {
          if (key_2 < min_hash_value)
            min_hash_value = key_2;
          if (key_1 > max_hash_value)
            max_hash_value = key_1;
        }
  }
  
  printf ("\n#define MIN_WORD_LENGTH %d\n#define MAX_WORD_LENGTH %d"
          "\n#define MIN_HASH_VALUE %d\n#define MAX_HASH_VALUE %d"
          "\n/*\n%5d keywords\n%5d is the maximum key range\n*/\n\n",
          min_key_len, max_key_len, min_hash_value, max_hash_value,
          total_keys, (max_hash_value - min_hash_value + 1));
  return max_hash_value;
}

/* Generates the output using a C switch.  This trades increased search
   time for decreased table space (potentially *much* less space for
   sparse tables). It the user has specified their own struct in the
   keyword file *and* they enable the POINTER option we have extra work to
   do.  The solution here is to maintain a local static array of user
   defined struct's, as with the Print_Lookup_Function.  Then we use for
   switch statements to perform either a strcmp or strncmp, returning 0 if
   the str fails to match, and otherwise returning a pointer to appropriate index
   location in the local static array. */

#ifdef sparc
#include <alloca.h>
#endif

void 
Key_List::print_switch (void)
{
  char *comp_buffer;
	int   pointer_and_type_enabled = option[POINTER] && option[TYPE];

  if (pointer_and_type_enabled)
    {
      comp_buffer = (char *) alloca (strlen ("!strncmp (str, resword->%s, len)" + strlen (option.get_key_name ()) + 1));
      sprintf (comp_buffer, option[COMP]
               ? "!strncmp (str, resword->%s, len)"
               : "!strcmp (str, resword->%s)", option.get_key_name ());
    }
  else
    comp_buffer = option[COMP] ? "!strncmp (str, resword, len)" : "!strcmp (str, resword)";

  printf ("  if (len <= MAX_WORD_LENGTH && len >= MIN_WORD_LENGTH)\n    {\n"
          "      register int key = %s (str, len);\n\n"
          "      if (key <= MAX_HASH_VALUE && key >= MIN_HASH_VALUE)\n        {\n",
					option.get_hash_name ());
  
  /* Output each keyword as part of a switch statement indexed by hash value. */
  
  if (option[POINTER] || option[DUP])
    {
			List_Node *temp;

			printf ("          %s *resword;\n\n          switch (key)\n            {\n",
							pointer_and_type_enabled ? struct_tag : "char");

			for (temp = head; temp; temp = temp->next)
				{
					printf ("            case %*d:\n", field_width, temp->hash_value);

					if (temp->link)
						{
							List_Node *links;

							for (links = temp; links; links = links->link)
								if (pointer_and_type_enabled)
									printf ("              resword = &wordlist[%d];\n              if (%s) return resword;\n", links->index, comp_buffer);
								else
									printf ("              resword = \"%s\";\n              if (%s) return resword;\n", links->key, comp_buffer);
							printf ("              return 0;\n");
						}
					else if (temp->next && temp->hash_value == temp->next->hash_value)
						{

							for ( ; temp->next && temp->hash_value == temp->next->hash_value;
									 temp = temp->next)
								if (pointer_and_type_enabled)
									printf ("              resword = &wordlist[%d];\n              if (%s) return resword;\n", temp->index, comp_buffer);
								else
									printf ("              resword = \"%s\";\n              if (%s) return resword;\n", temp->key, comp_buffer);

							if (pointer_and_type_enabled)
								printf ("              resword = &wordlist[%d];\n              return %s ? resword : 0;\n", temp->index, comp_buffer);
							else
								printf ("              resword = \"%s\";\n              return %s ? resword : 0;\n", temp->key, comp_buffer);
						}
					else 
						{
							if (pointer_and_type_enabled)
								printf ("              resword = &wordlist[%d];\n", temp->index);
							else 
								printf ("              resword = \"%s\";\n", temp->key);
							if (option[LENTABLE] && !option[DUP])
								printf ("              return len != %d || %s ? 0 : resword;\n", temp->length, comp_buffer);
							else 
								printf ("              return %s ? resword : 0;\n", comp_buffer);
						}
				}
      printf ("            }\n        }\n    }\n  return 0;\n}\n");
    }
  else                          /* Nothing special required here. */
    {                        
      List_Node *temp;

      printf ("          char *s = \"\";\n\n          switch (key)\n            {\n");
      
      for (temp = head; temp; temp = temp->next)
        if (option[LENTABLE])
          printf ("            case %*d: if (len == %d) s = \"%s\"; else return 0; break;\n", 
                  field_width, temp->hash_value, temp->length, temp->key);
        else
          printf ("            case %*d: s = \"%s\"; else return 0; break;\n", 
                  field_width, temp->hash_value, temp->key);
                      
        printf ("            default: return 0;\n            }\n          return !%s;\n        }\n    }\n  return 0;\n}\n", 
                option[COMP] ? "strncmp (s, str, len)" : "strcmp (s, str)");

		}
}

/* Prints out a table of keyword lengths, for use with the 
   comparison code in generated function ``in_word_set.'' */

void 
Key_List::print_keylength_table (void)
{
  const int  max_column = 15;
  int        index      = 0;
  int        column     = 0;
  char      *indent     = option[GLOBAL] ? "" : "  ";
  List_Node *temp;

  if (!option[DUP] && !option[SWITCH]) 
		{
      printf ("\n%sstatic unsigned %s lengthtable[] =\n%s%s{\n    ",
              indent, max_key_len < 256 ? "char" : (max_key_len < 65536 ? "short" : "long"),
              indent, indent);
  
			for (temp = head; temp; temp = temp->next, index++)
				{
    
					if (index < temp->hash_value)
						{
      
							for ( ; index < temp->hash_value; index++)
								printf ("%3d%s", 0, ++column % (max_column - 1) ? "," : ",\n    ");
						}
    
					printf ("%3d%s", temp->length, ++column % (max_column - 1 ) ? "," : ",\n    ");
				}
  
      printf ("\n%s%s};\n\n", indent, indent);
		}
}
/* Prints out the array containing the key words for the Perfect
   hash function. */
  
void 
Key_List::print_keyword_table (void)
{
  char      *l_brace = *head->rest ? "{" : "";
  char      *r_brace = *head->rest ? "}," : "";
  int        doing_switch = option[SWITCH];
  char      *indent       = option[GLOBAL] ? "" : "  ";
  int        index = 0;
  List_Node *temp;

  printf ("\n  static %s wordlist[] =\n    {\n", struct_tag);
  
  /* generate an array of reserved words at appropriate locations */
  
  for (temp = head; temp; temp = temp->next, index++)
    {
			temp->index = index;

			if (!doing_switch && index < temp->hash_value)
				{
					int column;

					printf ("      ");
      
					for (column = 1; index < temp->hash_value; index++, column++)
						printf ("%s\"\",%s %s", l_brace, r_brace, column % 9 ? "" : "\n      ");
      
					if (column % 10)
						printf ("\n");
					else 
						{
							printf ("%s\"%s\", %s%s\n", l_brace, temp->key, temp->rest, r_brace);
							continue;
						}
				}

			printf ("      %s\"%s\", %s%s\n", l_brace, temp->key, temp->rest, r_brace);

			/* Deal with links specially. */
			if (temp->link)
				{
					List_Node *links;

					for (links = temp->link; links; links = links->link)
						{
							links->index = ++index;
							printf ("      %s\"%s\", %s%s\n", l_brace, temp->key, temp->rest, r_brace);
						}
				}

		}

  printf ("%s%s};\n\n", indent, indent);
}

/* Generates C code for the hash function that returns the
   proper encoding for each key word. */

void 
Key_List::print_hash_function (int max_hash_value)
{
  const int max_column  = 10;
  int       count       = max_hash_value;

  /* Calculate maximum number of digits required for MAX_HASH_VALUE. */

  while ((count /= 10) > 0)
    field_width++;

  if (option[GNU])
    printf ("#ifdef __GNUC__\ninline\n#endif\n");
  
  printf (option[ANSI] 
          ? "static int\n%s (register char *str, register int len)\n{\n  static unsigned %s hash_table[] =\n    {"
          : "static int\n%s (str, len)\n     register char  *str;\n     register int  len;\n{\n  static unsigned %s hash_table[] =\n    {",
          option.get_hash_name (), max_hash_value < 256 ? "char" : (max_hash_value < 65536 ? "short" : "int"));
  
  for (count = 0; count < ALPHABET_SIZE; ++count)
    {
      if (!(count % max_column))
        printf ("\n    ");
      
      printf ("%*d,", field_width, occurrences[count] ? asso_values[count] : max_hash_value);
    }
  
  /* Optimize special case of ``-k 1,$'' */
  if (option[DEFAULTCHARS]) 
    printf ("\n    };\n  return %s + hash_table[str[len - 1]] + hash_table[str[0]];\n}\n\n",
            option[NOLENGTH] ? "0" : "len");
  else
    {
      int key_pos;

      option.reset ();

      /* Get first (also highest) key position. */
      key_pos = option.get (); 
      
      /* We can perform additional optimizations here. */
      if (!option[ALLCHARS] && key_pos <= min_key_len) 
        { 
          printf ("\n  };\n  return %s", option[NOLENGTH] ? "0" : "len");
          
          do
            printf (" + hash_table[str[%d]]", key_pos - 1);
          while ((key_pos = option.get ()) != EOS && key_pos != WORD_END);
          
          printf ("%s;\n}\n\n", key_pos == WORD_END ? " + hash_table[str[len - 1]]" : "");
        }

      /* We've got to use the correct, but brute force, technique. */
      else 
        {                    
          printf ("\n    };\n  register int hval = %s ;\n\n  switch (%s)\n    {\n      default:\n",
                  option[NOLENGTH] ? "0" : "len", option[NOLENGTH] ? "len" : "hval");
          
          /* User wants *all* characters considered in hash. */
          if (option[ALLCHARS]) 
            { 
              int i;

              for (i = max_key_len; i > 0; i--)
                printf ("      case %d:\n        hval += hash_table[str[%d]];\n", i, i - 1);
              
              printf ("    }\n  return hval;\n}\n\n");
            }
          else /* do the hard part... */
            {                
              count = key_pos + 1;
              
              do
                {
                  
                  while (--count > key_pos)
                    printf ("      case %d:\n", count);
                  
                  printf ("      case %d:\n        hval += hash_table[str[%d]];\n", key_pos, key_pos - 1);
                }
              while ((key_pos = option.get ()) != EOS && key_pos != WORD_END);
              
              printf ("    }\n  return hval%s ;\n}\n\n", key_pos == WORD_END ? " + hash_table[str[len - 1]]" : "");
          }
      }
  }
}

/* Generates C code to perform the keyword lookup. */

void 
Key_List::print_lookup_function (void)
{ 
  printf ("  if (len <= MAX_WORD_LENGTH && len >= MIN_WORD_LENGTH)\n    {\n"
          "      register int key = hash (str, len);\n\n"
          "      if (key <= MAX_HASH_VALUE && key >= MIN_HASH_VALUE)\n        {\n"
          "          register char *s = wordlist[key]");
  if (array_type != default_array_type)
    printf (".%s", option.get_key_name ());

  printf (";\n\n          if (%s*s == *str && !%s)\n            return %s",
          option[LENTABLE] ? "len == lengthtable[key]\n              && " : "",
          option[COMP] ? "strncmp (str + 1, s + 1, len - 1)" : "strcmp (str + 1, s + 1)",
          option[TYPE] && option[POINTER] ? "&wordlist[key]" : "s");
  printf (";\n        }\n    }\n  return 0;\n}\n");
}

/* Generates the hash function and the key word recognizer function
   based upon the user's Options. */
  
void 
Key_List::output (void)
{
  int global_table = option[GLOBAL]
;	
  printf ("/* C code produced by gperf version %s */\n", version_string);
  option.print_options ();
  
  printf ("%s\n", include_src);
  
  if (option[TYPE] && !option[NOTYPE]) /* Output type declaration now, reference it later on.... */
    printf ("%s;\n", array_type);
  
  print_hash_function (print_min_max ());
  
  if (global_table)
    if (option[SWITCH])
      {
        if (option[LENTABLE] && option[DUP])
          print_keylength_table ();
        if (option[POINTER] && option[TYPE])
          print_keyword_table ();
      }
    else
      {
        if (option[LENTABLE])
          print_keylength_table ();
        print_keyword_table ();
      }

  if (option[GNU]) /* Use the inline keyword to remove function overhead. */
    printf ("#ifdef __GNUC__\ninline\n#endif\n");
  
  if (option[ANSI]) /* Use ANSI function prototypes. */
    printf ("%s\n%s (register char *str, register int len)\n{\n", return_type, option.get_function_name ());
  else /* Use traditional K&R C style. */
    printf ("%s\n%s  (str, len)\n     register char *str;\n     register int len;\n{\n", return_type, option.get_function_name ());

  /* Use the switch in place of lookup table. */
  if (option[SWITCH])
    {               
      if (!global_table)
        {
          if (option[LENTABLE] && option[DUP])
            print_keylength_table ();
          if (option[POINTER] && option[TYPE]) 
            print_keyword_table ();
        }
      print_switch ();
    }
  else                /* Use the lookup table, in place of switch. */
    {           
      if (!global_table)
        {
          if (option[LENTABLE])
            print_keylength_table ();
          print_keyword_table ();
        }
      print_lookup_function ();
    }

  if (additional_code)
    {
      int c;

      while ((c = getchar ()) != EOF)
        putchar (c);
    }
}

/* Sorts the keys by hash value. */

void 
Key_List::sort (void) 
{ 
  hash_sort       = TRUE;
  occurrence_sort = FALSE;
  
  head = merge_sort (head);
}

/* Dumps the key list to stderr stream. */

void 
Key_List::dump () 
{      
  List_Node *ptr;

  report_error ("\nList contents are:\n"
                         "(hash value, key length, index, key set, uniq set, key):\n");
  
  for (ptr = head; ptr; ptr = ptr->next)
    report_error ("      %d,      %d,     %d, %s, %s, %s\n",
                           ptr->hash_value, ptr->length, ptr->index,
                           ptr->key_set, ptr->uniq_set, ptr->key);
}

/* Simple-minded constructor action here... */

Key_List::Key_List (void) 
{   
  total_keys      = 1;
  max_key_len     = NEG_MAX_INT;
  min_key_len     = MAX_INT;
  return_type     = default_return_type;
  array_type      = struct_tag  = default_array_type;
  head            = NULL;
  additional_code = FALSE;
}

/* Returns the length of entire key list. */

int 
Key_List::length (void) 
{ 
  return list_len;
}

/* Returns length of longest key read. */

int 
Key_List::max_key_length (void) 
{ 
  return max_key_len;
}

