/*
 * Electric(tm) VLSI Design System
 *
 * File: dblangmm.c
 * Mathematica interface module
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "config.h"
#if LANGMM

/******************* THESE DECLARATIONS BELONG IN "dblang.h" *******************/
# include <mathlink.h>
  void db_mathematicaprocesspackets(INTSML);
  INTSML db_mathematicagetstring(INTSML);
  INTSML db_mathematicainit(void);
  extern MLINK db_mathematicalink;
/********************* END OF DECLARATIONS FOR "dblang.h" **********************/

#include "global.h"
#include "dblang.h"

#ifdef	MATHEMATICA

/* prototypes for local routines */
void db_mathematicawritetype(INTBIG, INTBIG);
INTSML db_mathematicagettype(INTBIG*, INTBIG);
INTSML db_mathematicagetaddrtype(INTBIG*, INTBIG*);
void db_mathematicaHandleCallPacket(void);
void db_mathematicadefaulthandler(MLINK, unsigned long, unsigned long);

static int db_mm_getval(void);
static int db_mm_setval(void);
static int db_mm_setind(void);
static int db_mm_delval(void);
static int db_mm_initsearch(void);
static int db_mm_nextobject(void);
static int db_mm_getaid(void);
static int db_mm_maxaid(void);
static int db_mm_indexaid(void);
static int db_mm_aidturnon(void);
static int db_mm_aidturnoff(void);
static int db_mm_tellaid(void);
static int db_mm_curlib(void);
static int db_mm_getlibrary(void);
static int db_mm_newlibrary(void);
static int db_mm_killlibrary(void);
static int db_mm_eraselibrary(void);
static int db_mm_selectlibrary(void);
static int db_mm_curtech(void);
static int db_mm_gettechnology(void);
static int db_mm_changelambda(void);
static int db_mm_getnodeproto(void);
static int db_mm_getcell(void);
static int db_mm_newnodeproto(void);
static int db_mm_killnodeproto(void);
static int db_mm_copynodeproto(void);
static int db_mm_iconview(void);
static int db_mm_contentsview(void);
static int db_mm_newnodeinst(void);
static int db_mm_modifynodeinst(void);
static int db_mm_killnodeinst(void);
static int db_mm_replacenodeinst(void);
static int db_mm_nodefunction(void);
static int db_mm_newarcinst(void);
static int db_mm_modifyarcinst(void);
static int db_mm_killarcinst(void);
static int db_mm_replacearcinst(void);
static int db_mm_getarcproto(void);
static int db_mm_getnetwork(void);
static int db_mm_getpinproto(void);
static int db_mm_getportproto(void);
static int db_mm_newportproto(void);
static int db_mm_portposition(void);
static int db_mm_killportproto(void);
static int db_mm_moveportproto(void);
static int db_mm_undoabatch(void);
static int db_mm_noundoallowed(void);
static int db_mm_getview(void);
static int db_mm_newview(void);
static int db_mm_killview(void);

MLINK db_mathematicalink = NULL;

/************************* ELECTRIC INTERFACE ROUTINES *************************/

static int db_mm_getval(void)
{
	INTBIG addr, type;
	char *name;
	VARIABLE *var;

	/* get the arguments */
	if (db_mathematicagetaddrtype(&type, &addr) != 0) return(0);
	if (MLGetString(db_mathematicalink, &name) == 0) return(0);

	/* do the Electric function */
	var = getval(addr, type, -1, name);
	MLDisownString(db_mathematicalink, name);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	if (var == NOVARIABLE) MLPutString(db_mathematicalink, "NOVARIABLE"); else
		db_mathematicawritetype(var->addr, var->type);
	return(1);
}

static int db_mm_setval(void)
{
	INTBIG addr, type, naddr, ntype, opt;
	char *name;
	VARIABLE *var;

	/* get the arguments */
	if (db_mathematicagetaddrtype(&type, &addr) != 0) return(0);
	if (MLGetString(db_mathematicalink, &name) == 0) return(0);
	if (db_mathematicagetaddrtype(&ntype, &naddr) != 0) return(0);
	if (MLGetInteger(db_mathematicalink, &opt) == 0) return(0);

	/* do the Electric function */
	var = setval(addr, type, name, naddr, ntype|opt);
	MLDisownString(db_mathematicalink, name);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	if (var == NOVARIABLE) MLPutString(db_mathematicalink, "NOVARIABLE"); else
		db_mathematicawritetype(var->addr, var->type);
	return(1);
}

static int db_mm_setind(void)
{
	INTBIG addr, type, naddr, ntype, aindex;
	INTSML ret;
	char *name;
	VARIABLE *var;

	/* get the arguments */
	if (db_mathematicagetaddrtype(&type, &addr) != 0) return(0);
	if (MLGetString(db_mathematicalink, &name) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &aindex) == 0) return(0);
	if (db_mathematicagetaddrtype(&ntype, &naddr) != 0) return(0);

	/* do the Electric function */
	ret = setind(addr, type, name, aindex, naddr);
	MLDisownString(db_mathematicalink, name);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_delval(void)
{
	INTBIG addr, type;
	INTSML ret;
	char *name;

	/* get the arguments */
	if (db_mathematicagetaddrtype(&type, &addr) != 0) return(0);
	if (MLGetString(db_mathematicalink, &name) == 0) return(0);

	/* do the Electric function */
	ret = delval(addr, type, name);
	MLDisownString(db_mathematicalink, name);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_initsearch(void)
{
	NODEPROTO *facet;
	INTBIG lx, hx, ly, hy, search;

	/* get the arguments */
	if (MLGetInteger(db_mathematicalink, &lx) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &hx) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &ly) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &hy) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	search = initsearch(lx, hx, ly, hy, facet);

	/* send the result back */
	MLPutInteger(db_mathematicalink, search);
	return(1);
}

static int db_mm_nextobject(void)
{
	INTBIG search;
	GEOM *geom;

	/* get the argument */
	if (MLGetInteger(db_mathematicalink, &search) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	geom = nextobject(search);

	/* send the result back */
	db_mathematicawritetype((INTBIG)geom, VGEOM);
	return(1);
}

static int db_mm_getaid(void)
{
	char *aidname;
	AIDENTRY *aid;

	/* get the argument */
	if (MLGetString(db_mathematicalink, &aidname) == 0) return(0);

	/* do the Electric function */
	aid = getaid(aidname);
	MLDisownString(db_mathematicalink, aidname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)aid, VAID);
	return(1);
}

static int db_mm_maxaid(void)
{
	/* no arguments */
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	MLPutInteger(db_mathematicalink, el_maxaid);
	return(1);
}

static int db_mm_indexaid(void)
{
	int aidindex;
	AIDENTRY *aid;

	/* get the argument */
	if (MLGetInteger(db_mathematicalink, &aidindex) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	if (aidindex < 0 || aidindex >= el_maxaid) return(0);
	aid = &el_aids[aidindex];

	/* send the result back */
	db_mathematicawritetype((INTBIG)aid, VAID);
	return(1);
}

static int db_mm_aidturnon(void)
{
	AIDENTRY *aid;
	int ncu;
	INTSML nocatchup;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&aid, VAID) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &ncu) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	nocatchup = ncu;
	aidturnon(aid, nocatchup);

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_aidturnoff(void)
{
	AIDENTRY *aid;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&aid, VAID) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	aidturnoff(aid, 0);

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_tellaid(void)
{
	AIDENTRY *aid;
	long len;
	INTSML count, i, ret;
	char *par[20];

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&aid, VAID) == 0) return(0);
	if (MLCheckFunction(db_mathematicalink, "List", &len) == 0) return(0);
	count = len;
	if (count > 20) return(0);
	for(i=0; i<count; i++)
		if (MLGetString(db_mathematicalink, &par[i]) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = tellaid(aid, count, par);
	for(i=0; i<count; i++) MLDisownString(db_mathematicalink, par[i]);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_curlib(void)
{
	/* no arguments */
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)el_curlib, VLIBRARY);
	return(1);
}

static int db_mm_getlibrary(void)
{
	char *libname;
	LIBRARY *lib;

	/* get the argument */
	if (MLGetString(db_mathematicalink, &libname) == 0) return(0);

	/* do the Electric function */
	lib = getlibrary(libname);
	MLDisownString(db_mathematicalink, libname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)lib, VLIBRARY);
	return(1);
}

static int db_mm_newlibrary(void)
{
	char *libname, *libfile;
	LIBRARY *lib;

	/* get the arguments */
	if (MLGetString(db_mathematicalink, &libname) == 0) return(0);
	if (MLGetString(db_mathematicalink, &libfile) == 0) return(0);

	/* do the Electric function */
	lib = newlibrary(libname, libfile);
	MLDisownString(db_mathematicalink, libname);
	MLDisownString(db_mathematicalink, libfile);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)lib, VLIBRARY);
	return(1);
}

static int db_mm_killlibrary(void)
{
	LIBRARY *lib;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&lib, VLIBRARY) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	killlibrary(lib);

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_eraselibrary(void)
{
	LIBRARY *lib;

	/* get the library function and convert it to a library pointer */
	if (db_mathematicagettype((INTBIG *)&lib, VLIBRARY) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	eraselibrary(lib);

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_selectlibrary(void)
{
	LIBRARY *lib;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&lib, VLIBRARY) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	selectlibrary(lib);

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_curtech(void)
{
	/* no arguments */
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)el_curtech, VTECHNOLOGY);
	return(1);
}

static int db_mm_gettechnology(void)
{
	char *techname;
	TECHNOLOGY *tech;

	/* get the argument */
	if (MLGetString(db_mathematicalink, &techname) == 0) return(0);

	/* do the Electric function */
	tech = gettechnology(techname);
	MLDisownString(db_mathematicalink, techname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)tech, VTECHNOLOGY);
	return(1);
}

static int db_mm_changelambda(void)
{
	TECHNOLOGY *tech;
	INTBIG oldlam, newlam;

	/* get the arguments */
	if (MLGetInteger(db_mathematicalink, &oldlam) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &newlam) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&tech, VTECHNOLOGY) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	changelambda(oldlam, newlam, tech);

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_getnodeproto(void)
{
	char *npname;
	NODEPROTO *np;

	/* get the library name and convert it to a library pointer */
	if (MLGetString(db_mathematicalink, &npname) == 0) return(0);

	/* do the Electric function */
	np = getnodeproto(npname);
	MLDisownString(db_mathematicalink, npname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)np, VNODEPROTO);
	return(1);
}

static int db_mm_getcell(void)
{
	char *cellname;
	CELL *cell;

	/* get the argument */
	if (MLGetString(db_mathematicalink, &cellname) == 0) return(0);

	/* do the Electric function */
	cell = getcell(cellname);
	MLDisownString(db_mathematicalink, cellname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)cell, VCELL);
	return(1);
}

static int db_mm_newnodeproto(void)
{
	char *facetname;
	NODEPROTO *facet;
	LIBRARY *lib;

	/* get the arguments */
	if (MLGetString(db_mathematicalink, &facetname) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&lib, VLIBRARY) == 0) return(0);

	/* do the Electric function */
	facet = newnodeproto(facetname, lib);
	MLDisownString(db_mathematicalink, facetname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)facet, VNODEPROTO);
	return(1);
}

static int db_mm_killnodeproto(void)
{
	NODEPROTO *facet;
	INTSML ret;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = killnodeproto(facet);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_copynodeproto(void)
{
	NODEPROTO *fromfacet, *facet;
	LIBRARY *tolib;
	char *toname;
	INTSML ret;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&fromfacet, VNODEPROTO) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&tolib, VLIBRARY) == 0) return(0);
	if (MLGetString(db_mathematicalink, &toname) == 0) return(0);

	/* do the Electric function */
	facet = copynodeproto(fromfacet, tolib, toname);
	MLDisownString(db_mathematicalink, toname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)facet, VNODEPROTO);
	return(1);
}

static int db_mm_iconview(void)
{
	NODEPROTO *facet, *ifacet;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ifacet = iconview(facet);

	/* send the result back */
	db_mathematicawritetype((INTBIG)ifacet, VNODEPROTO);
	return(1);
}

static int db_mm_contentsview(void)
{
	NODEPROTO *facet, *cfacet;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	cfacet = contentsview(facet);

	/* send the result back */
	db_mathematicawritetype((INTBIG)cfacet, VNODEPROTO);
	return(1);
}

static int db_mm_newnodeinst(void)
{
	NODEPROTO *pro, *facet;
	NODEINST *ni;
	INTBIG lx, hx, ly, hy, tr, ro;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&pro, VNODEPROTO) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &lx) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &hx) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &ly) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &hy) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &tr) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &ro) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ni = newnodeinst(pro, lx, hx, ly, hy, tr, ro, facet);

	/* send the result back */
	db_mathematicawritetype((INTBIG)ni, VNODEINST);
	return(1);
}

static int db_mm_modifynodeinst(void)
{
	NODEINST *ni;
	INTBIG dlx, dhx, dly, dhy, dtr, dro;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&ni, VNODEINST) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dlx) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dly) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dhx) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dhy) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dro) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dtr) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	modifynodeinst(ni, dlx, dly, dhx, dhy, dro, dtr);

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_killnodeinst(void)
{
	NODEINST *ni;
	INTSML ret;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&ni, VNODEINST) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = killnodeinst(ni);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_replacenodeinst(void)
{
	NODEINST *ni, *newni;
	NODEPROTO *np;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&ni, VNODEINST) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&np, VNODEPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	newni = replacenodeinst(ni, np);

	/* send the result back */
	db_mathematicawritetype((INTBIG)newni, VNODEINST);
	return(1);
}

static int db_mm_nodefunction(void)
{
	NODEINST *ni;
	char *extra;
	INTSML fun;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&ni, VNODEINST) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	fun = nodefunction(ni, &extra);

	/* send the result back */
	MLPutInteger(db_mathematicalink, fun);
	return(1);
}

static int db_mm_newarcinst(void)
{
	ARCPROTO *pro;
	NODEPROTO *facet;
	NODEINST *na, *nb;
	PORTPROTO *pa, *pb;
	ARCINST *ai;
	INTBIG wid, bits, xa, ya, xb, yb;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&pro, VARCPROTO) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &wid) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &bits) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&na, VNODEINST) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&pa, VPORTPROTO) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &xa) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &ya) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&nb, VNODEINST) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&pb, VPORTPROTO) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &xb) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &yb) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ai = newarcinst(pro, wid, bits, na,pa,xa,ya, nb,pb,xb,yb, facet);

	/* send the result back */
	db_mathematicawritetype((INTBIG)ai, VARCINST);
	return(1);
}

static int db_mm_modifyarcinst(void)
{
	ARCINST *ai;
	INTBIG dwid, dx1, dy1, dx2, dy2;
	INTSML ret;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&ai, VARCINST) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dwid) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dx1) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dy1) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dx2) == 0) return(0);
	if (MLGetInteger(db_mathematicalink, &dy2) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = modifyarcinst(ai, dwid, dx1, dy1, dx2, dy2);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_killarcinst(void)
{
	ARCINST *ai;
	INTSML ret;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&ai, VARCINST) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = killarcinst(ai);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_replacearcinst(void)
{
	ARCINST *ai, *newai;
	ARCPROTO *ap;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&ai, VARCINST) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&ap, VARCPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	newai = replacearcinst(ai, ap);

	/* send the result back */
	db_mathematicawritetype((INTBIG)newai, VARCINST);
	return(1);
}

static int db_mm_getarcproto(void)
{
	char *arcname;
	ARCPROTO *ap;

	/* get the argument */
	if (MLGetString(db_mathematicalink, &arcname) == 0) return(0);

	/* do the Electric function */
	ap = getarcproto(arcname);
	MLDisownString(db_mathematicalink, arcname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)ap, VARCPROTO);
	return(1);
}

static int db_mm_getnetwork(void)
{
	char *netname;
	NODEPROTO *facet;
	NETWORK *net;

	/* get the arguments */
	if (MLGetString(db_mathematicalink, &netname) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);

	/* do the Electric function */
	net = getnetwork(netname, facet);
	MLDisownString(db_mathematicalink, netname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)net, VNETWORK);
	return(1);
}

static int db_mm_getpinproto(void)
{
	ARCPROTO *ap;
	NODEPROTO *np;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&ap, VARCPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	np = getpinproto(ap);

	/* send the result back */
	db_mathematicawritetype((INTBIG)np, VNODEPROTO);
	return(1);
}

static int db_mm_getportproto(void)
{
	char *portname;
	NODEPROTO *np;
	PORTPROTO *pp;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&np, VNODEPROTO) == 0) return(0);
	if (MLGetString(db_mathematicalink, &portname) == 0) return(0);

	/* do the Electric function */
	pp = getportproto(np, portname);
	MLDisownString(db_mathematicalink, portname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)pp, VPORTPROTO);
	return(1);
}

static int db_mm_newportproto(void)
{
	char *portname;
	NODEPROTO *facet;
	NODEINST *ni;
	PORTPROTO *pp, *subpp;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&ni, VNODEINST) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&subpp, VPORTPROTO) == 0) return(0);
	if (MLGetString(db_mathematicalink, &portname) == 0) return(0);

	/* do the Electric function */
	pp = newportproto(facet, ni, subpp, portname);
	MLDisownString(db_mathematicalink, portname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)pp, VPORTPROTO);
	return(1);
}

static int db_mm_portposition(void)
{
	NODEINST *ni;
	PORTPROTO *pp;
	INTBIG pos[2];

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&ni, VNODEINST) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&pp, VPORTPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	portposition(ni, pp, &pos[0], &pos[1]);

	/* send the result back */
	MLPutIntegerList(db_mathematicalink, pos, 2);
	return(1);
}

static int db_mm_killportproto(void)
{
	NODEPROTO *facet;
	PORTPROTO *pp;
	INTSML ret;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&pp, VPORTPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = killportproto(facet, pp);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_moveportproto(void)
{
	NODEPROTO *facet;
	NODEINST *newni;
	PORTPROTO *oldpp, *newpp;
	INTSML ret;

	/* get the arguments */
	if (db_mathematicagettype((INTBIG *)&facet, VNODEPROTO) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&oldpp, VPORTPROTO) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&newni, VNODEINST) == 0) return(0);
	if (db_mathematicagettype((INTBIG *)&newpp, VPORTPROTO) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = moveportproto(facet, oldpp, newni, newpp);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_undoabatch(void)
{
	INTSML ret;
	AIDENTRY *aid;

	/* no arguments */
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = undoabatch(&aid);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static int db_mm_noundoallowed(void)
{
	/* no arguments */
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	noundoallowed();

	/* no meaningful result */
	MLPutInteger(db_mathematicalink, 0);
	return(1);
}

static int db_mm_getview(void)
{
	char *viewname;
	VIEW *view;

	/* get the argument */
	if (MLGetString(db_mathematicalink, &viewname) == 0) return(0);

	/* do the Electric function */
	view = getview(viewname);
	MLDisownString(db_mathematicalink, viewname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)view, VVIEW);
	return(1);
}

static int db_mm_newview(void)
{
	char *viewname, *sviewname;
	VIEW *v;

	/* get the arguments */
	if (MLGetString(db_mathematicalink, &viewname) == 0) return(0);
	if (MLGetString(db_mathematicalink, &sviewname) == 0) return(0);

	/* do the Electric function */
	v = newview(viewname, sviewname);
	MLDisownString(db_mathematicalink, viewname);
	MLDisownString(db_mathematicalink, sviewname);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* send the result back */
	db_mathematicawritetype((INTBIG)v, VVIEW);
	return(1);
}

static int db_mm_killview(void)
{
	INTSML ret;
	VIEW *v;

	/* get the argument */
	if (db_mathematicagettype((INTBIG *)&v, VVIEW) == 0) return(0);
	if (MLNewPacket(db_mathematicalink) == 0) return(0);

	/* do the Electric function */
	ret = killview(v);

	/* send the result back */
	MLPutInteger(db_mathematicalink, ret);
	return(1);
}

static struct func
{
	int   f_nargs;
	int   (*f_func)(void);
	char *prototype;
	char *args;
} db_mathematicatraps[] =
{
	/* define the database examination predicates */
	2, db_mm_getval, "GetVal[object_, attribute_String]", "{object, attribute}",
	4, db_mm_setval, "SetVal[object_, attribute_String, value_, opt_Integer]", "{object, attribute, value, opt}",
	4, db_mm_setind, "SetInd[object_, attribute_String, index_Integer, value_]", "{object, attribute, index, value}",
	2, db_mm_delval, "DelVal[object_, attribute_String]", "{object, attribute}",
	5, db_mm_initsearch, "InitSearch[lx_Integer, hx_Integer, ly_Integer, hy_Integer, facet_NodeProto]", "{lx, hx, ly, hy, facet}",
	1, db_mm_nextobject, "NextObject[search_Integer]", "{search}",

	/* define the aid predicates */
	1, db_mm_getaid, "GetAid[name_String]", "{name}",
	0, db_mm_maxaid, "MaxAid[]", "{}",
	1, db_mm_indexaid, "IndexAid[index_Integer]", "{index}",
	2, db_mm_aidturnon, "AidTurnOn[aid_Aid, ncu_Integer]", "{aid, ncu}",
	1, db_mm_aidturnoff, "AidTurnOff[aid_Aid]", "{aid}",
	2, db_mm_tellaid, "TellAid[aid_Aid, message_List]", "{aid, message}",

	/* define the library predicates */
	0, db_mm_curlib, "CurLib[]", "{}",
	1, db_mm_getlibrary, "GetLibrary[name_String]", "{name}",
	2, db_mm_newlibrary, "NewLibrary[name_String, file_String]", "{name, file}",
	1, db_mm_killlibrary, "KillLibrary[lib_Library]", "{lib}",
	1, db_mm_eraselibrary, "EraseLibrary[lib_Library]", "{lib}",
	1, db_mm_selectlibrary, "SelectLibrary[lib_Library]", "{lib}",

	/* define the technology predicates */
	0, db_mm_curtech, "CurTech[]", "{}",
	1, db_mm_gettechnology, "GetTechnology[name_String]", "{name}",
	3, db_mm_changelambda, "ChangeLambda[oldlam_Integer, newlam_Integer, tech_Technology]", "{oldlam, newlam, tech}",

	/* define the facet predicates */
	1, db_mm_getnodeproto, "GetNodeProto[name_String]", "{name}",
	1, db_mm_getcell, "GetCell[name_String]", "{name}",
	2, db_mm_newnodeproto, "NewNodeProto[name_String, lib_Library]", "{name, lib}",
	1, db_mm_killnodeproto, "KillNodeProto[facet_NodeProto]", "{facet}",
	3, db_mm_copynodeproto, "CopyNodeProto[fromfacet_NodeProto, tolib_Library, toname_String]", "{fromfacet, tolib, toname}",
	1, db_mm_iconview, "IconView[facet_NodeProto]", "{facet}",
	1, db_mm_contentsview, "ContentsView[facet_NodeProto]", "{facet}",

	/* define the node predicates */
	8, db_mm_newnodeinst, "NewNodeInst[pro_NodeProto, lx_Integer, hx_Integer, ly_Integer, hy_Integer, tr_Integer, ro_Integer, facet_NodeProto]",
		"{pro, lx, hx, ly, hy, tr, ro, facet}",
	7, db_mm_modifynodeinst, "ModifyNodeInst[ni_NodeInst, dlx_Integer, dly_Integer, dhx_Integer, dhy_Integer, dro_Integer, dtr_Integer]",
		"{ni, dlx, dly, dhx, dhy, dro, dtr}",
	1, db_mm_killnodeinst, "KillNodeInst[ni_NodeInst]", "{ni}",
	2, db_mm_replacenodeinst, "ReplaceNodeInst[ni_NodeInst, pro_NodeProto]", "{ni, pro}",
	1, db_mm_nodefunction, "NodeFunction[ni_NodeInst]", "{ni}",

	/* define the arc predicates */
	12, db_mm_newarcinst, "NewArcInst[pro_ArcProto, wid_Integer, bits_Integer, na_NodeInst, pa_PortProto, xa_Integer, ya_Integer, nb_NodeInst, pb_PortProto, xb_Integer, yb_Integer, facet_NodeProto]",
		"{pro, wid, bits, na, pa, xa, ya, nb, pb, xb, yb, facet}",
	6, db_mm_modifyarcinst, "ModifyArcInst[ai_ArcInst, dwid_Integer, dx1_Integer, dy1_Integer, dx2_Integer, dy2_Integer]", "{ai, dwid, dx1, dy1, dx2, dy2}",
	1, db_mm_killarcinst, "KillArcInst[ai_ArcInst]", "{ai}",
	2, db_mm_replacearcinst, "ReplaceArcInst[ai_ArcInst, pro_ArcProto]", "{ai, pro}",
	1, db_mm_getarcproto, "GetArcProto[name_String]", "{name}",
	2, db_mm_getnetwork, "GetNetwork[name_String, facet_NodeProto]", "{name, facet}",
	1, db_mm_getpinproto, "GetPinProto[arc_ArcProto]", "{arc}",

	/* define the portproto predicates */
	2, db_mm_getportproto, "GetPortProto[np_NodeProto, name_String]", "{np, name}",
	4, db_mm_newportproto, "NewPortProto[facet_NodeProto, ni_NodeInst, subpp_PortProto, name_String]", "{facet, ni, subpp, name}",
	2, db_mm_portposition, "PortPosition[ni_NodeInst, pp_PortProto]", "{ni, pp}",
	2, db_mm_killportproto, "KillPortProto[facet_NodeProto, pp_PortProto]", "{facet, pp}",
	4, db_mm_moveportproto, "MovePortProto[facet_NodeProto, oldpp_PortProto, newni_NodeInst, newsubpp_PortProto]", "{facet, oldpp, newni, newsubpp}",

	/* define the change control predicates */
	0, db_mm_undoabatch, "UndoABatch[]", "{}",
	0, db_mm_noundoallowed, "NoUndoAllowed[]", "{}",

	/* define the view predicates */
	1, db_mm_getview, "GetView[name_String]", "{name}",
	2, db_mm_newview, "NewView[name_String, abbrev_String]", "{name, abbrev}",
	1, db_mm_killview, "KillView[view_View]", "{view}",
	-1
};
static INTSML db_mathematicatrapcount;
static INTSML db_mathematicadone, db_mathematicaabort;

/****************************** SUPPORT ******************************/

/*
 * routine to loop until the Mathematica RETURN packet is available.  If "conversation"
 * is nonzero, this is an interactive call, so print that packet.  Otherwise wait for
 * it BUT DO NOT GET IT.
 */
void db_mathematicaprocesspackets(INTSML conversation)
{
	REGISTER INTBIG pkt;
	REGISTER INTSML i, line, save;
	long count;
	double f;
	char *str, *sym, *pt, *ept;

	for(;;)
	{
		MLFlush(db_mathematicalink);
		pkt = MLNextPacket(db_mathematicalink);
		if (MLError(db_mathematicalink) != MLEOK)
		{
			ttyputerr("Mathematica error: %s", MLErrorMessage(db_mathematicalink));
			break;
		}
		if (pkt == 0) break;
		if (pkt == RETURNPKT)
		{
			if (conversation == 0) break;
			(void)initinfstr();
			(void)db_mathematicagetstring(0);
			ttyputmsg("%s", returninfstr());
			break;
		}
		if (pkt == CALLPKT)
		{
			db_mathematicaHandleCallPacket();
			continue;
		}
		if (pkt == RETURNTEXTPKT)
		{
			MLGetString(db_mathematicalink, &str);
			if (str != 0)
			{
				pt = str;
				for(line=0; ; line++)
				{
					for(ept = pt; *ept != 0 && *ept != '\n' && *ept != '\r'; ept++) ;
					save = *ept;
					if (*ept != 0) *ept = 0;
					ttyputmsg("%s", pt);
					*ept = save;
					if (save == 0) break;
					pt = ept+1;
				}
				MLDisownString(db_mathematicalink, str);
			}
			continue;
		}
		if (pkt == TEXTPKT)
		{
			if (MLGetString(db_mathematicalink, &str) == 0) break;
			if ((str[0] != '\r' && str[0] != '\n') || str[1] != 0)
				ttyputmsg("%s", str);
			MLDisownString(db_mathematicalink, str);
			continue;
		}
		if (pkt == MESSAGEPKT)
		{
			MLNewPacket(db_mathematicalink);
			continue;
		}
		if (pkt == INPUTNAMEPKT || pkt == OUTPUTNAMEPKT)
		{
			MLNewPacket(db_mathematicalink);
			continue;
		}
		switch (pkt)
		{
			case EVALUATEPKT:   ttyputmsg("IGNORING: Evaluate Packet");      break;
			case ENTERTEXTPKT:  ttyputmsg("IGNORING: Enter Text Packet");    break;
			case ENTEREXPRPKT:  ttyputmsg("IGNORING: Enter Expr Packet");    break;
			case RETURNEXPRPKT: ttyputmsg("IGNORING: Return Expr Packet");   break;
			case DISPLAYPKT:    ttyputmsg("IGNORING: Display Packet");       break;
			case DISPLAYENDPKT: ttyputmsg("IGNORING: Display End Packet");   break;
			case INPUTPKT:      ttyputmsg("IGNORING: Input Packet");         break;
			case INPUTSTRPKT:   ttyputmsg("IGNORING: Input Start Packet");   break;
			case MENUPKT:       ttyputmsg("IGNORING: Menu Packet");          break;
			case SYNTAXPKT:     ttyputmsg("IGNORING: Syntax Packet");        break;
			case SUSPENDPKT:    ttyputmsg("IGNORING: Suspend Packet");       break;
			case RESUMEPKT:     ttyputmsg("IGNORING: Resume Packet");        break;
			case BEGINDLGPKT:   ttyputmsg("IGNORING: Begin Dlg Packet");     break;
			case ENDDLGPKT:     ttyputmsg("IGNORING: End Dlg Packet");       break;
		}
		MLNewPacket(db_mathematicalink);
	}
}

/*
 * routine to get the next value from Mathematica into the infinite string
 */
INTSML db_mathematicagetstring(INTSML supresslist)
{
	long count;
	REGISTER INTSML islist, i, retval;
	double f;
	char *str, line[50];

	retval = 0;
	switch (MLGetType(db_mathematicalink))
	{
		case MLTKSTR:
			MLGetString(db_mathematicalink, &str);
			(void)addstringtoinfstr(str);
			MLDisownString(db_mathematicalink, str);
			break;
		case MLTKSYM:
			MLGetSymbol(db_mathematicalink, &str);
			if (strcmp(str, "List") == 0) retval = 1;
			if (supresslist == 0 || retval == 0) (void)addstringtoinfstr(str);
			MLDisownSymbol(db_mathematicalink, str);
			break;
		case MLTKINT:
			MLGetLongInteger(db_mathematicalink, &count);
			(void)sprintf(line, "%ld", count);
			(void)addstringtoinfstr(line);
			break;
		case MLTKREAL:
			MLGetReal(db_mathematicalink, &f);
			(void)sprintf(line, "%f", f);
			(void)addstringtoinfstr(line);
			break;
		case MLTKFUNC:
			MLGetArgCount(db_mathematicalink, &count);
			islist = db_mathematicagetstring(1);
			if (islist != 0) (void)addtoinfstr('{'); else (void)addtoinfstr('[');
			for(i=0; i<count; i++)
			{
				if (i > 0) (void)addstringtoinfstr(", ");
				(void)db_mathematicagetstring(0);
			}
			if (islist != 0) (void)addtoinfstr('}'); else (void)addtoinfstr(']');
			break;
	}
	return(retval);
}

void db_mathematicawritetype(INTBIG addr, INTBIG type)
{
	switch (type&VTYPE)
	{
		case VINTEGER:
		case VADDRESS:
		case VCHAR:
		case VFRACT:
		case VSHORT:	   MLPutInteger(db_mathematicalink, addr);                return;
		case VSTRING:      MLPutString(db_mathematicalink, (char *)addr);         return;
		case VFLOAT:
		case VDOUBLE:	   MLPutFloat(db_mathematicalink, castfloat(addr));       return;
		case VNODEINST:    MLPutFunction(db_mathematicalink, "NodeInst", 1);      break;
		case VNODEPROTO:   MLPutFunction(db_mathematicalink, "NodeProto", 1);     break;
		case VPORTARCINST: MLPutFunction(db_mathematicalink, "PortArcInst", 1);   break;
		case VPORTEXPINST: MLPutFunction(db_mathematicalink, "PortExpInst", 1);   break;
		case VPORTPROTO:   MLPutFunction(db_mathematicalink, "PortProto", 1);     break;
		case VARCINST:     MLPutFunction(db_mathematicalink, "ArcInst", 1);       break;
		case VARCPROTO:    MLPutFunction(db_mathematicalink, "ArcProto", 1);      break;
		case VGEOM:        MLPutFunction(db_mathematicalink, "Geom", 1);          break;
		case VLIBRARY:     MLPutFunction(db_mathematicalink, "Library", 1);       break;
		case VTECHNOLOGY:  MLPutFunction(db_mathematicalink, "Technology", 1);    break;
		case VAID:         MLPutFunction(db_mathematicalink, "Aid", 1);           break;
		case VRTNODE:      MLPutFunction(db_mathematicalink, "RTNode", 1);        break;
		case VNETWORK:     MLPutFunction(db_mathematicalink, "Network", 1);       break;
		case VCELL:        MLPutFunction(db_mathematicalink, "Cell", 1);          break;
		case VVIEW:        MLPutFunction(db_mathematicalink, "View", 1);          break;
		case VWINDOWPART:  MLPutFunction(db_mathematicalink, "Window", 1);        break;
		case VGRAPHICS:    MLPutFunction(db_mathematicalink, "Graphics", 1);      break;
		case VCONSTRAINT:  MLPutFunction(db_mathematicalink, "Constraint", 1);    break;
		case VWINDOWFRAME: MLPutFunction(db_mathematicalink, "WindowFrame", 1);   break;
	}
	MLPutInteger(db_mathematicalink, addr);
}

INTSML db_mathematicagettype(INTBIG *addr, INTBIG type)
{
	long c;
	INTBIG obj;

	switch (type)
	{
		case VNODEINST:    if (MLCheckFunction(db_mathematicalink, "NodeInst", &c) == 0) return(0);      break;
		case VNODEPROTO:   if (MLCheckFunction(db_mathematicalink, "NodeProto", &c) == 0) return(0);     break;
		case VPORTARCINST: if (MLCheckFunction(db_mathematicalink, "PortArcInst", &c) == 0) return(0);   break;
		case VPORTEXPINST: if (MLCheckFunction(db_mathematicalink, "PortExpInst", &c) == 0) return(0);   break;
		case VPORTPROTO:   if (MLCheckFunction(db_mathematicalink, "PortProto", &c) == 0) return(0);     break;
		case VARCINST:     if (MLCheckFunction(db_mathematicalink, "ArcInst", &c) == 0) return(0);       break;
		case VARCPROTO:    if (MLCheckFunction(db_mathematicalink, "ArcProto", &c) == 0) return(0);      break;
		case VGEOM:        if (MLCheckFunction(db_mathematicalink, "Geom", &c) == 0) return(0);          break;
		case VLIBRARY:     if (MLCheckFunction(db_mathematicalink, "Library", &c) == 0) return(0);       break;
		case VTECHNOLOGY:  if (MLCheckFunction(db_mathematicalink, "Technology", &c) == 0) return(0);    break;
		case VAID:         if (MLCheckFunction(db_mathematicalink, "Aid", &c) == 0) return(0);           break;
		case VRTNODE:      if (MLCheckFunction(db_mathematicalink, "RTNode", &c) == 0) return(0);        break;
		case VNETWORK:     if (MLCheckFunction(db_mathematicalink, "Network", &c) == 0) return(0);       break;
		case VCELL:        if (MLCheckFunction(db_mathematicalink, "Cell", &c) == 0) return(0);          break;
		case VVIEW:        if (MLCheckFunction(db_mathematicalink, "View", &c) == 0) return(0);          break;
		case VWINDOWPART:  if (MLCheckFunction(db_mathematicalink, "Window", &c) == 0) return(0);        break;
		case VGRAPHICS:    if (MLCheckFunction(db_mathematicalink, "Graphics", &c) == 0) return(0);      break;
		case VCONSTRAINT:  if (MLCheckFunction(db_mathematicalink, "Constraint", &c) == 0) return(0);    break;
		case VWINDOWFRAME: if (MLCheckFunction(db_mathematicalink, "WindowFrame", &c) == 0) return(0);   break;
	}
	if (c != 1) return(0);
	if (MLGetInteger(db_mathematicalink, (int *)addr) == 0) return(0);
	return(1);
}

/*
 * routine to get the function and return it as a "type" and "addr".
 * returns nonzero on error.
 */
INTSML db_mathematicagetaddrtype(INTBIG *type, INTBIG *addr)
{
	INTBIG mtype, *mmint;
	REGISTER INTSML i, floc;
	long c;
	float f, *mmflo;
	char *str;

	mtype = MLGetType(db_mathematicalink);
	switch (mtype)
	{
		case MLTKFUNC:
			if (MLGetFunction(db_mathematicalink, &str, &c) == 0) break;

			/* handle arrays of integers or reals */
			if (strcmp(str, "List") == 0)
			{
				if (c == 0)
				{
					*type = VINTEGER;
					*addr = 0;
					return(0);
				}
				mmint = (INTBIG *)emalloc(c * SIZEOFINTBIG, el_tempcluster);
				if (mmint == 0) return(1);
				mmflo = (float *)emalloc(c * (sizeof (float)), el_tempcluster);
				if (mmflo == 0) return(1);
				floc = 0;
				for(i=0; i<c; i++)
				{
					mtype = MLGetType(db_mathematicalink);
					if (mtype == MLTKINT)
					{
						if (MLGetInteger(db_mathematicalink, (int *)&mmint[i]) == 0) return(1);
						mmflo[i] = mmint[i];
					} else if (mtype == MLTKREAL)
					{
						if (MLGetFloat(db_mathematicalink, &mmflo[i]) == 0) return(1);
						floc++;
					} else
					{
						efree((char *)mmint);
						efree((char *)mmflo);
						return(1);
					}
				}
				if (floc != 0)
				{
					*type = VFLOAT | VISARRAY | (c << VLENGTHSH);
					*addr = (INTBIG)mmflo;
					efree((char *)mmint);
				} else
				{
					*type = VINTEGER | VISARRAY | (c << VLENGTHSH);
					*addr = (INTBIG)mmint;
					efree((char *)mmflo);
				}
				return(0);
			}

			/* must be a function that refers to an Electric object */
			*type = VUNKNOWN;
			if (c == 1)
			{
				if (MLGetInteger(db_mathematicalink, (int *)addr) != 0)
				{
					if (strcmp(str, "NodeInst") == 0)    *type = VNODEINST; else
					if (strcmp(str, "NodeProto") == 0)   *type = VNODEPROTO; else
					if (strcmp(str, "PortArcInst") == 0) *type = VPORTARCINST; else
					if (strcmp(str, "PortExpInst") == 0) *type = VPORTEXPINST; else
					if (strcmp(str, "PortProto") == 0)   *type = VPORTPROTO; else
					if (strcmp(str, "ArcInst") == 0)     *type = VARCINST; else
					if (strcmp(str, "ArcProto") == 0)    *type = VARCPROTO; else
					if (strcmp(str, "Geom") == 0)        *type = VGEOM; else
					if (strcmp(str, "Library") == 0)     *type = VLIBRARY; else
					if (strcmp(str, "Technology") == 0)  *type = VTECHNOLOGY; else
					if (strcmp(str, "Aid") == 0)         *type = VAID; else
					if (strcmp(str, "RTNode") == 0)      *type = VRTNODE; else
					if (strcmp(str, "Network") == 0)     *type = VNETWORK; else
					if (strcmp(str, "Cell") == 0)        *type = VCELL; else
					if (strcmp(str, "View") == 0)        *type = VVIEW; else
					if (strcmp(str, "Window") == 0)      *type = VWINDOWPART; else
					if (strcmp(str, "Graphics") == 0)    *type = VGRAPHICS; else
					if (strcmp(str, "Constraint") == 0)  *type = VCONSTRAINT; else
					if (strcmp(str, "WindowFrame") == 0) *type = VWINDOWFRAME;
				}
			}
			MLDisownString(db_mathematicalink, str);
			if (*type == VUNKNOWN) break;
			return(0);

		case MLTKSTR:
			if (MLGetString(db_mathematicalink, &str) == 0) break;
			*type = VSTRING;
			(void)initinfstr();
			(void)addstringtoinfstr(str);
			*addr = (INTBIG)returninfstr();
			MLDisownString(db_mathematicalink, str);
			return(0);

		case MLTKSYM:
			if (MLGetSymbol(db_mathematicalink, &str) == 0) break;
			*type = VSTRING;
			(void)initinfstr();
			(void)addstringtoinfstr(str);
			*addr = (INTBIG)returninfstr();
			MLDisownString(db_mathematicalink, str);
			return(0);

		case MLTKINT:
			if (MLGetInteger(db_mathematicalink, (int *)addr) == 0) break;
			*type = VINTEGER;
			return(0);

		case MLTKREAL:
			if (MLGetFloat(db_mathematicalink, &f) == 0) break;
			*type = VFLOAT;
			*addr = castint(f);
			return(0);
	}
	return(1);
}

/*
 * routine to make sure Mathematica is initialized and return nonzero on error
 */
INTSML db_mathematicainit(void)
{
	char *argv[4], *path;
	REGISTER INTSML i;

	if (db_mathematicalink != NULL) return(0);
	ttyputmsg("Initializing Mathematica...");
	if (MLInitialize(NULL) == 0) return(1);
	(void)initinfstr();
	(void)addstringtoinfstr("'");
	(void)addstringtoinfstr(el_libdir);
	(void)addstringtoinfstr("Mathematica Kernel");
	(void)addstringtoinfstr("'");
	path = returninfstr();
	argv[0] = "Kernel";
	argv[1] = "-linkname";
	argv[2] = path;
	argv[3] = "\0";
	db_mathematicalink = MLOpen(3, argv);
	if (db_mathematicalink == NULL)
	{
		ttyputerr("Cannot connect to Mathematica");
		return(1);
	}

	/* force the connection */
	if (MLConnect(db_mathematicalink) == 0) return(1);

/*    MLSetMessageHandler(db_mathematicalink, db_mathematicadefaulthandler); */

	/* prepare Mathematica to Install the template functions */
	MLPutFunction(db_mathematicalink, "ToExpression", 1);
	MLPutString(db_mathematicalink, "Install[$ParentLink]");
	MLEndPacket(db_mathematicalink);

	/* send the constants */
	MLPutFunction(db_mathematicalink, "ToExpression", 1);
	MLPutString(db_mathematicalink, "Displayable := 64");
	MLEndPacket(db_mathematicalink);

	/* install the Electric functions */
	ttyputmsg("Installing Electric interface...");
	for(i=0; db_mathematicatraps[i].f_nargs >= 0; i++)
	{
		MLPutFunction(db_mathematicalink, "DefineExternal", (long)3);
		MLPutString(db_mathematicalink, db_mathematicatraps[i].prototype);
		MLPutString(db_mathematicalink, db_mathematicatraps[i].args);
		MLPutInteger(db_mathematicalink, i);
	}
	db_mathematicatrapcount = i;
	MLPutSymbol(db_mathematicalink, "End");
	MLFlush(db_mathematicalink);

	/* now gobble up the result of the Installation */
	db_mathematicaprocesspackets(0);
	MLNewPacket(db_mathematicalink);

	return(0);
}

void db_mathematicaHandleCallPacket(void)
{
	long len;
	int n, res;
	struct func *funcp;

	res = 0;
	if (MLGetInteger(db_mathematicalink, &n) == 0) goto L0;
	if (n < 0 ||  n >= db_mathematicatrapcount) goto L0;
	funcp = &db_mathematicatraps[n];

	if (funcp->f_nargs >= 0)
	{
		if (MLCheckFunction(db_mathematicalink, "List", &len) == 0) goto L0;
		if (len != funcp->f_nargs) goto L0;
	}
	res = (*funcp->f_func)();

L0:
	if (res == 0)
	{
		MLClearError(db_mathematicalink);
		MLPutSymbol(db_mathematicalink, "$Failed");
	}
	MLEndPacket(db_mathematicalink);
	MLNewPacket(db_mathematicalink);
}

void db_mathematicadefaulthandler(MLINK mlp, unsigned long message, unsigned long n)
{
	switch (message)
	{
		case MLTerminateMessage:
			db_mathematicadone = 1;
			ttyputmsg("MATHEMATICA TERMINATION");
			break;
		case MLInterruptMessage:
			db_mathematicaabort = 1;
			ttyputmsg("MATHEMATICA INTERRUPT");
			break;
		case MLAbortMessage:
			db_mathematicaabort = 1;
			ttyputmsg("MATHEMATICA ABORT");
			break;
	}
}

#endif

#endif  /* LANGMM - at top */
