// This file is part of tdir, the Taler Directory implementation.
// Copyright (C) 2025 Martin Schanzenbach
//
// Taldir is free software: you can redistribute it and/or modify it
// under the terms of the GNU Affero General Public License as published
// by the Free Software Foundation, either version 3 of the License,
// or (at your option) any later version.
//
// Taldir is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// SPDX-License-Identifier: AGPL3.0-or-later

package taldir

import (
	"fmt"
	"html/template"
	"os/exec"
	"regexp"
)

type CommandValidator struct {

	// Name
	name string

	// Config
	config *TaldirConfig

	// Validator alias regex
	validAliasRegex string

	// The command to call for validation
	command string

	// registration/lookup page
	landingPageTpl *template.Template
}

func (t CommandValidator) LandingPageTpl() *template.Template {
	return t.landingPageTpl
}

func (t CommandValidator) Type() ValidatorType {
	return ValidatorTypeCommand
}

func (t CommandValidator) Name() string {
	return t.name
}

func (t CommandValidator) ChallengeFee() string {
	return t.config.Ini.Section("taldir-validator-" + t.name).Key("challenge_fee").MustString("KUDOS:0")
}

func (t CommandValidator) IsAliasValid(alias string) (err error) {
	if t.validAliasRegex != "" {
		matched, _ := regexp.MatchString(t.validAliasRegex, alias)
		if !matched {
			return fmt.Errorf("alias `%s' invalid", alias) // TODO i18n
		}
	}
	return
}

func (t CommandValidator) RegistrationStart(topic string, link string, message string, alias string, challenge string) (string, error) {
	path, err := exec.LookPath(t.command)
	if err != nil {
		return "", err
	}
	out, err := exec.Command(path, alias, challenge, topic, message).Output()
	// FIXME logger
	fmt.Printf("Executing `%s %s %s %s %s`\n", path, alias, challenge, topic, message)
	if err != nil {
		fmt.Printf("%s, %v\n", out, err)
		return "", err
	}
	return "", nil
}

func makeCommandValidator(cfg *TaldirConfig, name string, landingPageTpl *template.Template) CommandValidator {
	sec := cfg.Ini.Section("taldir-validator-" + name)
	return CommandValidator{
		name:            name,
		config:          cfg,
		landingPageTpl:  landingPageTpl,
		validAliasRegex: sec.Key("valid_alias_regex").MustString(""),
		command:         sec.Key("command").MustString(""),
	}
}
