/* -*-C++-*-
*******************************************************************************
*
* File:         check_dups
* Description:  check for duplicate pages
*
*******************************************************************************
*/

/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <unistd.h>
#include <stdlib.h>
#ifdef HAVE_SYS_MMAN_H
#include <sys/mman.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>


// return true if pages i and j are equal

inline bool
compare_pages (const char *base, int i, int j, int page_size)
{
  if (*(long *)&base[i * page_size] != *(long *)&base[j * page_size])
    return false;

  if (memcmp (&base[i * page_size],
	      &base[j * page_size],
	      page_size) == 0)
    return true;
  else
    return false;
}

int
check_file_for_duplicate_pages (const char *base, unsigned long nbytes)
{
#ifdef HAVE_MMAP
  int	page_size = getpagesize ();
#else
#warning default pagesize: 4KiB
  int	page_size = 4*1024;
#endif
  long 	npages = nbytes / page_size;
  long	i, j;
  int	nmatches = 0;

  for (i = 0; i < npages - 1; i++){
    for (j = i + 1; j < npages; j++){
      bool same = compare_pages (base, i, j, page_size);
      if (same){
	nmatches++;
	fprintf (stdout, "%6ld %6ld %6ld\n", i, j, j - i);
      }
    }
  }
  return nmatches;
}


void
usage (const char *name)
{
  fprintf (stderr, "usage: %s <filename>\n", name);
  exit (1);
}

int 
main (int argc, char **argv)
{
  int		fd;
  char		*base;
  struct stat	stat;
  off_t		file_size;

  if (argc != 2)
    usage (argv[0]);

  fd = open (argv[1], O_RDONLY);
  if (fd < 0){
    perror (argv[1]);
    exit (1);
  }

  if (fstat (fd, &stat) < 0){
    perror ("stat");
    exit (1);
  }

  file_size = stat.st_size;	// file size in bytes

#ifdef HAVE_MMAP
  if ((base = (char *) mmap (0, file_size, PROT_READ, MAP_SHARED, fd, 0)) == MAP_FAILED){
    perror ("mmap");
    exit (1);
  }
#else
    fprintf (stderr, "No mmap support on this system\n");
    exit (1);
#endif

  int nmatches = check_file_for_duplicate_pages (base, file_size);

#ifdef HAVE_MMAP
  munmap (base, file_size);
#endif
  close (fd);
  
  if (nmatches > 0){
    fprintf (stderr, "FAILED: %d pages are identical\n", nmatches);
    exit (1);
  }

  exit (0);
}
  
