/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VrCOMPLEXCASCADEFILTER_H_
#define _VrCOMPLEXCASCADEFILTER_H_

#include <VrDecimatingSigProc.h>

/*  Cascade filter definition:

    VrComplexCascadefilter(N, Num of taps[], decimation factor, center frequency[], gain[])

    N = number of filters
    decimation factor => set to one unless some integer >1 specified
    cutoff (Hz) = 0.0  => LPF using Hamming window, o/w LPF transformed to have higher cutoff freq
    center_freq (Hz) => used to specify a composite frequency-shifting filter (i.e. channel filter)
    gain = the gain
*/

// ********************************************************

template<class iType> 
class VrComplexCascadefilter : public VrDecimatingSigProc<iType,VrComplex> {
protected:
  int *numTaps, num_ch;
  VrComplex** taps;
  VrComplex *phase_correction, *phase_corr_incr;
  long time;
  float *center_freq, *gain, *cut_off;
  void buildFilter_complex(int);
public: 

  /*
  virtual void skip(int n) { 
    if(num_ch>1) {
      //until skip works right with multiple outputBuffers, we just call work
      work(n);
    } else
      VrDecimatingSigProc<iType,VrComplex>::skip(n);
  }
  */

  virtual void work(int n);
  virtual void initialize();
  int setCenter_Freq(int, float);
  int setCenter_Freq(float);
  int setNumber_Taps(int,int);
  int setNumber_Taps(int);
  VrComplexCascadefilter(int n, int d, const int t[], const float f[], 
			 const float co[], const float g[]);
  VrComplexCascadefilter(int d, int t,float f, float co, float g);
  ~VrComplexCascadefilter();
};

template<class iType> void
VrComplexCascadefilter<iType>::work(int n)
{
  VrComplex result = 0;
  int ch_num = 0;

  for (int i=0;i<n;i++,incReadPtr(decimation)) {
    result = 0;

    for (ch_num =0; ch_num<num_ch; ch_num++){
      //make input pointer local
      iType *inputArray = inputReadPtr(-numTaps[ch_num]+1);

      VrComplex *taps_tmp = taps[ch_num];
      for (int j=0; j < numTaps[ch_num]; j++)
	result += taps_tmp[j] * inputArray[j];     
   
      // Perform phase correction (non-trivial only for freq-xlating filter)
      if (center_freq[ch_num] != 0.0) {
	phase_correction[ch_num] *= phase_corr_incr[ch_num];
	result *= phase_correction[ch_num];
      }
      outputWriteN(ch_num,result);
    }
  }
  
}

template<class iType> void
VrComplexCascadefilter<iType>::buildFilter_complex(int ch){
  int inSampFreq;
  int index;
  float N = numTaps[ch];
  float M = N-1; /* filter Order */

  inSampFreq = getInputSamplingFrequencyN(0); 
  time = 0;
  

    // Build composite Complex Filter => includes freq-shifting part, hamming and cutoff 

    float arg=0.0, hamm_coef, cutoff_coef, prototype_coef;
    for ( index=0 ; index < numTaps[ch] ; index++) {

      hamm_coef = 0.54-0.46*cos(2*M_PI*index/(M));

      arg = 2*M_PI*cut_off[ch]/(float)inSampFreq;
      if (index-(M/2) != 0){     
	cutoff_coef = sin(arg*(index-(M/2)))/M_PI/(index-(M/2));
      } else {
	cutoff_coef = arg/M_PI;
      }
	
      arg = 2*M_PI*center_freq[ch] / (float)inSampFreq;
      prototype_coef = hamm_coef * cutoff_coef;

      taps[ch][index] = VrComplex(gain[ch]*cos(arg*index)*prototype_coef,
				    gain[ch]*(-1)*sin(arg*index)*prototype_coef);
    }
    phase_corr_incr[ch] = VrComplex(cos(arg*(float)decimation),
				(-1)*sin(arg*(float)decimation));

}


template<class iType> 
VrComplexCascadefilter<iType>::VrComplexCascadefilter(int n, int dec, const int t[], const float freq[], const float co_freq[], const float g[])
  :VrDecimatingSigProc<iType,VrComplex>(n,dec), num_ch(n)
{

  numTaps=new int[num_ch];
  phase_correction = new VrComplex[num_ch];
  phase_corr_incr = new VrComplex[num_ch];
  center_freq = new float[num_ch];
  cut_off= new float[num_ch];
  gain = new float[num_ch];
  taps = new (VrComplex *[num_ch]);

  for (int i=0; i<num_ch; i++){
    numTaps[i] = t[i];
    phase_correction[i] = VrComplex(1,0);
    phase_corr_incr[i] = VrComplex(1,0);
    center_freq[i] = freq[i];
    cut_off[i] = co_freq[i];
    gain[i] = g[i];

  }

}

template<class iType> 
VrComplexCascadefilter<iType>::VrComplexCascadefilter(int dec,int t,float freq, float co_freq, float g)
  :VrDecimatingSigProc<iType,VrComplex>(1,dec), num_ch(1)
{

  numTaps=new int[num_ch];
  phase_correction = new VrComplex[num_ch];
  phase_corr_incr = new VrComplex[num_ch];
  center_freq = new float[num_ch];
  cut_off= new float[num_ch];
  gain = new float[num_ch];
  taps = new (VrComplex *[num_ch]);

  numTaps[0] = t;
  phase_correction[0] = VrComplex(1,0);
  phase_corr_incr[0] = VrComplex(1,0);
  center_freq[0] = freq;
  cut_off[0] = co_freq;
  gain[0] = g;

}

template<class iType> 
void VrComplexCascadefilter<iType>::initialize()
{
  for (int i=0; i<num_ch; i++){ 
    taps[i]=new VrComplex[numTaps[i]];
    buildFilter_complex(i);
  }

  //Set history
  int max_numTaps = 0;
  for (int i=0; i<num_ch; i++){
    if (numTaps[i] > max_numTaps) max_numTaps = numTaps[i];
  }
  setHistory(max_numTaps);
}

template<class iType> 
int VrComplexCascadefilter<iType>::setCenter_Freq(int ch, float cf)
{
  center_freq[ch] = cf;
  buildFilter_complex(ch);
  return 1;
}

template<class iType> 
int VrComplexCascadefilter<iType>::setNumber_Taps(int ch, int numT)
{
  numTaps[ch] = numT;
  delete taps[ch];
  taps[ch]=new VrComplex[numTaps[ch]];

  //set history
  int max_numTaps = 0;
  for (int i=0; i<num_ch; i++){
    if (numTaps[i] > max_numTaps) max_numTaps = numTaps[i];
  }
  setHistory(max_numTaps);

  buildFilter_complex(ch);
  return 1;
}

template<class iType> 
VrComplexCascadefilter<iType>::~VrComplexCascadefilter()
{
  
  for (int i=0; i<num_ch; i++){ 
    delete taps[i];
  }
  delete numTaps;
  delete [] phase_correction;
  delete [] phase_corr_incr;
  delete center_freq;
  delete cut_off;
  delete gain;
  delete taps;

}



#endif



















