/* -*- c++ -*- */
/*
 * Copyright 2006 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef INCLUDED_GR_UNPACKED_TO_PACKED_BB_H
#define INCLUDED_GR_UNPACKED_TO_PACKED_BB_H

#include <gr_block.h>
#include <gr_endianness.h>

class gr_unpacked_to_packed_bb;
typedef boost::shared_ptr<gr_unpacked_to_packed_bb> gr_unpacked_to_packed_bb_sptr;

gr_unpacked_to_packed_bb_sptr 
gr_make_unpacked_to_packed_bb (unsigned int bits_per_chunk, gr_endianness_t endianness);

/*!
 * \brief Convert a stream of unpacked bytes into a stream of packed bytes.
 * \ingroup block
 *
 * input: stream of bytes; output: stream of bytes
 *
 * This is the inverse of gr_packed_to_unpacked_bb.
 *
 * The low \p bits_per_chunk bits are extracted from each input byte.
 * These bits are then packed densely into the output bytes, such that
 * all 8-bits of the output bytes are filled with valid input bits.
 * The right thing is done if bits_per_chunk is not a power of two.
 *
 * The combination of gr_packed_to_unpacked_bb followed by
 * gr_chunks_to_symbols_bf or gr_chunks_to_symbols_bc handles the
 * general case of mapping from a stream of bytes into arbitrary float
 * or complex symbols.
 *
 * \sa gr_packed_to_unpacked_bb, gr_unpacked_to_packed_bb,
 * \sa gr_chunks_to_symbols_bf, gr_chunks_to_symbols_bc.
 */
class gr_unpacked_to_packed_bb : public gr_block
{
  friend gr_unpacked_to_packed_bb_sptr 
  gr_make_unpacked_to_packed_bb (unsigned int bits_per_chunk, gr_endianness_t endianness);

  gr_unpacked_to_packed_bb (unsigned int bits_per_chunk, gr_endianness_t endianness);

  unsigned int    d_bits_per_chunk;
  gr_endianness_t d_endianness;
  unsigned int    d_index;

 public:
  void forecast(int noutput_items, gr_vector_int &ninput_items_required);
  int general_work (int noutput_items,
		    gr_vector_int &ninput_items,
		    gr_vector_const_void_star &input_items,
		    gr_vector_void_star &output_items);
};

#endif /* INCLUDED_GR_UNPACKED_TO_PACKED_BB_H */
