/*

      This file is part of the Goptical Core library.
  
      The Goptical library is free software; you can redistribute it
      and/or modify it under the terms of the GNU General Public
      License as published by the Free Software Foundation; either
      version 3 of the License, or (at your option) any later version.
  
      The Goptical library is distributed in the hope that it will be
      useful, but WITHOUT ANY WARRANTY; without even the implied
      warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
      See the GNU General Public License for more details.
  
      You should have received a copy of the GNU General Public
      License along with the Goptical library; if not, write to the
      Free Software Foundation, Inc., 59 Temple Place, Suite 330,
      Boston, MA 02111-1307 USA
  
      Copyright (C) 2010-2011 Free Software Foundation, Inc
      Author: Alexandre Becoulet

*/


#ifndef GOPTICAL_TRACEDRAY_HH_
#define GOPTICAL_TRACEDRAY_HH_

#include "Goptical/common.hh"

#include "Goptical/Light/ray.hh"
#include "Goptical/Math/vector.hh"

namespace _Goptical {

  namespace Trace {

    /**
       @short Propagated light ray class
       @header Goptical/Trace/Ray
       @module {Core}

       This class is used to describe a @ref Light::Ray with all
       tracing and propagation informations attached.
     */
    class Ray : public Light::Ray
    {
    public:

      /** Create a propagated light ray */
      inline Ray();

      /** Create a propagated light ray */
      inline Ray(const Light::Ray &r);

      GOPTICAL_ACCESSORS(const Sys::Element *, creator, "element which generated this ray.");

      GOPTICAL_ACCESSORS(const Material::Base *, material, "material ray is propagated in.");

      GOPTICAL_ACCESSORS(double, intercept_intensity, "ray intensity at interception point.");

      GOPTICAL_ACCESSORS(double, len, "light ray length.");

      /** Define a new child generated ray */
      inline void add_generated(Trace::Ray *r);

      /** Set light ray interception point and element */
      inline void set_intercept(const Sys::Element &e, const Math::Vector3 &point);
      /** Get light ray interception point */
      inline const Math::Vector3 & get_intercept_point() const;
      /** Get light ray interception element */
      inline Sys::Element & get_intercept_element() const;

      /** Get ray which generated this one */
      inline Ray * get_parent() const;
      /** Get first ray generated from this one */
      inline Ray * get_first_child() const;
      /** Get next sibling ray generated by same parent */
      inline Ray * get_next_child() const;

      /** Return true if ray is not intercepted */
      inline bool is_lost() const;

      /** Get global position */
      inline Math::Vector3 get_position() const;
      /** Get global direction */
      inline Math::Vector3 get_direction() const;

      /** Get position relative to given element */
      inline Math::Vector3 get_position(const Sys::Element &e) const;
      /** Get direction relative to given element */
      inline Math::Vector3 get_direction(const Sys::Element &e) const;

    private:
      Ray(const Ray &);
      const Ray & operator=(const Ray &r);

      Math::Vector3             _point;         // ray intersection point (intersect surface local)
      double                    _intercept_intensity;   // intersection point intensity
      double                    _len;           // ray length
      const Sys::Element        *_creator;      // element which generated this ray
      const Material::Base  *_material;     // material
      Sys::Element              *_i_element;    // intersect element
      Ray                       *_parent;       // ray which generated this one
      Ray                       *_child;        // pointer to generated ray
      Ray                       *_next;         // pointer to sibling generated ray
      bool                      _lost;          // does the ray intersect with an element ?
    };

  }

}

#endif

