# coding=utf-8

#    Copyright (C) 2008-2011  Luis Falcon

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

import time
from mx import DateTime
from datetime import datetime
from trytond.model import ModelView, ModelSQL, fields
from trytond.transaction import Transaction


class PatientData(ModelSQL, ModelView):
    'Add Lab test information to the Patient object'
    _name = 'medical.patient'
    _description = __doc__

    # TODO: Trytonize
    def name_get(self, cr, uid, ids, context={}):
        if not len(ids):
            return []
        rec_name = 'name'
        res = [(r['id'], r[rec_name][1]) for r in self.read(cr, uid, ids, [rec_name], context)]
        return res

    # TODO: Trytonize
    def name_search(self, cr, user, name='', args=None, operator='ilike',
                    context=None, limit=80):
        if not args:
            args = []
        if not context:
            context = {}
        if name:
            ids = self.search(cr, user, [('patient_id', '=', name)] + args,
                              limit=limit, context=context)
            if not len(ids):
                ids += self.search(cr, user, [('name', operator, name)] + args,
                                   limit=limit, context=context)
        else:
            ids = self.search(cr, user, args, limit=limit, context=context)
        result = self.name_get(cr, user, ids, context)
        return result

    lab_test_ids = fields.One2Many('medical.patient.lab.test', 'patient_id',
        'Lab Tests Required')

PatientData()


class TestType(ModelSQL, ModelView):
    'Type of Lab test'
    _name = 'medical.test_type'
    _description = __doc__

    name = fields.Char('Test', size=128,
        help="Test type, eg X-Ray, hemogram,biopsy...")
    code = fields.Char('Code', size=128,
        help="Short name - code for the test")
    info = fields.Text('Description')
    product_id = fields.Many2One('product.product', 'Service', required=True)
    critearea = fields.One2Many('medical_test.critearea', 'test_type_id',
        'Test Cases')

    def __init__(self):
        super(TestType, self).__init__()
        self._sql_constraints = [
            ('code_uniq', 'unique(name)', 'The Lab Test code must be unique'),
        ]

TestType()


class Lab(ModelSQL, ModelView):
    'Lab Test'
    _name = 'medical.lab'
    _description = __doc__

    name = fields.Char('ID', size=128, help="Lab result ID")
    test = fields.Many2One('medical.test_type', 'Test type',
        help="Lab test type")
    patient = fields.Many2One('medical.patient', 'Patient', help="Patient ID")
    pathologist = fields.Many2One('medical.physician', 'Pathologist',
        help="Pathologist")
    requestor = fields.Many2One('medical.physician', 'Physician',
        help="Doctor who requested the test")
    results = fields.Text('Results')
    diagnosis = fields.Text('Diagnosis')
    critearea = fields.One2Many('medical_test.critearea', 'medical_lab_id',
        'Test Cases')
    date_requested = fields.DateTime('Date requested')
    date_analysis = fields.DateTime('Date of the Analysis')

    def __init__(self):
        super(Lab, self).__init__()
        self._sql_constraints += [
            ('id_uniq', 'unique (name)', 'The test ID code must be unique'),
        ]

    def default_date_requested(self):
        return datetime.now()

    def default_analysis(self):
        return datetime.now()

    def default_name(self):
        sequence_obj = self.pool.get('ir.sequence')
        return sequence_obj.get('medical.lab')

Lab()


class MedicalLabTestUnits(ModelSQL, ModelView):
    'Lab Test Units'
    _name = 'medical.lab.test.units'
    _description = __doc__

    name = fields.Char('Unit', size=25)
    code = fields.Char('Code', size=25)

    def __init__(self):
        super(MedicalLabTestUnits, self).__init__()
        self._sql_constraints = [
            ('name_uniq', 'unique(name)', 'The Unit name must be unique'),
        ]

MedicalLabTestUnits()


class MedicalTestCritearea(ModelSQL, ModelView):
    'Lab Test Critearea'
    _name = 'medical_test.critearea'
    _description = __doc__

    name = fields.Char('Test', size=64)
    result = fields.Text('Result')
    normal_range = fields.Text('Normal Range')
    units = fields.Many2One('medical.lab.test.units', 'Units')
    test_type_id = fields.Many2One('medical.test_type', 'Test type')
    medical_lab_id = fields.Many2One('medical.lab', 'Test Cases')
    sequence = fields.Integer('Sequence')

    def __init__(self):
        super(MedicalTestCritearea, self).__init__()
        self._order.insert(0, ('sequence', 'ASC'))

    def default_sequence(self):
        return 1

MedicalTestCritearea()


class MedicalPatientLabTest(ModelSQL, ModelView):
    'Patient Lab Test'
    _name = 'medical.patient.lab.test'
    _description = __doc__

    name = fields.Many2One('medical.test_type', 'Test Type')
    date = fields.DateTime('Date')
    state = fields.Selection([
        ('draft', 'Draft'),
        ('tested', 'Tested'),
        ('cancel', 'Cancel'),
        ], 'State', readonly=True)
    patient_id = fields.Many2One('medical.patient', 'Patient')
    doctor_id = fields.Many2One('medical.physician', 'Doctor',
        help="Doctor who Request the lab test.")

    def default_date(self):
        return datetime.now()

    def default_state(self):
        return 'draft'

    def default_doctor_id(self):
        user_obj = self.pool.get('res.user')
        user = user_obj.browse(Transaction().user)
        uid = int(user.id)

        party_id = self.pool.get('party.party').search([
                ('internal_user', '=', uid)])
        if party_id:
            dr_id = self.pool.get('medical.physician').search([
                    ('name', '=', party_id[0])])
            if dr_id:
                return dr_id[0]
            else:
                raise Exception('There is no physician defined ' \
                                'for current user.')
        else:
            return False

MedicalPatientLabTest()
