########### dns.tcl
# DNS lookup code, using `host'
#
# This file is part of SAUCE, a very picky anti-spam receiver-SMTP.
# SAUCE is Copyright (C) 1997-1999 Ian Jackson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
#
# $Id: dns.tcl,v 1.12 1999/09/09 21:49:06 ian Exp $


########## dns threads
#
# thread_start dns $desc $domain $type
#
# success => $answers {} OK
# permanent failure (domain unknown) => {} $errors NXDOMAIN
# permanent failure (type unknown) => {} $errors NOTYPE
# temporary failure =>X

# state variables:
# chan      channel onto `host' (unset => closed)
# domain    query domain
# type      query type (class is always IN)
# permfail  0 => no permanent failure detected yet; 1 => has been
# errs      error message(s) found so far
# answers   answer value(s) so far
# toid      timeout id

proc rrdata_check.MX {val} {
    if {![regexp -nocase {^([0-9]+)[ \t]+([-.0-9a-z]+)$} $val all pri dom]} {
	error "MX data invalid ($val)"
    } elseif {![domain_ok $dom]} {
	error "MX domain invalid (in $val)"
    }
}

proc rrdata_check.A {val} {
    if {![regexp -nocase {^[0-9]{1,3}(\.[0-9]{1,3}){3}$} $val]} {
	error "A RR data invalid ($val)"
    }
}

proc rrdata_check.TXT {val} {
    if {![regexp -nocase {^\".*\"$} $val]} {
	error "TXT RR data invalid ($val)"
    }
}

proc rrdata_check.PTR {val} {
    if {![domain_ok $val]} { error "PTR RR data invalid ($val)" }
}

proc rrdata_check.CNAME {val} {
}

thread_typedefine dns {domain type} {
    global dns_timeout var_dir
    set dns_tmp $var_dir/dns-tmp
    file delete -force $dns_tmp
    set state(efile) [open $dns_tmp w+]
    file delete $dns_tmp
    set state(chan) [open [concat |host -t $type $domain. 2>@$state(efile)] r]
    chanset_desc $state(chan) "$state(desc) / host -t $type $domain"
    set state(domain) [string tolower $domain]
    set state(type) [string tolower $type]
    set state(permfail) {}
    set state(errs) {}
    set state(answers) {}
    set state(toid) [thread_after dns $id $dns_timeout timeout]
    threadio_gets dns $id $state(chan) readline {}
    return $id
} ERROR-ON-SHUTDOWN {
    catch { after cancel $state(toid) }
    catch { close $state(efile) }
    catch_close_cleardesc state(chan)
}

thread_chainproc dns readline {data} {
    if {![string length $data] && [eof $state(chan)]} {
	seek $state(efile) 0 start
	while {[gets $state(efile) data] >= 0} {
	    debug 1 "$state(desc) / host errors << $data"
	    if { \
 [regexp -nocase \
 {^([-.0-9a-z]+) does not exist.*\(authoritative answer\)$} $data all lhs] \
	    } {
		set lhs [string tolower $lhs]
		if {"$lhs" == "$state(domain)"} {
		    set state(permfail) NXDOMAIN
		}
		append state(errs) " $data"
	    } elseif { \
 [regexp -nocase \
 {^([-.0-9a-z]+) has no ([a-z]+) record.*\(authoritative answer\)$} $data all lhs type] \
		|| \
 [regexp -nocase \
 {^([-.0-9a-z]+) ([a-z]+) record currently not present} $data all lhs type] \
	    } {
 # we hope that this means the record didn't exist.  This is probably true because
 # a recursive query won't produce this if there is a temporary failure.
		set lhs [string tolower $lhs]
		set type [string tolower $type]
		if {"$lhs" == "$state(domain)" && "$type" == "$state(type)"} {
		    set state(permfail) NOTYPE
		}
		append state(errs) " $data"
	    } else {
		append state(errs) " [string trimleft $data {!* 	}]"
	    }
	}
	set errs [string trim $state(errs)]

	set answers $state(answers)
	if {![string length $errs] && [string length $answers]} {
	    close $state(chan)
	    unset state(chan)
	    thread_finish dns $id $answers {} OK
	} else {
	    if {![string length $errs]} {
		set errs "cause unknown"
	    }
	    if {[llength $state(permfail)]} {
		thread_finish dns $id {} $errs $state(permfail)
	    } else {
		thread_error dns $id $errs {}
	    }
	}
	return
    } elseif {[regexp -nocase \
	    "^(\\S+)\[ \\t\]+($state(type)|cname)\[ \\t\]+(\[^ \\t\].*)\$" \
	    $data all lhs type rhs]} {
	set lhs [string tolower $lhs]
	set type [string tolower $type]
	if {"$lhs" == "$state(domain)"} {
	    if {"$type" == "$state(type)"} {
		rrdata_check.[string toupper $type] $rhs
		lappend state(answers) $rhs
	    } elseif {"$type" == "cname"} {
		rrdata_check.CNAME $rhs
		set state(domain) [string tolower $rhs]
	    }
	} else {
	    append state(errs) " expecting records for $state(domain), got $lhs"
	}
    } else {
	append state(errs) " [string trimleft $data {!* 	}]"
    }
    threadio_gets dns $id $state(chan) readline {}
}	

thread_chainproc dns timeout {} {
    append state(errs) " lookup timed out"
    thread_error dns $id [string trim $state(errs)] {}
}

########## dnsptr threads
#
# thread_start dnsptr $desc $ipaddr
#
# success => $ipaddr
# permanent or temporary failure =>X

# state variables:
# ipaddr    address for which PTR is requested
# dnsid     id of DNS query subthread (unset => none)
# remain    list of unchecked returns from PTR in-addr lookup (unset until DNS finishes)
# errs      list of error message(s)

thread_typedefine dnsptr {ipaddr} {
    set state(ipaddr) $ipaddr
    set ptr in-addr.arpa
    foreach octet [split $ipaddr .] {
	set ptr $octet.$ptr
    }
    set state(dnsid) [thread_start dns $state(desc) $ptr PTR]
    thread_join dnsptr $id dns $state(dnsid) dns_rvok dns_rverr
} ERROR-ON-SHUTDOWN {
    catch { thread_cancel $state(dnsid) }
}

thread_chainproc dnsptr dns_rvok {answers errors how} {
    unset state(dnsid)
    if {[llength $answers]} {
	set state(remain) $answers
	set state(errs) {}
	dnsptr_continue
    } else {
	thread_error dnsptr $id $errors {}
    }
}

thread_chainproc dnsptr dns_rverr {emsg} {
    unset state(dnsid)
    thread_error dnsptr $id $emsg {}
}

thread_subproc dnsptr continue {} {
    if {![llength $state(remain)]} {
	thread_error dnsptr $id "$state(ipaddr) -> [join $state(errs) {;... }]" {}
	return
    }
    set remain $state(remain)
    set try [lindex $remain 0]
    set state(remain) [lreplace $remain 0 0]
    set state(dnsid) [thread_start dns $state(desc) $try A]
    thread_join dnsptr $id dns $state(dnsid) dns_fwok dns_fwerr $try
}

thread_chainproc dnsptr dns_fwok {try answers errors how} {
    unset state(dnsid)
    if {![string length $answers]} {
	lappend state(errs) "$try -> $errors"
    } else {
	foreach ans $answers {
	    if {"$ans"=="$state(ipaddr)"} {
		thread_finish dnsptr $id $try
		return
	    }
	}
	lappend state(errs) "$try -> [join $answers {, }]"
    }
    dnsptr_continue
}

thread_chainproc dnsptr dns_fwerr {try emsg} {
    unset state(dnsid)
    lappend state(errs) "$try -> $emsg"
    dnsptr_continue
}
