// Copyright (C) 2010 David Sugar, Tycho Softworks.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "server.h"

#ifdef  HAVE_SYS_INOTIFY_H
#include <sys/inotify.h>
#include <sys/poll.h>
#endif

NAMESPACE_SIPWITCH

#ifdef HAVE_SIGWAIT

signals signals::thread;

signals::signals() :
JoinableThread()
{
    shutdown = started = false;
}

signals::~signals()
{
    if(!shutdown)
        cancel();
}

void signals::cancel(void)
{
    if(started) {
        shutdown = true;
#ifdef  __FreeBSD__
        raise(SIGINT);
#endif
        pthread_kill(tid, SIGALRM);
        join();
    }
}

void signals::run(void)
{
    int signo;
    unsigned period = 900;

    started = true;
    shell::log(DEBUG1, "starting signal handler");

    for(;;) {
        alarm(period);
#ifdef  HAVE_SIGWAIT2
        sigwait(&sigs, &signo);
#else
        signo = sigwait(&sigs);
#endif
        alarm(0);
        if(shutdown)
            return;

        shell::log(DEBUG1, "received signal %d", signo);

        switch(signo) {
        case SIGALRM:
            shell::log(shell::INFO, "system housekeeping");
            registry::cleanup(period);
            events::sync(period);
            break;
        case SIGINT:
        case SIGTERM:
            control::send("down");
            break;
        case SIGUSR1:
            control::send("snapshot");
            break;
        case SIGHUP:
            control::send("reload");
            break;
        default:
            break;
        }
    }
    shell::log(DEBUG1, "stopping signal handler");
}

void signals::service(const char *name)
{
}

void signals::setup(void)
{
    sigemptyset(&thread.sigs);
    sigaddset(&thread.sigs, SIGALRM);
    sigaddset(&thread.sigs, SIGHUP);
    sigaddset(&thread.sigs, SIGINT);
    sigaddset(&thread.sigs, SIGTERM);
    sigaddset(&thread.sigs, SIGUSR1);
    pthread_sigmask(SIG_BLOCK, &thread.sigs, NULL);

    signal(SIGPIPE, SIG_IGN);
}

void signals::start(void)
{
    thread.background();
}

void signals::stop(void)
{
    thread.cancel();
}

#elif defined(WIN32)

static SERVICE_STATUS_HANDLE hStatus = 0;
static SERVICE_STATUS status;

static void WINAPI handler(DWORD sigint)
{
    switch(sigint) {
    case 128:
        // control::request("reload");
        return;
    case 129:
        // control::request("snapshot");
    case SERVICE_CONTROL_SHUTDOWN:
    case SERVICE_CONTROL_STOP:
        status.dwCurrentState = SERVICE_STOP_PENDING;
        status.dwWin32ExitCode = 0;
        status.dwCheckPoint = 0;
        status.dwWaitHint = 6000;
        SetServiceStatus(hStatus, &status);
        // control::request("down");
        break;
    default:
        break;
    }
}

void signals::service(const char *name)
{
    memset(&status, 0, sizeof(SERVICE_STATUS));
    status.dwServiceType = SERVICE_WIN32;
    status.dwCurrentState = SERVICE_START_PENDING;
    status.dwControlsAccepted = SERVICE_ACCEPT_STOP|SERVICE_ACCEPT_SHUTDOWN;
    hStatus = ::RegisterServiceCtrlHandler(name, &handler);
}

void signals::setup(void)
{
}

void signals::start(void)
{
    if(!hStatus)
        return;

    status.dwCurrentState = SERVICE_RUNNING;
    ::SetServiceStatus(hStatus, &status);
}

void signals::stop(void)
{
    if(!hStatus)
        return;

    status.dwCurrentState = SERVICE_STOPPED;
    ::SetServiceStatus(hStatus, &status);
}

#else

void signals::service(const char *name)
{
}

void signals::setup(void)
{
}

void signals::start(void)
{
}

void signals::stop(void)
{
}

#endif

#ifdef  HAVE_SYS_INOTIFY_H

static const char *dirpath;
static fd_t watcher = -1;
static uint32_t dirnode;

notify notify::thread;

notify::notify() : JoinableThread()
{
}

notify::~notify()
{
    notify::stop();
}

void notify::start(void)
{
    dirpath = control::env("users");

    if(!dirpath)
        dirpath = control::env("prefix");

    thread.background();
}

void notify::run(void)
{
    timeout_t timeout = -1;
    unsigned updates = 0;
    struct pollfd pfd;

    shell::log(DEBUG1, "notify watching %s", dirpath);

    watcher = inotify_init();
    dirnode = inotify_add_watch(watcher, dirpath, IN_CLOSE_WRITE|IN_MOVED_TO|IN_MOVED_FROM|IN_DELETE);

    while(watcher != -1) {
        // we want 500 ms of inactivity before actual updating...
        if(updates)
            timeout = 500;
        else
            timeout = 1000;

        pfd.fd = watcher;
        pfd.events = POLLIN | POLLNVAL | POLLERR;
        pfd.revents = 0;
        if(!poll(&pfd, 1, timeout)) {
            if(!updates)
                continue;

            // clear updates and process config on timeout
            control::send("reload");
            updates = 0;
            continue;
        }
        if(pfd.revents & (POLLNVAL|POLLERR))
            break;

        if(pfd.revents & POLLIN) {
            char buffer[512];
            size_t offset = 0;

            size_t len = ::read(watcher, &buffer, sizeof(buffer));
            if(len < sizeof(struct inotify_event))
                continue;

            while(offset < len) {
                struct inotify_event *event = (struct inotify_event *)&buffer[offset];
                if(!event->len)
                    break;

                // only if xml files updated do we care...
                const char *ext = strrchr(event->name, '.');
                if(ext && case_eq(ext, ".xml")) {
                    shell::log(DEBUG2, "%s updated", event->name);
                    ++updates;
                }
                offset += sizeof(struct inotify_event) + event->len;
            }
        }
    }

    shell::log(DEBUG1, "notify terminating");
}

void notify::stop(void)
{
    if(watcher != -1) {
        fd_t fd = watcher;
        watcher = -1;
        ::close(fd);
        thread.join();
    }
}

#else

void notify::start(void)
{
}

void notify::stop(void)
{
}

#endif

END_NAMESPACE

