/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.common

private val BASE32_32B_UPPER_PATTERN = Regex("(KYC:)?([0-9A-Z]{52})")
private val BASE32_32B_PATTERN = Regex("(KYC:)?([a-z0-9A-Z]{52})")
private val CLEAN_PATTERN = Regex(" ?[\\n\\-\\+] ?")

data class TalerIncomingMetadata(val type: TalerIncomingType, val key: EddsaPublicKey)

/** 
 * Extract the reserve public key from an incoming Taler transaction subject
 * 
 * We first try to match an uppercase key then a lowercase key. If none are
 * found we clean the subject and retry.
 **/
fun parseIncomingTxMetadata(subject: String): TalerIncomingMetadata {
    /** 
     * Extract the reserve public key from [subject] using [pattern]
     * 
     * Return null if found none and throw if find too many
     **/
    fun run(subject: String, pattern: Regex): TalerIncomingMetadata? {
        val matches = pattern.findAll(subject).iterator()
        if (!matches.hasNext()) return null
        val match = matches.next()
        if (matches.hasNext()) {
            throw Exception("Found multiple reserve public key")
        }
        val (prefix, key) = match.destructured
        val type = if (prefix == "KYC:") TalerIncomingType.kyc else TalerIncomingType.reserve
        return TalerIncomingMetadata(type, EddsaPublicKey(key))
    }

    // Wire transfer subjects are generally small in size, and not
    // being able to find the encoded reserve public key poses a huge
    // usability problem. So we're ready to work hard to find it.

    // Bank interfaces doesn't always allow a complete encoded key to 
    // be entered on a single line, so users have to split them, which
    // may lead them to add an erroneous space or + or - separator.
    // If we can't find a key, we try to eliminate these common errors 
    // before trying again.

    // Since any sequence of 52 upper and lowercase characters can be a 
    // valid encoded key, deleting spaces and separators can create false 
    // positives, so we always start by searching for a key coded in 
    // uppercase and then mixed once.

    val matchWithoutCleaning = run(subject, BASE32_32B_UPPER_PATTERN)
                            ?: run(subject, BASE32_32B_PATTERN)
    if (matchWithoutCleaning != null) return matchWithoutCleaning

    val cleaned = subject.replace(CLEAN_PATTERN, "")
    return run(cleaned, BASE32_32B_UPPER_PATTERN)
        ?: run(cleaned, BASE32_32B_PATTERN)
        ?: throw Exception("Missing reserve public key")
}

/** Extract the reserve public key from an incoming Taler transaction subject */
fun parseOutgoingTxMetadata(subject: String): Pair<ShortHashCode, ExchangeUrl>  {
    val (wtid, baseUrl) = subject.splitOnce(" ") ?: throw Exception("Malformed outgoing subject")
    return Pair(EddsaPublicKey(wtid), ExchangeUrl(baseUrl))
}