/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

import org.junit.Test
import tech.libeufin.common.*
import tech.libeufin.nexus.*
import tech.libeufin.nexus.ebics.*
import tech.libeufin.nexus.iso20022.*
import kotlin.io.path.*
import kotlin.test.*
import java.time.Instant

class Iso20022Test {
    @Test
    fun pain001() {
        val creditor = IbanAccountMetadata(
            iban = "CH4189144589712575493",
            bic = null,
            name = "Test"
        )
        val msg = Pain001Msg(
            messageId = "MESSAGE_ID",
            timestamp = dateToInstant("2024-09-09"),
            debtor = IbanAccountMetadata(
                iban = "CH7789144474425692816",
                bic = "BIC",
                name = "myname"
            ),
            sum = TalerAmount("EUR:47.32"),
            txs = listOf(
                Pain001Tx(
                    creditor = creditor,
                    amount = TalerAmount("EUR:42"),
                    subject = "Test 42",
                    endToEndId = "TX_FIRST"
                ),
                Pain001Tx(
                    creditor = creditor,
                    amount = TalerAmount("EUR:5.11"),
                    subject = "Test 5.11",
                    endToEndId = "TX_SECOND"
                ),
                Pain001Tx(
                    creditor = creditor,
                    amount = TalerAmount("EUR:0.21"),
                    subject = "Test 0.21",
                    endToEndId = "TX_THIRD"
                )
            )
        )
        for (dialect in Dialect.entries) {
            assertEquals(
                Path("sample/platform/${dialect}_pain001.xml").readText().replace("VERSION", VERSION),
                createPain001(msg, dialect).asUtf8()
            )
        }
    }

    @Test
    fun hac() {
        assertContentEquals(
            parseCustomerAck(Path("sample/platform/hac.xml").inputStream()),
            listOf(
                CustomerAck(
                    actionType = HacAction.FILE_DOWNLOAD,
                    orderId = null,
                    code = ExternalStatusReasonCode.TS01,
                    info = "",
                    timestamp = dateTimeToInstant("2024-09-02T15:47:30.350Z")
                ),
                CustomerAck(
                    actionType = HacAction.FILE_UPLOAD,
                    orderId = "ORDER_SUCCESS",
                    code = ExternalStatusReasonCode.TS01,
                    info = "",
                    timestamp = dateTimeToInstant("2024-09-02T20:48:43.153Z")
                ),
                CustomerAck(
                    actionType = HacAction.ES_VERIFICATION,
                    orderId = "ORDER_SUCCESS",
                    code = ExternalStatusReasonCode.DS01,
                    info = "",
                    timestamp = dateTimeToInstant("2024-09-02T20:48:43.153Z")
                ),
                CustomerAck(
                    actionType = HacAction.ORDER_HAC_FINAL_POS,
                    orderId = "ORDER_SUCCESS",
                    code = null,
                    info = "Some multiline info",
                    timestamp = dateTimeToInstant("2024-09-02T20:48:43.153Z")
                ),
                CustomerAck(
                    actionType = HacAction.FILE_DOWNLOAD,
                    orderId = null,
                    code = ExternalStatusReasonCode.TD01,
                    info = "",
                    timestamp = dateTimeToInstant("2024-09-02T15:47:31.754Z")
                ),
                CustomerAck(
                    actionType = HacAction.FILE_UPLOAD,
                    orderId = "ORDER_FAILURE",
                    code = ExternalStatusReasonCode.TS01,
                    info = "",
                    timestamp = dateTimeToInstant("2024-08-23T15:34:11.987")
                ),
                CustomerAck(
                    actionType = HacAction.ES_VERIFICATION,
                    orderId = "ORDER_FAILURE",
                    code = ExternalStatusReasonCode.TD03,
                    info = "",
                    timestamp = dateTimeToInstant("2024-08-23T15:34:13.307")
                ),
                CustomerAck(
                    actionType = HacAction.ORDER_HAC_FINAL_NEG,
                    orderId = "ORDER_FAILURE",
                    code = null,
                    info = "",
                    timestamp = dateTimeToInstant("2024-08-23T15:34:13.307")
                ),
            )
        )
    }

    @Test
    fun postfinance_camt054() {
        assertEquals(
            parseTx(Path("sample/platform/postfinance_camt054.xml").inputStream(), Dialect.postfinance),
            listOf(AccountTransactions(
                iban = "CH9289144596463965762",
                currency = "CHF",
                txs = listOf(
                    OutgoingPayment(
                        endToEndId = "ZS1PGNTSV0ZNDFAJBBWWB8015G",
                        msgId = "ZS1PGNTSV0ZNDFAJBBWWB8015G",
                        amount = TalerAmount("CHF:3.00"),
                        subject = null,
                        executionTime = dateToInstant("2024-01-15"),
                        creditorPayto = null
                    ),
                    IncomingPayment(
                        bankId = "62e2b511-7313-4ccd-8d40-c9d8e612cd71",
                        amount = TalerAmount("CHF:10"),
                        subject = "G1XTY6HGWGMVRM7E6XQ4JHJK561ETFDFTJZ7JVGV543XZCB27YBG",
                        executionTime = dateToInstant("2023-12-19"),
                        debtorPayto = ibanPayto("CH7389144832588726658", "Mr Test")
                    ),
                    IncomingPayment(
                        bankId = "62e2b511-7313-4ccd-8d40-c9d8e612cd71",
                        amount = TalerAmount("CHF:2.53"),
                        subject = "G1XTY6HGWGMVRM7E6XQ4JHJK561ETFDFTJZ7JVGV543XZCB27YB",
                        executionTime = dateToInstant("2023-12-19"),
                        debtorPayto = ibanPayto("CH7389144832588726658", "Mr Test")
                    ),
                    OutgoingBatch(
                        msgId = "ZS1PGNTSV0ZNDFAJBBWWB8015G",
                        executionTime = dateToInstant("2024-01-15")
                    )
                )
            ))
        )
    }

    @Test
    fun postfinance_camt053() {
        assertEquals(
            parseTx(Path("sample/platform/postfinance_camt053.xml").inputStream(), Dialect.postfinance),
            listOf(AccountTransactions(
                iban = "CH9289144596463965762",
                currency = "CHF",
                txs = listOf(
                    OutgoingReversal(
                        endToEndId = "889d1a80-1267-49bd-8fcc-85701a",
                        msgId = "889d1a80-1267-49bd-8fcc-85701a",
                        reason = "InconsistenWithEndCustomer 'Identification of end customer is not consistent with associated account number, organisation ID or private ID.' - 'more info here ...'",
                        executionTime = dateToInstant("2023-11-22")
                    ),
                    OutgoingReversal(
                        endToEndId = "4cc61cc7-6230-49c2-b5e2-b40bbb",
                        msgId = "4cc61cc7-6230-49c2-b5e2-b40bbb",
                        reason = "MissingCreditorNameOrAddress 'Specification of the creditor’s name and/or address needed for regulatory requirements is insufficient or missing.' - 'more info here ...'",
                        executionTime = dateToInstant("2023-11-22")
                    )
                )
            ))
        )
    }

    @Test
    fun gls_camt052() {
        assertEquals(
            parseTx(Path("sample/platform/gls_camt052.xml").inputStream(), Dialect.gls),
            listOf(AccountTransactions(
                iban = "DE84500105177118117964",
                currency = "EUR",
                txs = listOf(
                    OutgoingPayment(
                        endToEndId = "COMPAT_SUCCESS",
                        msgId = "COMPAT_SUCCESS",
                        amount = TalerAmount("EUR:2"),
                        subject = "TestABC123",
                        executionTime = dateToInstant("2024-04-18"),
                        creditorPayto = ibanPayto("DE20500105172419259181", "John Smith")
                    ),
                    OutgoingReversal(
                        endToEndId = "8XK8Z7RAX224FGWK832FD40GYC",
                        reason = "IncorrectAccountNumber 'Format of the account number specified is not correct' - 'IBAN fehlerhaft und ungültig'",
                        executionTime = dateToInstant("2024-09-05")
                    ),
                    IncomingPayment(
                        bankId = "BYLADEM1WOR-G2910276709458A2",
                        amount = TalerAmount("EUR:3"),
                        subject = "Taler FJDQ7W6G7NWX4H9M1MKA12090FRC9K7DA6N0FANDZZFXTR6QHX5G Test.,-",
                        executionTime = dateToInstant("2024-04-12"),
                        debtorPayto = ibanPayto("DE84500105177118117964", "John Smith")
                    ),
                    OutgoingReversal(
                        endToEndId = "COMPAT_FAILURE",
                        reason = "IncorrectAccountNumber 'Format of the account number specified is not correct' - 'IBAN ...'",
                        executionTime = dateToInstant("2024-04-12")
                    ),
                    OutgoingPayment(
                        endToEndId = "FD622SMXKT5QWSAHDY0H8NYG3G",
                        msgId = "BATCH_SINGLE_SUCCESS",
                        amount = TalerAmount("EUR:1.1"),
                        subject = "single 2024-09-02T14:29:52.875253314Z",
                        executionTime = dateToInstant("2024-09-02"),
                        creditorPayto = ibanPayto("DE89500105173198527518", "Grothoff Hans")
                    ),
                    OutgoingPayment(
                        endToEndId = "YF5QBARGQ0MNY0VK59S477VDG4",
                        msgId = "YF5QBARGQ0MNY0VK59S477VDG4",
                        amount = TalerAmount("EUR:1.1"),
                        subject = "Simple tx",
                        executionTime = dateToInstant("2024-04-18"),
                        creditorPayto = ibanPayto("DE20500105172419259181", "John Smith")
                    ),
                    OutgoingBatch(
                        msgId = "BATCH_MANY_SUCCESS",
                        executionTime = dateToInstant("2024-09-20"),
                    ),
                    OutgoingPayment(
                        endToEndId = "KLJJ28S1LVNDK1R2HCHLN884M7EKM5XGM5",
                        msgId = "BATCH_SINGLE_RETURN",
                        amount = TalerAmount("EUR:0.42"),
                        subject = "This should fail because bad iban",
                        executionTime = dateToInstant("2024-09-23"),
                        creditorPayto = ibanPayto("DE18500105173385245163", "John Smith")
                    ),
                    OutgoingReversal(
                        endToEndId = "KLJJ28S1LVNDK1R2HCHLN884M7EKM5XGM5",
                        reason = "IncorrectAccountNumber 'Format of the account number specified is not correct' - 'IBAN fehlerhaft und ungültig'",
                        executionTime = dateToInstant("2024-09-24")
                    ),
                )
            ))
        )
    }

    @Test
    fun gls_camt053() {
        assertEquals(
            parseTx(Path("sample/platform/gls_camt053.xml").inputStream(), Dialect.gls),
            listOf(AccountTransactions(
                iban = "DE84500105177118117964",
                currency = "EUR",
                txs = listOf(
                    OutgoingPayment(
                        endToEndId = "COMPAT_SUCCESS",
                        msgId = "COMPAT_SUCCESS",
                        amount = TalerAmount("EUR:2"),
                        subject = "TestABC123",
                        executionTime = dateToInstant("2024-04-18"),
                        creditorPayto = ibanPayto("DE20500105172419259181", "John Smith")
                    ),
                    OutgoingReversal(
                        endToEndId = "KGTDBASWTJ6JM89WXD3Q5KFQC4",
                        reason = "Retoure aus SEPA Überweisung multi line",
                        executionTime = dateToInstant("2024-09-04")
                    ),
                    OutgoingBatch(
                        msgId = "BATCH_MANY_PART",
                        executionTime = dateToInstant("2024-09-04")
                    ),
                    IncomingPayment(
                        bankId = "BYLADEM1WOR-G2910276709458A2",
                        amount = TalerAmount("EUR:3"),
                        subject = "Taler FJDQ7W6G7NWX4H9M1MKA12090FRC9K7DA6N0FANDZZFXTR6QHX5G Test.,-",
                        executionTime = dateToInstant("2024-04-12"),
                        debtorPayto = ibanPayto("DE84500105177118117964", "John Smith")
                    ),
                    OutgoingReversal(
                        endToEndId = "COMPAT_FAILURE",
                        reason = "IncorrectAccountNumber 'Format of the account number specified is not correct' - 'IBAN ...'",
                        executionTime = dateToInstant("2024-04-12")
                    ),
                    OutgoingPayment(
                        endToEndId = "FD622SMXKT5QWSAHDY0H8NYG3G",
                        msgId = "BATCH_SINGLE_SUCCESS",
                        amount = TalerAmount("EUR:1.1"),
                        subject = "single 2024-09-02T14:29:52.875253314Z",
                        executionTime = dateToInstant("2024-09-02"),
                        creditorPayto = ibanPayto("DE89500105173198527518", "Grothoff Hans")
                    ),
                    OutgoingPayment(
                        endToEndId = "YF5QBARGQ0MNY0VK59S477VDG4",
                        msgId = "YF5QBARGQ0MNY0VK59S477VDG4",
                        amount = TalerAmount("EUR:1.1"),
                        subject = "Simple tx",
                        executionTime = dateToInstant("2024-04-18"),
                        creditorPayto = ibanPayto("DE20500105172419259181", "John Smith")
                    ),
                ))
            )
        )
    }

    @Test
    fun gls_camt054() {
        assertEquals(
            parseTx(Path("sample/platform/gls_camt054.xml").inputStream(), Dialect.gls),
            listOf(AccountTransactions(
                iban = "DE84500105177118117964",
                currency = "EUR",
                txs = listOf<TxNotification>(
                    IncomingPayment(
                        bankId = null, //"IS11PGENODEFF2DA8899900378806",
                        amount = TalerAmount("EUR:2.5"),
                        subject = "Test ICT",
                        executionTime = dateToInstant("2024-05-05"),
                        debtorPayto = ibanPayto("DE84500105177118117964", "Mr Test")
                    )
                )
            ))
        )
    }

    @Test
    fun maerki_baumann_camt053() {
        assertEquals(
            parseTx(Path("sample/platform/maerki_baumann_camt053.xml").inputStream(), Dialect.maerki_baumann),
            listOf(AccountTransactions(
                iban = "CH7389144832588726658",
                currency = "CHF",
                txs = listOf<TxNotification>(
                    IncomingPayment(
                        bankId = "adbe4a5a-6cea-4263-b259-8ab964561a32",
                        amount = TalerAmount("CHF:1"),
                        creditFee = TalerAmount("CHF:0.2"),
                        subject = "SFHP6H24C16A5J05Q3FJW2XN1PB3EK70ZPY 5SJ30ADGY68FWN68G",
                        executionTime = dateToInstant("2024-11-04"),
                        debtorPayto = ibanPayto("CH7389144832588726658", "Mr Test")
                    ),
                    IncomingPayment(
                        bankId = "7371795e-62fa-42dd-93b7-da89cc120faa",
                        amount = TalerAmount("CHF:1"),
                        creditFee = TalerAmount("CHF:0.2"),
                        subject = "Random subject",
                        executionTime = dateToInstant("2024-11-04"),
                        debtorPayto = ibanPayto("CH7389144832588726658", "Mr Test")
                    ),
                    OutgoingPayment(
                        endToEndId = "5IBJZOWESQGPCSOXSNNBBY49ZURI5W7Q4H",
                        msgId = "BATCH_SINGLE_REPORTING",
                        amount = TalerAmount("CHF:0.1"),
                        subject = "multi 0 2024-11-21T15:21:59.8859234 63Z",
                        executionTime = dateToInstant("2024-11-27"),
                        creditorPayto = ibanPayto("CH7389144832588726658", "Grothoff Hans")
                    ),
                    OutgoingPayment(
                        endToEndId = "XZ15UR0XU52QWI7Q4XB88EDS44PLH7DYXH",
                        msgId = "BATCH_SINGLE_REPORTING",
                        amount = TalerAmount("CHF:0.13"),
                        subject = "multi 3 2024-11-21T15:21:59.8859234 63Z",
                        executionTime = dateToInstant("2024-11-27"),
                        creditorPayto = ibanPayto("CH7389144832588726658", "Grothoff Hans")
                    ),
                    OutgoingPayment(
                        endToEndId = "A09R35EW0359SZ51464E7TC37A0P2CBK04",
                        msgId = "BATCH_SINGLE_REPORTING",
                        amount = TalerAmount("CHF:0.12"),
                        subject = "multi 2 2024-11-21T15:21:59.8859234 63Z",
                        executionTime = dateToInstant("2024-11-27"),
                        creditorPayto = ibanPayto("CH7389144832588726658", "Grothoff Hans")
                    ),
                    OutgoingPayment(
                        endToEndId = "UYXZ78LE9KAIMBY6UNXFYT1K8KNY8VLZLT",
                        msgId = "BATCH_SINGLE_REPORTING",
                        amount = TalerAmount("CHF:0.11"),
                        subject = "multi 1 2024-11-21T15:21:59.8859234 63Z",
                        executionTime = dateToInstant("2024-11-27"),
                        creditorPayto = ibanPayto("CH7389144832588726658", "Grothoff Hans")
                    ),
                    IncomingPayment(
                        bankId = "f203fbb4-6e13-4c78-9b2a-d852fea6374a",
                        amount = TalerAmount("CHF:0.15"),
                        creditFee = TalerAmount("CHF:0.2"),
                        subject = "mini",
                        executionTime = dateToInstant("2024-12-02"),
                        debtorPayto = ibanPayto("CH7389144832588726658", "Grothoff Hans")
                    )
                )
            ))
        )
    }

    @Test
    fun pain002() {
        assertEquals(
            parseCustomerPaymentStatusReport(Path("sample/platform/pain002.xml").inputStream()),
            MsgStatus(
                id = "05BD4C5B4A2649B5B08F6EF6A31F197A",
                code = ExternalPaymentGroupStatusCode.PART,
                reasons = emptyList(),
                payments = listOf(
                    PmtStatus(
                        id = "NOTPROVIDED",
                        code = ExternalPaymentGroupStatusCode.PART,
                        reasons = listOf(
                            Reason(
                                code = ExternalStatusReasonCode.DT06,
                                information = "Due date is not a working day. Order will be executed on the next working day"
                            )
                        ),
                        transactions = listOf(
                            TxStatus(
                                id = "AQCXNCPWD8PHW5JTN65Y5XTF7R",
                                endToEndId = "AQCXNCPWD8PHW5JTN65Y5XTF7R",
                                code = ExternalPaymentTransactionStatusCode.RJCT, 
                                reasons = listOf(
                                    Reason(
                                        code = ExternalStatusReasonCode.AC04, 
                                        information = "Error message"
                                    )
                                )
                            ),
                            TxStatus(
                                id = "EE9SX76FC5YSC657EK3GMVZ9TC",
                                endToEndId = "EE9SX76FC5YSC657EK3GMVZ9TC",
                                code = ExternalPaymentTransactionStatusCode.RJCT,
                                reasons = listOf(
                                    Reason(
                                        code = ExternalStatusReasonCode.MS03,
                                        information = "Error message"
                                    )
                                )
                            ),
                            TxStatus(
                                id = "V5B3MXPEWES9VQW1JDRD6VAET4",
                                endToEndId = "V5B3MXPEWES9VQW1JDRD6VAET4",
                                code = ExternalPaymentTransactionStatusCode.RJCT,
                                reasons = listOf(
                                    Reason(
                                        code = ExternalStatusReasonCode.RR02,
                                        information = "Error message"
                                    )
                                )
                            )
                        )
                    )
                )
            )
        )
    }
}